/* Copyright (C) 1992 Imperial College */
/************************************************************************/
/*		Quintus style interface					*/
/*	PREDICATES:							*/
/*		load_foreign_files(ObjectFile, LibraryList)		*/
/*	USER PREDS							*/
/*		foreign_file(ObjectFile, FunctionList)			*/
/*		foreign(Function, Language, PredicateSpecification)	*/
/************************************************************************/
load_foreign_files(File) :-
	load_foreign_files(File, []).

load_foreign_files([], _) :- !.
load_foreign_files([File|T], LibraryList) :- !,
	'$load_foreign_file$'(File, LibraryList),
	load_foreign_files(T, LibraryList).
load_foreign_files(File, LibraryList) :-
	'atom%f'(File),
	'$load_foreign_file$'(File, LibraryList).

'$load_foreign_file$'(File, LibraryList) :-
	defined(foreign_file/2),
	defined(foreign/3),
	'$foreign_wrapper$'(File, ObjectFile, NewSourceFile, NewObjectFile),
	writeseqnl(user_error, ['\nGenerating interface file', NewSourceFile, '...']),
	'$gen_foreign_file$'(ObjectFile, NewSourceFile, FnList),
	'$install_directory'(Dir),
	concat_atom(['cc -c -w -I', Dir, '/include ', NewSourceFile,
		     ' -o ', NewObjectFile], Command),
	writeseqnl(user_error, ['Compiling interface file\n', Command]),
	unix(Command),
	writeseqnl(user_error, ['Loading interface file', NewObjectFile, '...']),
        'load_foreign%f'(NewObjectFile, FnList, [ObjectFile|LibraryList]).

'$install_directory'(Dir) :-
	getenv('ICP_INSTALLDIR', Dir), !.
'$install_directory'(_) :-
	write(user_error, '\nERROR: the environment variable ICP_INSTALLDIR is not set\n'),
	fail.

'$foreign_wrapper$'(ObjectFile, ObjectFile, NewSource, NewObject) :-
	concat(BaseName, '.o', ObjectFile), !,
	'$construct_new_names'(BaseName, NewSource, NewObject).
'$foreign_wrapper$'(BaseName, ObjectFile, NewSource, NewObject) :-
	concat(BaseName, '.o', ObjectFile),
	'$construct_new_names'(BaseName, NewSource, NewObject).

'$construct_new_names'(Base, NewSource, NewObject) :-
	name(Base, NameList),
	remove_path(NameList, NameList, BaseList),
	name(BaseName, BaseList),
	concat_atom(['/tmp/', BaseName, '_interface_'], NewBase),
	repeat,
	N is rand,
	concat_atom([NewBase, N, '.c'], NewSource),
	not file_exists(Filename),
	concat_atom([NewBase, N, '.o'], NewObject),
	not file_exists(Filename), !.

remove_path([], BaseList, BaseList).
remove_path([0'/|Tail], _, BaseList) :- !,
	remove_path(Tail, Tail, BaseList).
remove_path([_|Tail], NameList, BaseList) :-
	remove_path(Tail, NameList, BaseList).

'$gen_foreign_file$'(ObjectFile, NewSourceFile, FnList) :-
	foreign_file(ObjectFile, FunctionList),
	current_output(Old),
	open(NewSourceFile, write, Fp),
	set_output(Fp),
	write('#include <ICprolog.h>\n\n'),
	(
		'$gen_foreign_functions$'(FunctionList, ObjectFile, FnList)
	->
		Ok = yes
	;
		Ok = no
	),
	close(Fp),
	set_output(Old), !,
	Ok = yes.

'$gen_foreign_functions$'([Function|FunctionList], ObjectFile, [Fn|FnList]) :-
	'$gen_foreign_function$'(ObjectFile, Function, Fn), !,
	'$gen_foreign_functions$'(FunctionList, ObjectFile, FnList).
'$gen_foreign_functions$'([], _, []).

'$gen_foreign_function$'(ObjectFile, Cname, RealCname(Pname,Arity)) :-
	foreign(Cname, c, PredicateSpec),
	PredicateSpec =.. [_|ArgsSpec],
	'$partition_foreign_args$'(ArgsSpec, 1, Args, Ret, Errors),
	'$check_foreign_errors$'(Errors, Ret, ObjectFile:Cname),
	functor(PredicateSpec, Pname, Arity),
	concat(Cname, '_interface', RealCname),
	write('bool '), write(RealCname), write('()\n{\n'),
	write('\tcellpo '), '$write_foreign_A_args$'(1, Arity),
	'$write_foreign_args$'(Args), '$write_foreign_args$'(Ret),
	'$write_extrn_c_fn$'(Ret, Cname), nl,
	'$write_delnk$'(Args, in), nl,
	'$write_delnk$'(Ret, out), nl,
	'$get_foreign_values$'(Args, in), nl,
	'$write_foreign_call$'(Ret, Args, Cname), nl,
	'$set_foreign_output_args$'(Ret, Args, Arity),
	write('\treturn(SUCCESS);\n}\n\n').

'$write_extrn_c_fn$'([], Cname) :-
	write('\tint ret;\n'),
	write('\textern bool '), write(Cname), write('();\n').
'$write_extrn_c_fn$'([Type(_,_)], Cname) :-
	write('\textern '), '$write_foreign_arg_type$'(Type),
	write(Cname), write('();\n').

'$write_foreign_A_args$'(N, Arity) :-  N > Arity, !.
'$write_foreign_A_args$'(Arity, Arity) :- !,
	'$write_foreign_reg$'(Arity), write(';\n').
'$write_foreign_A_args$'(N, Arity) :-
	'$write_foreign_reg$'(N), write(','),
	N1 is N + 1,
	'$write_foreign_A_args$'(N1, Arity).

'$write_foreign_args$'([]) :- !.
'$write_foreign_args$'([Type(N,_)|T]) :-
	'$write_foreign_arg_type$'(Type),
	'$write_foreign_val$'(Type, N),
	write(';\n'), !,
	'$write_foreign_args$'(T).

'$write_foreign_val$'(Type, N) :- write(Type), write(N).

'$write_foreign_reg$'(N) :- write(arg), write(N).

'$write_foreign_arg_type$'(integer) :- write('\tlong ').
'$write_foreign_arg_type$'(float) :- write('\tFLOAT ').
'$write_foreign_arg_type$'(atom) :- write('\tchar *').

'$partition_foreign_args$'([], _, [], [], []) :- !.
'$partition_foreign_args$'([+(IType)|Args], N, [OType(N,in)|Out], Ret, Errs) :-
	'$check_foreign_arg$'(IType, OType),
	N1 is N + 1, !,
	'$partition_foreign_args$'(Args, N1, Out, Ret, Errs).
'$partition_foreign_args$'([-(IType)|Args], N, [OType(N,out)|Out], Ret, Errs) :-
	'$check_foreign_arg$'(IType, OType),
	N1 is N + 1, !,
	'$partition_foreign_args$'(Args, N1, Out, Ret, Errs).
'$partition_foreign_args$'([[-(IType)]|Args], N, Out, [OType(N,out)|Ret], Errs) :-
	'$check_foreign_arg$'(IType, OType),
	N1 is N + 1, !,
	'$partition_foreign_args$'(Args, N1, Out, Ret, Errs).
'$partition_foreign_args$'([Arg|Args], N, Out, Ret, [Arg|Errs]) :-
	N1 is N + 1, !,
	'$partition_foreign_args$'(Args, N1, Out, Ret, Errs).

'$check_foreign_arg$'(integer, integer).
'$check_foreign_arg$'(float, float).
'$check_foreign_arg$'(atom, atom).
'$check_foreign_arg$'(string, atom).

'$check_foreign_errors$'([], [], _) :- !.
'$check_foreign_errors$'([], [_], _) :- !.
'$check_foreign_errors$'(Errors, _, ObjectFile) :- !,
	writeseqnl(user_error, ['ERRORS: for', ObjectFile, '=', Errors]),
	fail.
'$check_foreign_errors$'(_, List, ObjectFile) :- !,
	writeseqnl(user_error, ['ERRORS: for', ObjectFile, '=', List]),
	fail.

'$write_delnk$'([], _) :- !.
'$write_delnk$'([_(N, InOut)|R], InOut) :- !,
	write('\t'), '$write_foreign_reg$'(N),
	write(' = &A['), write(N), write(']; '),
	write('delnk('), '$write_foreign_reg$'(N), write(');\n'),
	'$write_delnk$'(R, InOut).
'$write_delnk$'([_|R], InOut) :- '$write_delnk$'(R, InOut).

/* No return value (bool) */
'$write_foreign_call$'([], Args, Function) :-
	write('\t'), write('if ((ret = '),
	'$write_foreign_call$'(Function, Args),
	write(') != SUCCESS)\n\t\treturn(ret);\n').
'$write_foreign_call$'([Type(N, InOut)], Args, Function) :-
	'$foreign_type_check$'(InOut, Type, N),
	write('\t'), '$write_foreign_val$'(Type, N), write(' = '),
	'$write_foreign_call$'(Function, Args),
	write(';\n').

'$write_foreign_call$'(Function, Args) :-
	write(Function),
	write('('),
	'$write_foreign_call_args$'(Args),
	write(')').

'$write_foreign_call_args$'([]) :- !.
'$write_foreign_call_args$'([A]) :- !,
	'$write_foreign_C_arg$'(A).
'$write_foreign_call_args$'([A|Args]) :-
	'$write_foreign_C_arg$'(A), write(', '), !,
	'$write_foreign_call_args$'(Args).

'$write_foreign_C_arg$'(Type(N,out)) :- !,
	write('&'),
	'$write_foreign_val$'(Type, N).
'$write_foreign_C_arg$'(Type(N,in)) :- !, '$write_foreign_val$'(Type, N).

'$get_foreign_values$'([], _).
'$get_foreign_values$'([Type(N, InOut)|T], InOut) :- !,
	'$foreign_type_check$'(InOut, Type, N),
	write('\t'), '$write_foreign_val$'(Type, N), write(' = '),
	'$get_foreign_value$'(Type(N, InOut)),
	write(';\n'), !,
	'$get_foreign_values$'(T, in).
'$get_foreign_values$'([_|T], InOut) :- !,
	'$get_foreign_values$'(T, InOut).

'$foreign_type_check$'(InOut, Type, N) :-
	write('\tif ('),
	'$foreign_type_check1$'(Type), '$write_foreign_reg$'(N), write(')'),
	'$foreign_type_check2$'(InOut, N),
	write(') return(FAIL);\n').

'$foreign_type_check1$'(integer) :- !, write('NotInt(').
'$foreign_type_check1$'(float) :- !, write('NotFloat(').
'$foreign_type_check1$'(atom) :- !, write('NotSymb(').

'$foreign_type_check2$'(in, _) :- !.
'$foreign_type_check2$'(out, N) :- !,
	write(' && NotVar('), '$write_foreign_reg$'(N), write(')').

'$get_foreign_value$'(integer(N, _)) :-
	write('intvl('), '$write_foreign_reg$'(N), write(')').
'$get_foreign_value$'(float(N, _)) :-
	write('floatvl('), '$write_foreign_reg$'(N), write(')').
'$get_foreign_value$'(atom(N, _)) :-
	write('symbname(symbvl('), '$write_foreign_reg$'(N), write('))').

'$set_foreign_output_args$'([], Args, Arity) :-
	'$set_foreign_output_args$'(Args, Arity).
'$set_foreign_output_args$'([Ret], Args, Arity) :-
	'$set_foreign_output_args$'([Ret|Args], Arity).

'$set_foreign_output_args$'([], _) :- !.
'$set_foreign_output_args$'([Type(N, out)|Args], Arity) :-
	write('\tforeign_out_'), write(Type), write('('),
	'$write_foreign_val$'(Type, N), write(', '), write(Arity), write(', '),
	'$write_foreign_reg$'(N), write(', '), write(N), write(');\n'), !,
	'$set_foreign_output_args$'(Args, Arity).
'$set_foreign_output_args$'([_|Args], Arity) :-
	'$set_foreign_output_args$'(Args, Arity).

