/* Copyright (C) 1992 Imperial College */
end_of_file().          % included to cover up deficiency of reader

tag(X,Y)        :- 'tag%f'(X,Y).
write_term(X)   :- 'write_term%f'(X).

write_term(S, Term) :-
	current_output(Out),
	set_output(S),
	('write_term%f'(Term)-> set_output(Out); set_output(Out), fail).

system(on)      :- 'system%f'(0).
system(off)     :- 'system%f'(1).


/* The following block are all the primitives which might consume heap space.
   It is essential that these primitives are only accessed indirectly, as
   otherwise the garbage collector does not know which local vars are valid. */
'$ntuple$'(Len, Tpl) :-
    'ntpl%f'(Len,Tpl).
'%=..%'(Tpl,List) :-
    'univ%f'(Tpl,List).
'?tohollow?'(Ground,Hollow,Names,Variables) :-
    'to_hollow%f'(Ground,Hollow,Names,Variables).
'?toground?'(Hollow,Ground,Vars,Varnames,Usednames) :-
    'to_ground%f'(Hollow,Ground,Vars,Varnames,Usednames).
'?varsin?'(Term,Vars) :-
    'varsin%f'(Term,Vars).
'?fadd?'(N,M,Sum) :-
    'fadd%f'(N,M,Sum).
'?fsub?'(N,M,Difference) :-
    'fsub%f'(N,M,Difference).
'?fmul?'(N,M,Product) :-
    'fmul%f'(N,M,Product).
'?fdiv?'(N,M,Quotient) :-
    'fdiv%f'(N,M,Quotient).
'?name?'(Atom,Str,Type) :-
    'name%f'(Atom,Str,Type).
'?concat?'(First,Second,Join) :-
    'concat%f'(First,Second,Join).
'?get_prop?'(Object,Property,Value) :-
    'get_prop%f'(Object,Property,Value).
'?get_cons?'(Property,List) :-
    'get_cons%f'(Property,List).
'?get_props?'(Object,List) :-
    'get_props%f'(Object,List).
'$read$'(Prio,Type,Op) :-
    'interm%f'(Prio,Type,Op).
read_term(Term) :-
    'read_term%f'(Term).
'$ram_const$'(Stream,Var) :-
    'ram_const%f'(Stream,Var).

read_term(S, Term) :-
	current_output(Out),
	set_output(S),
	(read_term(Term)-> set_output(Out); set_output(Out), fail).

/***************************************************/
/*                 Type Primitives                 */
/***************************************************/

nonvar(X) :-
    'var%f'(X), !,
    fail.
nonvar(_).

atom(X) :-
    'atom%f'(X), !.
atom(X) :-
    X == [].

atomic(X) :-
    atom(X), !.
atomic(X) :-
    number(X).

float(X) :-
    'float%f'(X).

compound(X) :-
    'tpl%f'(X), !.
compound(X) :-
    X == [], !,
    fail.
compound(X) :-
    'list%f'(X).

tuple(X) :-
    'tpl%f'(X).

tuple(T,L):- 'tpl%f'(T), 'arity%f'(T,L).


/***************************************************/
/*             Metalogical Primitives              */
/***************************************************/

arg(N, Term, X) :-
    'tpl%f'(Term),
    'int%f'(N), !,
    'arg%f'(N, Term, X).
arg(1, [X|_], X).
arg(2, [_|X], X).

functor(Term, Name, Arity) :-
    'tag%f'(Term, N),
    '?functor?'(N, Term , Name, Arity).

/* Note that the predicate name of [] is [], not '[]' */
'?functor?'(0, Term, Name, Arity) :-
    'int%f'(Arity),
    '$functor$'(Arity, Name, Term).
'?functor?'(1, Term, Term, 0).
'?functor?'(2, Term, Term, 0).
'?functor?'(3, Term, Term, 0).
'?functor?'(4, [], [], 0).
'?functor?'(5, [_|_], '.', 2).
'?functor?'(6, [_|_], '.', 2) :- !.
'?functor?'(6, Term, Name, Arity) :-
    'funct%f'(Term, Name),
    'arity%f'(Term, Ar),
    'sub%f'(Ar, 1, Arity).

'$functor$'(0, Term, Term) :- !.
'$functor$'(2, Name, [_|_]) :- 'cmp%f'(0, Name, '.'), !.
'$functor$'(Arity, Name, Term) :-
    'gt%f'(Arity,0),
    'increment%f'(Arity, Ar),
    '$ntuple$'(Ar, Term),
    'arg%f'(0, Term, Name).

ntpl(Len, Tpl) :-
    'int%f'(Len),
    '$ntuple$'(Len, T),
    Tpl = T.

X =.. Y :-
    'var%f'(Y), !,
    '?=..?'(X, Y).
X =.. Y :-
    'var%f'(X),
    '?unsafelist?'(Y), !,
    fail.
X =.. Y :-
    atomic(X), !,
    Y = [X].
X =.. [H|T] :-
    T == [],
    atomic(H), !,
    X = H.
[H|T] =.. ['.',H,T] :- !.
X =.. Y :-
    '$=..$'(X, Y).

'?=..?'(X, Y) :-
    'var%f'(X), !,
    '$=..$'(X, Y).
'?=..?'(X, [X]) :-
    atomic(X), !.
'?=..?'([H|T], ['.',H,T]) :- !.
'?=..?'(X, Y) :-
    '$=..$'(X, Y).

'$=..$'(X, Y) :- '$ primcatch'('%=..%'(X, Y), X =.. Y).

/* only proper lists fail */
'?unsafelist?'(X) :-
    'var%f'(X).
'?unsafelist?'([]) :- !, fail.
'?unsafelist?'([_|T]) :- !,
    '?unsafelist?'(T).
'?unsafelist?'(_).

varsin(X,Y) :- '$ primcatch'('?varsin?'(X,Y), varsin(X,Y)).

tohollow(X,Y,Z) :-
	'$ primcatch'('?tohollow?'(X,Y1,Z,_), tohollow(X,Y,Z)),
	Y1=Y.
tohollow(X,Y,Z,U) :-
	'$ primcatch'('?tohollow?'(X,Y1,Z,U1), tohollow(X,Y,Z,U)),
	Y1=Y, U1=U.

toground(X,Y) :-
	'$ primcatch'('?toground?'(X,Y1,_,_,_), toground(X,Y)),
	Y1=Y.
toground(X,Y,Z)     :-
	'$ primcatch'('?toground?'(X,Y1,_,_,Z1), toground(X,Y,Z)),
	Y1=Y, Z1=Z.
toground(X,Y,Z,U)   :-
	'$ primcatch'('?toground?'(X,Y1,Z1,_,U1), toground(X,Y,Z,U)),
	Y1=Y, Z1=Z, U1=U.
toground(X,Y,Z,U,V) :-
	'$ primcatch'('?toground?'(X,Y1,Z,U,V1), toground(X,Y,Z,U,V)),
	Y1=Y, V1=V.

copy_term(Term, Copy) :-
    '$ primcatch'('$ copy_term'(Term, Copy), copy_term(Term, Copy)).

'$ copy_term'(Term, Copy) :- 'copy%f'(Term, Copy).



/***************************************************/
/*             Arithmetical Primitives             */
/***************************************************/

/* X is Exp :- '$ primcatch'('?is?'(Exp,X), X is Exp). */
X is Exp :- '?is?'(Exp,X).

'?is?'(X, Y) :- 
  '$is$'(Y),
  '%is%'(X,Y).

'$is$'(Y) :-
  'var%f'(Y), !.
'$is$'(Y) :-
  number(Y), !.
'$is$'(_) :-
  'throw%f'(218).
    
'%is%'(X, _) :- 
    'var%f'(X),
    'throw%f'(100).
'%is%'(-N, X) :- !,
    '%is%'(N, NN),
    '?fsub?'(0, NN, X).
'%is%'(N+M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    '?fadd?'(NN, MM, X).
'%is%'(N++M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'add%f'(NN, MM, X).
'%is%'(N-M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    '?fsub?'(NN, MM, X).
'%is%'(N--M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'sub%f'(NN, MM, X).
'%is%'(N*M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    '?fmul?'(NN, MM, X).
'%is%'(N**M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'mul%f'(NN, MM, X).
'%is%'(N/M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    '?fdiv?'(NN, MM, X).
'%is%'(N//M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'div%f'(NN, MM, X).
'%is%'(N mod M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'mod%f'(NN, MM, X).
'%is%'((N<<M), X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'lshift%f'(NN, MM, X).
'%is%'(N>>M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'rshift%f'(NN, MM, X).
'%is%'(\N, X) :- !,
    '%is%'(N, NN),
    'not%f'(NN, X).
'%is%'(N/\M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'and%f'(NN, MM, X).
'%is%'(N\/M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'or%f'(NN, MM, X).
'%is%'(int(N), X) :- !,
    '%is%'(N, NN),
    'int%f'(NN, X).
'%is%'(sin(N), X) :- !,
    '%is%'(N, NN),
    'sin%f'(NN, X).
'%is%'(cos(N), X) :- !,
    '%is%'(N, NN),
    'cos%f'(NN, X).
'%is%'(tan(N), X) :- !,
    '%is%'(N, NN),
    'tan%f'(NN, X).
'%is%'(asin(N), X) :- !,
    '%is%'(N, NN),
    'asin%f'(NN, X).
'%is%'(acos(N), X) :- !,
    '%is%'(N, NN),
    'acos%f'(NN, X).
'%is%'(atan(N), X) :- !,
    '%is%'(N, NN),
    'atan%f'(NN, X).
'%is%'(atan2(N,M), X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'atan2%f'(NN, MM, X).
'%is%'(sinh(N), X) :- !,
    '%is%'(N, NN),
    'sinh%f'(NN, X).
'%is%'(cosh(N), X) :- !,
    '%is%'(N, NN),
    'cosh%f'(NN, X).
'%is%'(tanh(N), X) :- !,
    '%is%'(N, NN),
    'tanh%f'(NN, X).
'%is%'(exp(N), X) :- !,
    '%is%'(N, NN),
    'exp%f'(NN, X).
'%is%'(log(N), X) :- !,
    '%is%'(N, NN),
    'log%f'(NN, X).
'%is%'(log10(N), X) :- !,
    '%is%'(N, NN),
    'log10%f'(NN, X).
'%is%'(sqrt(N), X) :- !,
    '%is%'(N, NN),
    'sqrt%f'(NN, X).
'%is%'(N^M, X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'pow%f'(NN, MM, X).
'%is%'(pow(N,M), X) :- !,
    '%is%'(N, NN),
    '%is%'(M, MM),
    'pow%f'(NN, MM, X).
'%is%'(abs(N), X) :- !,
    '%is%'(N, NN),
    'abs%f'(NN, X).
'%is%'(sign(N), X) :- !,
    '%is%'(N, NN),
    'sign%f'(NN, X).
'%is%'(ceil(N), X) :- !,
    '%is%'(N, NN),
    'ceil%f'(NN, X).
'%is%'(floor(N), X) :- !,
    '%is%'(N, NN),
    'floor%f'(NN, X).
'%is%'(pi, X) :- !,
    'pi%f'(X).
'%is%'(rand, X) :- !,
    'rand%f'(X).
'%is%'(deg2rad(N), X) :- !,
    '%is%'(N, NN),
    'deg2rad%f'(NN, X).
'%is%'(rad2deg(N), X) :- !,
    '%is%'(N, NN),
    'rad2deg%f'(NN, X).
'%is%'(float(N), X) :- !,
	'%is%'(N, NN),
	'ar_float%f'(NN, X).
'%is%'(integer(N), X) :- !,
	'%is%'(N, NN),
	'ar_int%f'(NN, X).
'%is%'([X],X) :- 
    'int%f'(X),!.    
'%is%'(X, X) :- 
    '?is_number?'(X).

'?is_number?'(X) :-
    number(X), !.
'?is_number?'(_) :-
    'throw%f'(219).


X =:= Y :- '$ primcatch'('?=:=?'(X,Y), X =:= Y).
X =\= Y :- '$ primcatch'('?=\=?'(X,Y), X =\= Y).
X < Y   :- '$ primcatch'('?<?'(X,Y), X < Y).
X > Y   :- '$ primcatch'('?>?'(X,Y), X > Y).
X >= Y  :- '$ primcatch'('?>=?'(X,Y), X >= Y).
X =< Y  :- '$ primcatch'('?=<?'(X,Y), X =< Y).

'?=:=?'(X,Y) :- '%is%'(X,XX), '%is%'(Y,YY), 'eq%f'(XX,YY).
'?=\=?'(X,Y) :- '%is%'(X,XX), '%is%'(Y,YY), 'ne%f'(XX,YY).
'?<?'(X,Y)   :- '%is%'(X,XX), '%is%'(Y,YY), 'lt%f'(XX,YY).
'?>?'(X,Y)   :- '%is%'(X,XX), '%is%'(Y,YY), 'gt%f'(XX,YY).
'?>=?'(X,Y)  :- '%is%'(X,XX), '%is%'(Y,YY), 'ge%f'(XX,YY).
'?=<?'(X,Y)  :- '%is%'(X,XX), '%is%'(Y,YY), 'le%f'(XX,YY).



/***************************************************/
/*        Term Comparison and List Handling        */
/***************************************************/

X = X.

X \= Y :-
    X = Y, !,
    fail.
_ \= _.

X == Y :-
    'cmp%f'(0, X, Y).

X \== Y :-
    'cmp%f'(0, X, Y), !,
    fail.
_ \== _.

compare(R, X, Y) :-
    'cmp%f'(N, X, Y),
    '?compare?'(N, R), !.

'?compare?'( 0, = ).
'?compare?'( 0, =<).
'?compare?'( 0, >=).
'?compare?'(-1, < ).
'?compare?'(-1, =<).
'?compare?'( 1, > ).
'?compare?'( 1, >=).

X @> Y :-
    'cmp%f'(1, X, Y).

X @< Y :-
    'cmp%f'(-1, X, Y).

X @>= Y :-
    'cmp%f'(-1, X, Y), !,
    fail.
_ @>= _.

X @=< Y :-
    'cmp%f'(1, X, Y), !,
    fail.
_ @=< _.

append([], L, L).
append([H|T], L, [H|R]) :-
    append(T, L, R).

on(El, List) :-
    member(El, List).

member(X, [X|_]    ).
member(X, [_,X|_]  ).
member(X, [_,_,X|_]).
member(X, [_,_,_|L]) :-
    member(X, L).

memberchk(X, [X|_]    ) :- !.
memberchk(X, [_,X|_]  ) :- !.
memberchk(X, [_,_,X|_]) :- !.
memberchk(X, [_,_,_|L]) :-
    memberchk(X, L).

length(X, 0) :-
    (   X == [],
        !
    ;   X = []
    ).
length([_|Rest], N) :-
    'var%f'(N), !,
    '?length?'(Rest, N1),
    'increment%f'(N1, N).
length([_|Rest], L) :-
    'int%f'(L),
    'gt%f'(L, 0),
    'sub%f'(L, 1, L1), !,
    length(Rest, L1).
length(Tpl, L) :-
    'tpl%f'(Tpl),
    'arity%f'(Tpl, L).
 
'?length?'([], 0).
'?length?'([_|Rest], L) :-
    '?length?'(Rest, L1),
    'increment%f'(L1, L).

nth0(Index, List, Elem) :-
    integer(Index), !,
    '$nth0_1$'(Index, List, Elem).
nth0(Index, List, Elem) :-
    var(Index), !,
    '$nth0_2$'(Index, List, Elem).

'$nth0_1$'(0, [Elem|_], Elem) :- !.	% take nth deterministically
'$nth0_1$'(N, [_|Tail], Elem) :-
    M is N - 1,
    nth0(M, Tail, Elem).

'$nth0_2$'(0, [Elem|_], Elem).		% match
'$nth0_2$'(N, [_|Tail], Elem) :-
    '$nth0_2$'(M, Tail, Elem), 
    N is M + 1.

nth1(Index, List, Elem) :-
    integer(Index), !,
    '$nth1_1$'(Index, List, Elem).
nth1(Index, List, Elem) :-
    var(Index), !,
    '$nth1_2$'(Index, List, Elem).

'$nth1_1$'(1, [Elem|_], Elem) :- !.	% take nth deterministically
'$nth1_1$'(N, [_|Tail], Elem) :-
    M is N - 1,
    nth1(M, Tail, Elem).

'$nth1_2$'(1, [Elem|_], Elem).		% match
'$nth1_2$'(N, [_|Tail], Elem) :-
    '$nth1_2$'(M, Tail, Elem), 
    N is M + 1.

remove(Item, List, Remainder) :-
    delete(Item, List, Remainder).

delete(Item, [Item|List], List).
delete(Item, [Head|List], [Head|Remainder]) :-
    delete(Item, List, Remainder).

/*
remove_all(Filter, [], []) :- !.
remove_all(Filter, [H|T], Remainder) :-
    on(H, Filter), !,
    remove_all(Filter, T, Remainder).
remove_all(Filter, [H|T], [H|Remainder]) :-
    remove_all(Filter, T, Remainder).
*/

reverse(List, Rev) :-
    '?reverse?'(List, [], Rev).

'?reverse?'([], List, List) :- !.
'?reverse?'([H|T], List, Rev) :-
    '?reverse?'(T, [H|List], Rev).



/***************************************************/
/*            String and Atom Handling             */
/***************************************************/

name(X,Y)     :- '$ primcatch'('?name?'(X,Y,0), name(X,Y)).
atom_chars(X,Y)     :- '$ primcatch'('?name?'(X,Y,1), atom_chars(X,Y)).
number_chars(X,Y)     :- '$ primcatch'('?name?'(X,Y,2), number_chars(X,Y)).
concat(X,Y,Z) :- '$ primcatch'('?concat?'(X,Y,Z), concat(X,Y,Z)).
pname(X,Y)    :- '$ primcatch'('?pname?'(X,Y,_,_), pname(X,Y)).
pname(X,Y,Z,U):- '$ primcatch'('?pname?'(X,Y,Z,U), pname(X,Y,Z,U)).
gensym(X,Y)   :- '$ primcatch'('?gensym?'(X,Y), gensym(X,Y)).

'?pname?'(Term, Name, Vnames, Vars) :-
    'atom%f'(Name), !,
    open_ram(Name, read, In),
    gread(In, NewTerm, Vnames),
    close(In),
    tohollow(NewTerm, Term, Vnames, Vars).
'?pname?'(Term, Name, Vnames, Vars) :-
    'var%f'(Name), !,
    toground(Term, GTerm, Vars, Vnames, _),
    open_ram('', write, Out),
    write(Out, GTerm),
    ram_const(Out, Name).
'?pname?'(_, _, _, _) :-
    throw(210).

concat_atom([H|T], At) :-
    '$ concat_atom'(T, H, At).

'$ concat_atom'([], At, At).
'$ concat_atom'([H|T], T1, At) :-
    concat(T1, H, T2),
    '$ concat_atom'(T, T2, At).

'?gensym?'(Base, NewSymbol) :-
	'gensym%f'(Base, NewSymbol).


/***************************************************/
/*                    Execution                    */
/***************************************************/

abort :-
    'throw%f'(1000).

catch(Goal) :-
    'read_catch%f'(OldCatch),
    '$catch$'(Goal, system_error(Goal,Code), Code, OldCatch).

catch(Goal, ReportGoal) :-
    'read_catch%f'(OldCatch),
    '$catch$'(Goal, system_error(ReportGoal, Code), Code, OldCatch).

catch(Goal, ErrHandler, Code) :-
    'read_catch%f'(OldCatch),
    '$catch$'(Goal, ErrHandler, Code, OldCatch).

'$catch$'(Goal, _, _, OldCatch) :-
    'catch%f'(NewCatch),
    Goal,
    (   'set_catch%f'(OldCatch)
    ;   'set_catch%f'(NewCatch),
        fail
    ).
'$catch$'(_, Err, Code, OldCatch) :-
    'read_error%f'(Code),
    'set_catch%f'(OldCatch),
    Code \== 0,         % check for normal (unification) failure
    Err.

'$ primcatch'(Goal) :-
    'read_catch%f'(OldCatch),
    '$catch$'(Goal, system_error(Goal,Code), Code, OldCatch), !.
'$ primcatch'(Goal, ReportGoal) :-
    'read_catch%f'(OldCatch),
    '$catch$'(Goal, system_error(ReportGoal, Code), Code, OldCatch), !.

throw(X) :-
    'throw%f'(X).



/***************************************************/
/*               Property Management               */
/***************************************************/

set_prop(X,Y,Z) :- '$ primcatch'('?set_prop?'(X,Y,Z), set_prop(X,Y,Z)).

'?set_prop?'(X,Y,Z) :-
    'set_prop%f'(X,Y,Z).

get_prop(X,Y,Z) :- '$ primcatch'('?get_prop?'(X,Y,Z), get_prop(X,Y,Z)).

del_prop(X,Y)   :- '$ primcatch'('?del_prop?'(X,Y), del_prop(X,Y)).

'?del_prop?'(X,Y) :-
    'del_prop%f'(X,Y).

del_props(X)    :- '$ primcatch'('?del_props?'(X), del_props(X)).

'?del_props?'(X) :-
    'del_props%f'(X).

del_cons(X)     :- '$ primcatch'('?del_cons?'(X), del_cons(X)).

'?del_cons?'(X) :-
    'del_cons%f'(X).

remember(Atom, Value) :-
    set_prop(Atom, Atom, Value).

recall(Atom, Value) :-
    get_prop(Atom, Atom, Value).

default_prop(Atom, Value, _) :-
    get_prop(Atom, Atom, Value), !.
default_prop(_, Default, Default).

forget(Atom) :-
    del_prop(Atom, Atom).

get_cons(X,Y)   :- '$ primcatch'('?get_cons?'(X,Y), get_cons(X,Y)).

get_props(X,Y)  :- '$ primcatch'('?get_props?'(X,Y), get_props(X,Y)).

list_props(Object, L) :-
    get_props(Object, Props),
    '$retrieve_props$'(Props, Object, L).

'$retrieve_props$'([], _, []).
'$retrieve_props$'([H|T], Object, [(H=V)|R]) :-
    get_prop(Object, H, V),
    '$retrieve_props$'(T, Object, R).


/***************************************************/
/*             Input/Output Primitives             */
/***************************************************/

read(T) :-
    '?read?'(T, _).

read(S, T) :-
    '$ read_stream'(S, T, _).

gread(T) :-
    '?read?'(T, W),
    '?ground?'(W, _).

gread(S, T) :-
    gread(S, T, _).

gread(S, T, V) :-
    '$ read_stream'(S, T, W),
    '?ground?'(W, V).

'$ read_stream'(S, T, W) :-
    current_input(I),
    set_input(S),
    '?read_stream?'(I, T, W) .

'?read_stream?'(I, T, W) :- 
    '?read?'(T,W), !,
    set_input(I).
'?read_stream?'(I, _, _) :-
    set_input(I),
    fail.

'?read?'(T, W) :-
    repeat,
    '$ primcatch'('$read$'(Term, 1200, W), read(T)), !,
    T = Term.

'?ground?'([], []) :- !.
'?ground?'([V=V|T], [V|R]) :-
    '?ground?'(T, R).

writeseq([]).
writeseq([H|T]) :-
    write(H),
    'put%f'(32),
    writeseq(T).

writeseq(S, L) :-
    current_output(Out),
    set_output(S),
    writeseq(L),
    set_output(Out).

writeqseq([]).
writeqseq([H|T]) :-
    writeq(H),
    'put%f'(32),
    writeqseq(T).

writeqseq(S, L) :-
    current_output(Out),
    set_output(S),
    writeqseq(L),
    set_output(Out).

writenl(X) :-
    write(X), nl.

writenl(S, L) :-
    current_output(Out),
    set_output(S),
    writenl(L),
    set_output(Out).

writeqnl(X) :-
    writeq(X), nl.

writeqnl(S, L) :-
    current_output(Out),
    set_output(S),
    writeqnl(L),
    set_output(Out).

writeseqnl(X) :-
    writeseq(X), nl.

writeseqnl(S, L) :-
    current_output(Out),
    set_output(S),
    writeseqnl(L),
    set_output(Out).

writeqseqnl(X) :-
    writeqseq(X), nl.

writeqseqnl(S, L) :-
    current_output(Out),
    set_output(S),
    writeqseqnl(L),
    set_output(Out).

nl :-
    'put%f'(0'\n).

nl(S) :-
    current_output(Out),
    set_output(S),
    'put%f'(0'\n),
    set_output(Out).

get0(S, X) :-
	current_input(In),
	set_input(S),
	('get0%f'(X)-> set_input(In); set_input(In), fail).

get(X) :-
	'get0%f'(C),
	'$checkget$'(C, X).

get(S, X) :-
	current_input(In),
	set_input(S),
	(get(X)-> set_input(In); set_input(In), fail).

'$checkget$'(26, X) :- !, X = 26.
'$checkget$'(C, X) :-
	C > 32, C < 127, !,
	X = C.
'$checkget$'(_, X) :- get(X).

skip(Char) :-
    'int%f'(Char),
    repeat,
    'get0%f'(C),
    C == Char, !.
skip(Exp) :-
    '$ primcatch'('?is?'(Exp,X), skip(Exp)),
    skip(X).

skip(S, Exp) :-
	current_input(In),
	set_input(S),
	(skip(Exp)-> set_input(In); set_input(In), fail).

put(X) :-
    'int%f'(X), !,
    'put%f'(X).
put(Exp) :-
    '$ primcatch'('?is?'(Exp,X), put(Exp)),
    'put%f'(X).

put(S, X) :-
	current_output(Out),
	set_output(S),
	(put(X)-> set_output(Out); set_output(Out), fail).

tab(S, X) :-
	current_output(Out),
	set_output(S),
	('tab%f'(X)-> set_output(Out); set_output(Out), fail).


/***************************************************/
/*             File and Stream Handling            */
/***************************************************/

open(X,Y,Z)     :- '$ primcatch'('?open?'(X,Y,Z), open(X,Y,Z)).

'?open?'(File, Mode, Where) :-
    'atom%f'(File), !,  
    '$open$'(File, Mode, Where).
'?open?'(_, _, _) :-
    'throw%f'(203).

'$open$'(File, Mode, Where) :-
    'var%f'(Where), !,
    '?filemode?'(Mode, X),
    Where = stream(N),
    '%open%'(File, X, N).
'$open$'(_, _, _) :-
    'throw%f'(205).

'%open%'(File, X, N) :-
    'o_stream%f'(File, X, N).
'%open%'(_, _, _) :-
    throw(300).

'?filemode?'(Mode, _) :- 
    'var%f'(Mode), !,
    'throw%f'(103).
'?filemode?'(read, 0) :- !.
'?filemode?'(write, 1) :- !.
'?filemode?'(append, 2) :- !.
'?filemode?'(_, _) :- 
    'throw%f'(400).

open_ram(X,Y,Z) :- '$ primcatch'('?open_ram?'(X,Y,Z,_), open_ram(X,Y,Z)).
open_ram(V,X,Y,Z) :- '$ primcatch'('?open_ram?'(V,X,Y,Z), open_ram(V,X,Y,Z)).

'?open_ram?'(Const, Mode, Where, Size) :-
    'atom%f'(Const), !,
    '$open_ram$'(Const, Mode, Where, Size).
'?open_ram?'(Ram, Mode, Where, Size) :-
    'tpl%f'(Ram),
    Ram = memory(N), !,
    '$open_ram$'(N, Mode, Where, Size).
'?open_ram?'(_, _, _, _) :-
    'throw%f'(206).

'$open_ram$'(Const, Mode, Where, Size) :-
    'var%f'(Where), !,
    '?filemode?'(Mode, X),
    Where = memory(N),
    'o_mem%f'(Const, X, N, Size).
'$open_ram$'(_, _, _, _) :-
    'throw%f'(205).

set_input(X)    :- '$ primcatch'('?set_input?'(X), set_input(X)).

'?set_input?'(X) :-
    'var%f'(X), !,
    'throw%f'(102).
'?set_input?'(user) :- !,
    '?set_input?'(user_input).
'?set_input?'(user_input) :- !,
    thread(Th),
    get_prop(Th, initial, (_, In, _, _)),
    '?set_input?'(In).
'?set_input?'(stream(N)) :- !,
    'set_in%f'(0, N).
'?set_input?'(memory(N)) :-
    'set_in%f'(1, N).

set_output(X)   :- '$ primcatch'('?set_output?'(X), set_output(X)).

'?set_output?'(X) :-
    'var%f'(X), !,
    'throw%f'(102).
'?set_output?'(user) :- !,
    '?set_output?'(user_output).
'?set_output?'(user_output) :- !,
    thread(Th),
    get_prop(Th, initial, (_, _, Out, _)),
    '?set_output?'(Out).
'?set_output?'(user_error) :- !,
    thread(Th),
    get_prop(Th, initial, (_, _, _, Err)),
    '?set_output?'(Err).
'?set_output?'(stream(N)) :- !,
    'set_out%f'(0, N).
'?set_output?'(memory(N)) :-
    'set_out%f'(1, N).

current_input(X) :-
    'curr_input%f'(Type, Index),
    '?decode_io?'(Type, Index, X).
current_output(X) :-
    'curr_output%f'(Type, Index),
    '?decode_io?'(Type, Index, X).

'?decode_io?'(1, N, stream(N)).
'?decode_io?'(2, N, memory(N)).
'?decode_io?'(3, N, port(N)).
'?decode_io?'(4, N, socket(N)).

close(X)        :- '$ primcatch'('?close?'(X), close(X)).

'?close?'(X) :-
    'var%f'(X), !,
    'throw%f'(102).
'?close?'(user) :- !.
'?close?'(user_output) :- !.
'?close?'(user_error) :- !.
'?close?'(stream(N)) :- !,
    'c_stream%f'(N).
'?close?'(memory(N)) :- !,
    'c_mem%f'(N).
'?close?'(port(N)) :- !,
    'close_port%f'(N).
'?close?'(_) :-
    'throw%f'(301).

ram_const(X,Y)  :- '$ primcatch'('?ram_const?'(X,Y), ram_const(X,Y)).

'?ram_const?'(X, _) :-
    'var%f'(X), !,
    'throw%f'(102).
'?ram_const?'(memory(N), X) :- !,
    '$ram_const$'(N, X).
'?ram_const?'(_, _) :-
    'throw%f'(301).

flush :-
    current_output(Out),
    flush_output(Out).

flush_output(X) :- '$ primcatch'('?flush_output?'(X), flush_output(X)).

'?flush_output?'(X) :-
    'var%f'(X), !,
    'throw%f'(102).
'?flush_output?'(user) :- !,
    'flush%f'(1).
'?flush_output?'(user_output) :- !,
    'flush%f'(1).
'?flush_output?'(user_error) :- !,
    'flush%f'(2).
'?flush_output?'(stream(N)) :- !,
    'flush%f'(N).
'?flush_output?'(memory(_)) :- !.
'?flush_output?'(_) :-
    'throw%f'(301).

file_exists(File) :- 'stat%f'(File, _, _, _, _, _, _).

cursor(stream(N), X) :-
	'cursor%f'(N, X, _).


/***************************************************/
/*         Command line Argument Processing        */
/***************************************************/

argv(L) :- get_prop('$prolog$', '$argv$', L).
argc(N) :- get_prop('$prolog$', '$argv$', L), '?length?'(L, N).


/***************************************************/
/*                   TTY Handling                  */
/***************************************************/

tty(Name, stream(Read), stream(Write)) :-
    'tty%f'(Name, Read, Write).

