/*  public list :
        [write/1, writeq/1, display/1, print/1, write_canonical/1,
         write/2, writeq/2, display/2, print/2, write_canonical/2,
	 writeq/3]
*/

/*
    IC Prolog ][ Term Writer
    by Damian Chu
    June, 1990

    Based on Richard O'Keefe's public domain term writer, with substantial
    modifications for speed.  It uses many ICP primitives which are not
    available to other Prologs, so it is not portable.

    The four output routines differ in the following respects:
    [a] display doesn't use operator information or handle {X} or
        [H|T] specially.  The others do.
    [b] print calls portray/1 to give the user a chance to do
        something different.  The others don't.
    [c] writeq puts quotes around atoms that can't be read back.
        The others don't.
    Since they have such a lot in common, we just pass around a
    single Style argument to say what to do.

    $VAR(X) will write the atom X without quotes, this so that you can write
    out a clause in a readable way by binding each input variable to its name.
*/



write_canonical(Term) :-
        write_out(write_canonical, Term, 1200, punct, _).

write_canonical(S, Term) :-
        current_output(Out),
        set_output(S),
        write_canonical(Term),
        set_output(Out).

display(Term) :-
        write_out(display, Term, 1200, punct, _).

display(S, Term) :-
        current_output(Out),
        set_output(S),
        display(Term),
        set_output(Out).

print(Term) :-
        write_out(print, Term, 1200, punct, _).

print(S, Term) :-
        current_output(Out),
        set_output(S),
        print(Term),
        set_output(Out).

write(Term) :-
        write_out(write, Term, 1200, punct, _).

write(S, Term) :-
        current_output(Out),
        set_output(S),
        write(Term),
        set_output(Out).

writeq(Term) :-
        write_out(writeq, Term, 1200, punct, _).

writeq(S, Term) :-
        current_output(Out),
        set_output(S),
        writeq(Term),
        set_output(Out).

writeq(S, Term, Names1) :-
	list(Names1),
	current_output(Out),
	set_output(S),
	toground(Term, Ground, _, Names2),
	append(Names1, Names2, Names),
	tohollow(Ground, Hollow, Names, Vars),
	bind_name(Names, Vars),
	writeq(Hollow),
	set_output(Out).

bind_name([], []).
bind_name([H|T], ['$VAR'(H)|Rest]) :-
	bind_name(T, Rest).



%   write_out(Style, Term, Priority, Ci, Co)
%   writes out a Term in a given Style (display,write,writeq,print)
%   in a context of priority Priority (that is, operators with
%   greater priority have to be quoted), where the last token to be
%   written was of type Ci, and reports that the last token it wrote
%   was of type Co.

write_out(write_canonical, Term, Priority, Ci, Co) :-
        'tag%f'(Term, N),
        display_out(N, Term, Priority, Ci, Co, write_canonical).
write_out(display, Term, Priority, Ci, Co) :-
        'tag%f'(Term, N),
        display_out(N, Term, Priority, Ci, Co, display).
write_out(print, Term, Priority, Ci, Co) :-
        'tag%f'(Term, N),
        print_out(N, Term, Priority, Ci, Co).
write_out(write, Term, Priority, Ci, Co) :-
        'tag%f'(Term, N),
        generic_write_out(N, Term, write, Priority, Ci, Co).
write_out(writeq, Term, Priority, Ci, Co) :-
        'tag%f'(Term, N),
        generic_write_out(N, Term, writeq, Priority, Ci, Co).



display_out(0, Term, _, Ci, alpha, _) :- !,
        maybe_space(Ci, alpha),
        write_variable(Term).
display_out(1, N, _, Ci, alpha, _) :- !,
        write_number(Ci, N).
display_out(2, N, _, Ci, alpha, _) :- !,
        write_float(Ci, N).
display_out(3, Atom, Prio, _, punct, Style) :-
        current_op(P, _, Atom),
        P > Prio, !,
        put(40),
	(   Style = write_canonical, write_atom(Atom, Style, punct, _)
	;   'put_atom%f'(Atom)
	), !,
        put(41).
display_out(3, Atom, _, Ci, Co, Style) :- !,
        write_atom(Atom, Style, Ci, Co).
display_out(4, [], _, _, punct, _) :- !,
        put(91), put(93).
display_out(_, Term, _, Ci, punct, Style) :-
        functor(Term, Fsymbol, Arity),
        'tag%f'(Fsymbol, N),
        display_out(N, Fsymbol, 1200, Ci, _, Style),
        write_args(0, Arity, Term, 40, Style).



print_out(0, Term, _, Ci, alpha) :- !,
        maybe_space(Ci, alpha),
        write_variable(Term).
print_out(6, F(N), _, Ci, Co) :-
        F == '$VAR', !,
        write_VAR(N, print, Ci, Co).
print_out(_, Term, _, _, alpha) :-
        defined(portray/1),
	portray(Term), !.
print_out(N, Term, Priority, Ci, Co) :-
        generic_write_out(N, Term, print, Priority, Ci, Co).



generic_write_out(0, Term, _, _, Ci, alpha) :-
        maybe_space(Ci, alpha),
        write_variable(Term).
generic_write_out(1, N, _, _, Ci, alpha) :-
        write_number(Ci, N).
generic_write_out(2, N, _, _, Ci, alpha) :-
        write_float(Ci, N).
generic_write_out(3, Atom, Style, Prio, _, punct) :-
        current_op(P, _, Atom),
        P > Prio, !,
        put(40),
        (   Style = writeq, write_atom(Atom, Style, punct, _)
        ;   'put_atom%f'(Atom)
        ),  !,
        put(41).
generic_write_out(3, Atom, Style, _, Ci, Co) :-
        write_atom(Atom, Style, Ci, Co).
generic_write_out(4, [], _, _, _, punct) :-
        put(91), put(93).
generic_write_out(5, List, _, _, _, punct) :-
	get_prop(display, strings, on),
        possible_string(List), !,
        'put_str%f'(List, 34).
generic_write_out(5, [Head|Tail], Style, _, _, punct) :-
        put(91),
        write_out(Style, Head, 999, punct, _),
        write_tail(Tail, Style).
generic_write_out(6, Term, Style, Prio, Ci, Co) :-
        'grnd_funct%f'(Term), !,
        write_functor(Term, Style, Prio, Ci, Co).
generic_write_out(6, Term, Style, _, Ci, punct) :-   % variable functor
        functor(Term, F, N),
        write_out(Style, F, 0, Ci, _),
        write_args(0, N, Term, 40, Style).

write_functor('$VAR'(N), Style, _, Ci, Co) :- !,
        write_VAR(N, Style, Ci, Co).
write_functor('{}'(Term), Style, _, _, punct) :- !,
        put(123),
        write_out(Style, Term, 1200, punct, _),
        put(125).
write_functor(','(A,B), Style, Prio, Ci, Co) :- !,
        %  This clause stops writeq quoting commas.
        maybe_paren(1000, Prio, 40, Ci, C1),
        write_out(Style, A, 999, C1, _),
        put(44),
        write_out(Style, B, 1000, punct, C2),
        maybe_paren(1000, Prio, 41, C2, Co).
write_functor(Term, Style, Prio, Ci, Co) :-
        functor(Term, F, N),
        write_out(N, F, Term, Style, Prio, Ci, Co).



% dac 6/6/90
% rewrote this so that for non-operators, we do not call
% current_op eight times as before !  (Also faster for operators)
write_out(1, F, Term, Style, Prio, Ci, Co) :-
        'atom%f'(F),
        'op_look%f'(F, O, P, _, _, _, _, _),
        O =< 1200, !,
        maybe_paren(O, Prio, 40, Ci, C1),
        write_atom(F, Style, C1, _),
        arg(1, Term, A),
        put(32),
        write_out(Style, A, P, punct, C3),
        maybe_paren(O, Prio, 41, C3, Co).
write_out(1, F, Term, Style, Prio, Ci, Co) :-
        'atom%f'(F),
        'op_look%f'(F, _, _, _, _, _, O, P),
        O =< 1200, !,
        maybe_paren(O, Prio, 40, Ci, C1),
        arg(1, Term, A),
        write_out(Style, A, P, C1, C2),
        write_atom(F, Style, C2, C3),
        maybe_paren(O, Prio, 41, C3, Co).
write_out(2, F, Term, Style, Prio, Ci, Co) :-
        'atom%f'(F),
        'op_look%f'(F, _, _, L, O, R, _, _),
        O =< 1200, !,
        maybe_paren(O, Prio, 40, Ci, C1),
        arg(1, Term, A),
        write_out(Style, A, L, C1, C2),
        write_oper(F, O, Style, C2, C3),
        arg(2, Term, B),
        write_out(Style, B, R, C3, C4),
        maybe_paren(O, Prio, 41, C4, Co).
% dac 24/2/90
% write out functor as a term0
write_out(N, F, Term, Style, _, Ci, punct) :-
        write_out(Style, F, 0, Ci, _),
        write_args(0, N, Term, 40, Style).

write_oper(Op, Prio, Style, Ci, Co) :-
        Prio < 700, !,
        write_atom(Op, Style, Ci, Co).
write_oper(Op, _, Style, _, punct) :-
        put(32),
        write_atom(Op, Style, punct, _),
        put(32).



%   write_args(DoneSoFar, Arity, Term, Separator, Style)
%   writes the remaining arguments of a Term with Arity arguments
%   all told in Style, given that DoneSoFar have already been written.
%   Separator is 0'( initially and later 0', .

% dac 10/11/89
% We allow functors with 0 arity, which displays as "functor()"
write_args(0, 0, _, _, _) :- !,
        put(40),
        put(41).
write_args(N, N, _, _, _) :- !,
        put(41).
write_args(I, N, Term, C, Style) :-
        put(C),
        'increment%f'(I, J),
        arg(J, Term, A),
        write_out(Style, A, 999, punct, _),
        write_args(J, N, Term, 44, Style).



% If all the list elements are in the range 32 to 127,
% write it out as a string.
possible_string(X) :-
        X==[], !.
possible_string([H|T]) :-
        integer(H),
        H >= 32, H =< 127,
        possible_string(T).

%   write_tail(Tail, Style)
%   writes the tail of a list of a given style.
write_tail(Var, _) :-                   %  |var]
        var(Var), !,
        put(124),
        write_variable(Var),
        put(93).
write_tail([], _) :- !,                 %  ]
        put(93).
write_tail([Head|Tail], Style) :- !,    %  ,Head tail
        put(44),
        write_out(Style, Head, 999, punct, _),
        write_tail(Tail, Style).
write_tail(Other, Style) :-             %  |junk]
        put(124),
        write_out(Style, Other, 999, punct, _),
        put(93).

write_number(Context, N) :-
        N < 0, !,
        maybe_space(Context, other),
        'put_number%f'(N).
write_number(Context, N) :-
        maybe_space(Context, alpha),
        'put_number%f'(N).

write_float(Context, N) :-
        N < 0, !,
        maybe_space(Context, other),
        'put_float%f'(N).
write_float(Context, N) :-
        maybe_space(Context, alpha),
        'put_float%f'(N).

/* 22/9/89 Damian Chu.  Write a variable by prepending an '_' to the address */
write_variable(V) :-
        address(V,N),
        put(95),        % an '_'
        'put_number%f'(N).

write_VAR(N, _, Ci, alpha) :-
        integer(N), N >= 0, !,
        maybe_space(Ci, alpha),
        Letter is N mod 26 ++ 65,
        put(Letter),
        (   N < 26
        ;   Rest is N//26, 'put_number%f'(Rest)
        ), !.
write_VAR(A, _, Ci, Co) :-
        atom(A), !,
        (   'prefix%f'(A, '_$'),
            put("_"), Co = alpha
        ;   write_atom(A, write, Ci, Co)
        ), !.
write_VAR(X, Style, Ci, punct) :-
        write_quoted(Style, '$VAR', Ci, _),
        put(40),
        write_out(Style, X, 999, punct, _),
        put(41).

% The following three clauses have been folded into the C code of 'atom_type%f'
% write_atom(('!'), _, _, punct) :- !,
%       put(33).
% write_atom((';'), _, _, punct) :- !,
%       put(59).
% write_atom('{}', _, _, punct) :- !,
%       put(123), put(125).
write_atom(Atom, _, Ci, Co) :-
        classify_name(Atom, Co), !,
        maybe_space(Ci, Co),
        'put_atom%f'(Atom).
write_atom(Atom, Style, Ci, Co) :-
        write_quoted(Style, Atom, Ci, Co).

write_quoted(writeq, Atom, Ci, quote) :- !,
        maybe_space(Ci, quote),
        'put_q_atom%f'(Atom).
write_quoted(write_canonical, Atom, Ci, quote) :- !,
        maybe_space(Ci, quote),
        'put_q_atom%f'(Atom).
write_quoted(_, Atom, _, alpha) :-
        'put_atom%f'(Atom).



%   classify_name(String, Co)
%   says whether a String is an alphabetic identifier starting
%   with a lower case letter (Co=alpha) or a string of symbol characters
%   like ++/=? (Co=other).  If it is neither of these, it fails.  That
%   means that the name needs quoting.  The special atoms ! ; [] {} are
%   handled directly in write_atom.  In a basic Prolog system with no
%   way of changing the character classes this information can be
%   calculated when an atom is created, and just looked up.  This has to
%   be as fast as you can make it.

classify_name(L, Type) :-
        'atom_type%f'(L, Index),
        types(Index, Type).

types(1, alpha).
types(2, other).
types(3, punct).



%   maybe_paren(P, Prio, Char, Ci, Co)
%   writes a parenthesis if the context demands it.

maybe_paren(P, Prio, Char, _, punct) :-
        P > Prio, !,
        put(Char).
maybe_paren(_, _, _, C, C).



%   maybe_space(LeftContext, TypeOfToken)
%   generates spaces as needed to ensure that two successive
%   tokens won't run into each other.

maybe_space(punct, _) :- !.
maybe_space(X, X) :- !,
        put(32).
maybe_space(quote, alpha) :- !,
        put(32).
maybe_space(_, _).
