/*  $Id: compile.pl,v 1.2 92/02/25 17:52:29 jan Exp Locker: jan $

    Part of XPCE
    Designed and implemented by Anjo Anjewierden and Jan Wielemaker
    E-mail: jan@swi.psy.uva.nl

    Copyright (C) 1992 University of Amsterdam. All rights reserved.
*/



:- module(pce_compile,
	[ pce_begin_class/2
	, pce_begin_class/3
        , pce_extend_class/1
	, pce_end_class/0
	, pce_expand/2
	, pce_current_class/1
	]).

:- use_module(library(pce_operator)).

?- dynamic
	compiling/1,
	load_module/1,
	verbose/0.

/*
trace :-
	style_check(+dollar),
	system:trace.
*/

:- module_transparent
	pce_begin_class/2,
	pce_begin_class/3,
	pce_extend_class/1.


		/********************************
		*           BEGIN CLASS		*
		********************************/


%	pce_begin_class(+TermDef, +Super[, Documentation])
%	Start definition of a class whose name is the functor of `TermDef'
%	and whose object/2 arguments are created using the selectors from
%	the arguments of `TermDef',  `Super' is the name of the superclass

pce_begin_class(TermDef, Super) :-
	pce_begin_class(TermDef, Super, "").

pce_begin_class(TermDef, Super, Doc) :-
	verify_not_nested,
	context_module(Module),
/*	asserta(pce_compile:load_module(Module)), */
	asserta(load_module(Module)),
	pce_begin_class_(TermDef, Super, Doc).
	

pce_begin_class_(TermDef, Super, Doc) :-
	TermDef =.. [ClassName|TermArgs],
	class_name(Class, ClassName),
	object(Class), !,			% redefined existing class
	(   get(Class, creator, built_in)
	->  error('Cannot redefine built-in class: ~w', [ClassName]),
	    fail
	;   true
	),
	get(Class, super_class, SuperClass),
	(   (   Super == @nil,
		SuperClass == @nil
	    ;   SuperClass \== @nil,
		class_name(SuperClass, Super)
	    )
	->  true
        ;   error('Cannot change super-class of Class ~w', [ClassName])
    	),
	term_names(Class, TermArgs),
	set_source_location(Class),
	feedback('Reloading PCE class ~w ...~n', [ClassName]),
	start_class(ClassName, Doc).
pce_begin_class_(TermDef, SuperName, Doc) :-
	TermDef =.. [ClassName|TermArgs],
	(   get(@pce, convert, SuperName, class, Super)
	->  true
	;   error('Superclass ~w of ~w does not exist', [SuperName, ClassName])
	), !,
	get(Super, class_name, ClassClass),
	ClassTerm =.. [ClassClass, ClassName, Super],
	new(Class, ClassTerm),
	term_names(Class, TermArgs),
	set_source_location(Class),
	feedback('Loading PCE class ~w ...~n', [ClassName]),
	start_class(ClassName, Doc).


source_location_term(source_location(File, Line)) :-
	source_location(File, Line), !.
source_location_term(@nil).

set_source_location(Obj) :-
	source_location_term(Loc),
	send(Obj, source, Loc).


term_names(_, []) :- !.
term_names(Class, Selectors) :-
	VectorTerm =.. [vector|Selectors],
	send(Class, term_names, new(VectorTerm)).

pce_extend_class(ClassName) :-
	verify_not_nested,
	context_module(Module),
/*	asserta(pce_compile:load_module(Module)), */
	asserta(load_module(Module)),
	class_name(Class, ClassName),
	object(Class),
	feedback('Extending PCE class ~w ...~n', [ClassName]),
	start_class(ClassName).


%	start_class(+ClassName[, +Doc])
%	Install term-expansion predicate and operators to perform the actual
%	loading of the class.

start_class(ClassName, "") :- !,
	start_class(ClassName).
start_class(ClassName, Doc) :-
	class_name(Class, ClassName),
	send(Class, summary, string(Doc)),
	start_class(ClassName).

start_class(ClassName) :-
	asserta(compiling(ClassName)),
	push_compile_operators.

%	pce_end_class.
%	Restore the old environment for term_expansion and operators.

pce_end_class :-
	retractall(load_module(_)),
	retractall(compiling(ClassName)),
	pop_compile_operators,
	feedback('Class ~w loaded~n', [ClassName]).

%	verify_not_nested
%	Warn if the a begin_class is not ended.

verify_not_nested :-
	compiling(ClassName), !,
	error('Definition of class ~w is not closed', ClassName).
verify_not_nested.


%	push_compile_operators.
%	Push the current 
/*
push_compile_operators :-
	push_operators(
		[ op(1200, xfx, :->)
		, op(1200, xfx, :<-)
		, op(1190, xfx, ::)
		, op(100,  xf,  *)
		, op(125,  xf,  ?)
		, op(150,  xf,  ...)
		, op(100,  xfx, ..)
		]).

pop_compile_operators :-
	pop_operators.
*/	

%	current_class(-Class)
%	Class for which we are currently compiling.

current_class(Class) :-
	compiling(ClassName),
	class_name(Class, ClassName).

%	pce_current_class(-ClassName)
%	External function to get the current classname

pce_current_class(ClassName) :-
	compiling(ClassName).

		/********************************
		*            EXPAND		*
		********************************/

:- push_compile_operators.

pce_expand(Term, Expanded) :-
	class_term(Term),
	compiling(_),
	(   do_expand(Term, Expanded)
	->  true
	;   error('Failed to expand ~w', [Term]),
	    Expanded = []
	).

class_term(variable(_Name, _Type, _Access)).
class_term(variable(_Name, _Type, _Access, _Doc)).
class_term(resource(_Name, _Type, _Default)).
class_term(resource(_Name, _Type, _Default, _Doc)).
class_term(handle(_X, _Y, _Kind)).
class_term(handle(_X, _Y, _Kind, _Name)).
class_term((_Head :-> _Body)).
class_term((_Head :<- _Body)).


:- discontiguous
	do_expand/2.


		/********************************
		*           DO_EXPAND		*
		********************************/

do_expand(variable(Name, Type, Acs),
	(?- send(Class, instance_variable, variable(Name, PceType, Acs)))) :- !,
	access(Acs),
	current_class(Class),
	type(Type, PceType).
do_expand(variable(Name, Type, Acs, Doc),
	(?- send(Class, instance_variable, variable(Name, PceType, Acs,
						   string(Doc))))) :- !,
	access(Acs),
	current_class(Class),
	type(Type, PceType).

do_expand(resource(Name, Type, Def),
	(?- send(Class, resource, resource(Name, @default, Type, Def,
					  Class)))) :- !,
	current_class(Class).
do_expand(resource(Name, Type, Def, Doc),
	(?- send(Class, resource, resource(Name, @default, Type, Def,
					  Class, string(Doc))))) :- !,
	current_class(Class).

do_expand(handle(X, Y, Kind),
	(?- send(Class, handle, handle(X, Y, Kind)))) :- !,
	current_class(Class).

do_expand(handle(X, Y, Kind, Name),
	(?- send(Class, handle, handle(X, Y, Kind, Name)))) :- !,
	current_class(Class).

do_expand((Head :-> DocBody),			% Prolog send
	[ (?- send(Class, send_method,
		  send_method(Selector, Types, Cascade, Doc, Loc)))
	, (PlHead :- Body)
	]) :- !,
	(   DocBody = (DocText::Body)
	->  Doc = string(DocText)
	;   DocBody = Body,
	    Doc = @nil
	),
	source_location_term(Loc),
	current_class(Class),
	class_name(Class, ClassName),
	prolog_head(send, Head, Selector, Types, PlHead, Cascade),
	feedback('~t~8|~w :->~w ... ok~n', [ClassName, Selector]).


do_expand((Head :<- DocBody),			% Prolog get
	[ (?- send(Class, get_method,
		  get_method(Selector, RType, Types, Cascade, Doc, Loc)))
	, (PlHead :- Body)
	]) :- !,
	(   DocBody = (DocText::Body)
	->  Doc = string(DocText)
	;   DocBody = Body,
	    Doc = @nil
	),
	source_location_term(Loc),
	current_class(Class),
	class_name(Class, ClassName),
	return_type(Head, RType),
	prolog_head(get, Head, Selector, Types, PlHead, Cascade),
	feedback('~t~8|~w :<-~w ... ok~n', [ClassName, Selector]).


return_type(Term, RType) :-
	functor(Term, _, Arity),
	arg(Arity, Term, Last),
	(   nonvar(Last),
	    Last = _:Type
	->  type(Type, RType)
	;   RType = @default
	).


prolog_head(SendGet, Head, Selector, TypeVector, PlHead, Cascade) :-
	Head =.. [Selector, Receiver | Args],
	predicate_name(SendGet, Selector, PredName),
	pl_head_args(SendGet, Args, Types, PlArgs, FArgs),
	create_type_vector(Types, TypeVector),
	PlHead =.. [PredName, Receiver | PlArgs],
	(   SendGet == send
	->  Class = message
	;   Class = ?
	),
	Cascade =.. [Class, @prolog, call, PredName, @receiver | FArgs].

create_type_vector([],      @default) :- !.
create_type_vector(List,    new(VectorTerm)) :-
	VectorTerm =.. [vector|List].


predicate_name(SendGet, Selector, Name) :-
	current_class(Class),
	class_name(Class, ClassName),
	concat_atom([SendGet, '_', Selector, '_', ClassName], Name).


pl_head_args(SendGet, Args, Types, PlArgs, FArgs) :-
	pl_head_args(SendGet, Args, 1, Types, PlArgs, FArgs).
	
pl_head_args(send, [], _, [], [], []) :- !.
pl_head_args(get,  [Return], _, [], [ReturnVar], []) :- !,
	(   var(Return)
	->  ReturnVar = Return
	;   Return = ReturnVar:_Type
	).
pl_head_args(SG, [ArgAndType|RA], AN, [T|RT], [Arg|TA], [@ArgN|MArgs]) :- !,
	head_arg(ArgAndType, Arg, Type),
	type(Type, T),
	concat(arg, AN, ArgN),
	ANN is AN + 1,
	pl_head_args(SG, RA, ANN, RT, TA, MArgs).


head_arg(Var, Var, any) :-
	var(Var), !.
head_arg(Arg:Type, Arg, Type).

%	class_name(?Class, ?ClassName)
%	Convert between PCE class-name and PCE Class object

class_name(Class, ClassName) :-
	object(Class), !,
	get(Class, name, ClassName).
class_name(Class, ClassName) :-
	atom(ClassName), !,
	get(@classes, member, ClassName, Class).
class_name(_, _) :-
	error('class_name/2: Instantiation fault', []),
	fail.

%	access(?Access)
%	List of legal access names.

access(none).
access(get).
access(send).
access(both).

%	type(+Spec, -PceTypeName)
%	Convert type specification into legal PCE type-name

type(Type ..., T) :- !,
	type(Type, Atom),
	concat(Atom, ' ...', T).
type(Type?, T) :- !,
	type(Type, Atom),
	concat(Atom, ?, T).
type([PrimType]*, T) :-	!,
	prim_type(PrimType, Atom),
	concat_atom(['[', Atom, ']*'], T).
type(PrimType*, T) :- !,
	prim_type(PrimType, Atom),
	concat(Atom, '*', T).
type([PrimType], T) :- !,
	prim_type(PrimType, Atom),
	concat_atom(['[', Atom, ']'], T).
type(PrimType, T) :-
	prim_type(PrimType, T).

prim_type(atom,		name) :- !.
prim_type(integer,	int) :- !.
prim_type(boolean,	bool) :- !.
prim_type('{}'(Atoms),	Atom) :- !,
	name_of_type(Atoms, List),
	concat_atom(['{'|List], Atom).
prim_type(A..B,		Atom) :-
	number(A), number(B), !,
	concat_atom([A, '..', B], Atom).
prim_type(Class,	Class).

name_of_type((A,B), [A, ','| NB]) :- !,
	name_of_type(B, NB).
name_of_type(A, [A, '}']).

:- pop_compile_operators.

		/********************************
		*            FEEDBACK		*
		********************************/

%	error(+Format, +Arguments)
%	Print standard error message.

error(Format, Arguments) :-
	format(user_error, '[PCE Class compiler: ', []),
	format(user_error, Format, Arguments),
	format(user_error, ']~n', []).


%	feedback(+Format, +Arguments)
%	Print standard feedback message.

feedback(Format, Arguments) :-
	verbose, !,
	format(user_output, Format, Arguments),
	flush_output(user_output).
feedback(_, _).

		/********************************
		*           UTILITIES		*
		********************************/

term_member(El, Term) :-
	El == Term.
term_member(El, Term) :-
	functor(Term, _, Arity),
	term_member(Arity, El, Term).

term_member(0, _, _) :- !,
	fail.
term_member(N, El, Term) :-
	arg(N, Term, Sub),
	term_member(El, Sub).
term_member(N, El, Term) :-
	NN is N - 1,
	term_member(NN, El, Term).


		/********************************
		*         TERM EXPANSION	*
		********************************/

?- user:assert((term_expansion(A, B) :- pce_expand(A, B))).



