/*
 *  icp.h - main header file for IC-Porlog ][ emulator
 *  Written by Frank McCabe, Damian Chu and Yannis Cosmadopoulos
 *  Imperial College, 1989, 1992
 */

#include <stdio.h>
#include <signal.h>
#include <setjmp.h>
#include <string.h>
#include <math.h>

/* uncomment the next line if running SunOS >= 4.1 */
#define SUNOS41

/* the following statement enables debugging info */
/* #define DEBUG	/* uncomment this line, or define in Makefile */

/* comment out if can index on any argument */
#define INDEX_FIRST_ARG

/*-----------------------------------------------------------------------*/

/* System dependent stuff, more to be added ... */

#if defined(THINK_C)		/* Mac Systems */
#define ANSI
#include <console.h>
#define EMPTY
#define READBIN		"rb"
#define WRITEBIN	"wb"
#define APPENDBIN	"ab"
#define SIZE_TYPE	size_t
#define FLOAT		short double

#elif defined(GNUDOS)	/* GCC under DOS */
#define ANSI
#define EMPTY 0
#define READBIN		"rb"
#define WRITEBIN	"wb"
#define APPENDBIN	"ab"
#define SIZE_TYPE	int
#define FLOAT		double
#define CLOCKS_PER_SEC	1E3
#define size_t		unsigned
typedef long 		clock_t;

#else		/*	Default System: Suns */

#if defined(GCC) || defined(__GNUC__)
#define ANSI
#else
extern char *memcpy();
#endif

#ifdef lint
#define EMPTY		1
#else
#define EMPTY		0
#endif

#define READBIN		"r"
#define WRITEBIN	"w"
#define APPENDBIN	"a"
#define SIZE_TYPE	int
#define FLOAT		double
#define CLOCKS_PER_SEC	1E6
#define size_t		unsigned

#ifndef SUNOS41
typedef long		clock_t;
#endif

#endif

/*-----------------------------------------------------------------------*/

/* configuration parameters */

/* Minimum number of cells between top of local stack and top of trail.
 * Needed to prevent stacks from colliding.  May need a higher figure. */
#define FUZZ		20

#define REGISTERS	32	/* number of registers */
#define PDLSIZE		128	/* size of push down list for unification */
#define DISPLREG        8       /* default number of register displayed */ 
#define AUDITSIZE       100     /* size of audit table */
#define MAXFILENAME	255	/* max. length of a filename */

/*-----------------------------------------------------------------------*/

/* Debugging - the levels are :
 *	0 - none,
 *	1 - call level tracing,
 *	2 - instruction level tracing
 *	3 - extra info (e.g. unification, g/c statistics)
 *	4 - debug garbage collector
 */
#define MAXDUMP		4
#define STARTLEVEL	2

#ifdef DEBUG
#define debug(level,stmnts)	if (debugLevel >= level) {stmnts;}
#define debug2(level,stmnts)	if (debugLevel == level) {stmnts;}
#else
#define debug(level,stmnt)
#define debug2(level,stmnt)
#endif

/*-----------------------------------------------------------------------*/

/* General purpose declarations */

/* define our own integer for portability */
typedef	long		fourBytes;
typedef unsigned long	ufourBytes;
typedef	short		twoBytes;
typedef	unsigned short	utwoBytes;
typedef	char		oneByte;
typedef	unsigned char	uchar;

/* boolean type */
typedef short		bool;
#define FALSE		0
#define TRUE		1

/* general purpose macros */
#define max(a,b)		((a)<(b)?(b):(a))
#define min(a,b)		((a)<(b)?(a):(b))

#define READ		"r"
#define WRITE		"w"
#define APPEND		"a"

/* declarations for common memory allocation functions */
#ifndef GNUDOS	/* already defined in the header */
extern char *malloc();
extern char *calloc();
extern char *realloc();
extern char *memalign();
extern free();
#endif

#ifdef GNUDOS
#define alloc(Number,Size)	calloc((size_t)(Number),(size_t)(Size))
#else
#define alloc(Number,Size)	memalign(4, (size_t)(Number)*(size_t)(Size))
#endif

/*-----------------------------------------------------------------------*/

#define CHARTYPE	int
typedef uchar		*strpo;	/* define a pointer to a print name */

/*-----------------------------------------------------------------------*/

/* this is the declaration of the cell structure */

/* the most important definition is the cell data structure */

typedef unsigned long	cell;
typedef cell		*cellpo;

#define var_ref		0x00000000
#define int_ref		0x20000000
#define float_ref	0x40000000
#define symb_ref	0x60000000
#define nil_ref		0x80000000
#define list_ref	0xa0000000
#define tpl_ref		0xc0000000
#define guard_ref	0xe0000000

#define int_mask	0xfc000000

/* accessing macros for a cell */
/* see also gc.h for garbage collection bits in the tag */
/* reserve 3 TOP bits for tag, bottom 2 bits for G/C	*/
#define tg(x)		(*(x) & 0xe0000000)
#define tgc(x)		(*(x) & 0xe0000003)
#define vl(x)		(*(x) & 0x1ffffffc)
#define intvl(x)	((long)*(x) << 3 >> 5)
#define symbvl(x)	((symbpo) vl(x))
#define hd(x)		((cellpo) vl(x))
#define tl(x)		(((cellpo) vl(x)) + 1)
#define arity(x)	intvl((cellpo) vl(x))
#define functor(x)	(((cellpo) vl(x)) + 1)	
#define arg(x, n)	(((cellpo) vl(x)) + 1 + (n))

/* On Sun-4, floatvl is a function defined in arith.c */
#ifdef sparc
extern	FLOAT	floatvl();
#else
#define floatvl(x)	(* (FLOAT *) vl(x))
#endif

#define IsVar(addr)		(tg(addr) == var_ref)
#define IsInt(addr)		(tg(addr) == int_ref)
#define IsFloat(addr)		(tg(addr) == float_ref)
#define IsSymb(addr)		(tg(addr) == symb_ref)
#define IsNil(addr)		(tg(addr) == nil_ref)
#define IsList(addr)		(tg(addr) == list_ref)
#define IsTpl(addr)		(tg(addr) == tpl_ref)
#define IsGuard(addr)		(tg(addr) == guard_ref)

#define NotVar(addr)		(tg(addr) != var_ref)
#define NotInt(addr)		(tg(addr) != int_ref)
#define NotFloat(addr)		(tg(addr) != float_ref)
#define NotSymb(addr)		(tg(addr) != symb_ref)
#define NotNil(addr)		(tg(addr) != nil_ref)
#define NotList(addr)		(tg(addr) != list_ref)
#define NotTpl(addr)		(tg(addr) != tpl_ref)
#define NotGuard(addr)		(tg(addr) != guard_ref)

#define tgVal(addr)		(*(addr) >> 29)
#define pureTag(addr)		tg(addr)
#define sameTag(a,b)		(tg(a) == tg(b))

#define setval(addr, val)	*(addr) = (tgc(addr) | (cell) (val))

/* common cell operations */
#define mkint(addr,numb)	*(addr)=(int_ref | ((cell)(numb) & 0x07ffffff) << 2)
#define mksymb(addr,symb)	*(addr)=(symb_ref | (cell) (symb))
#define mknil(addr)		*(addr)=nil_ref
#define mkguard(addr,len)	*(addr)=(guard_ref | ((cell)(len)& 0x07ffffff) << 2)
#define mkunb(addr)		*(addr)=(var_ref | ((cell) (addr)))
#define mklnk(addr,dest)	*(addr)=(var_ref | (cell) (dest))
#define mklst(addr,list)	*(addr)=(list_ref | (cell) (list))
#define mktpl(addr,tuple)	*(addr)=(tpl_ref | (cell) (tuple))
#define mkflt(addr,flt)		*(addr)=(float_ref | (cell) (flt))
#ifdef GNUDOS
#define strsymb(Xaddr, Xlen, Xpt) { \
	register strpo target, pt = Xpt; \
	int hashtotal = 0; \
	target = symbname((symbpo) Xaddr); \
	while (*target = *pt++) \
		hashtotal += *target++; \
	*target = '\0'; \
	symbhashvalin((symbpo) Xaddr) = ((hashtotal & 0x0FFFF) | Xlen<< 16); \
}
#else
#define strsymb(Xaddr, Xlen, Xpt) { \
	register strpo target, pt = Xpt; \
	int hashtotal = 0; \
	target = symbname((symbpo) Xaddr); \
	while (*target = *pt++) \
		hashtotal += *target++; \
	*target = '\0'; \
	symbhashval((symbpo) Xaddr) = ((hashtotal & 0x0FFFF) | Xlen<< 16); \
}
#endif

#define mkint1(addr, num)	{ \
	fourBytes Zhigh, Zval = (num); \
	Zhigh = Zval & int_mask; \
	if (!Zhigh || (Zhigh == int_mask)) { \
		mkint(addr,Zval); \
	} else { \
		 /* some danger here with heap */ \
		 alloc_float(addr, Zval); \
	}}

#define mkreset(addr) \
	if ((addr) < HB || ((addr) >= BLS && (addr) < (cellpo)B)) \
		*--TR = (cell)(addr)
#define delnk(addr) \
	{register cellpo Zval; \
	while (IsVar(addr) && (Zval=(cellpo)*(addr)) != (addr)) \
		(addr) = Zval;}


/* structure of a constant symbol */

typedef struct symbol {
    union {
	struct {
	    utwoBytes lngth;	/* length of print name */
	    twoBytes hsh;	/* hash code of print name */
	} hc;
	fourBytes hshcde;	/* combination for fast test */
    } hhc;
    uchar prntname[EMPTY];	/* print name itself */
} *symbpo;

#define symblngth(addr)		(((symbpo)(addr))->hhc.hc.lngth)
#define symbhash(addr)		(((symbpo)(addr))->hhc.hc.hsh)
#define symbname(addr)		(((symbpo)(addr))->prntname)
#ifdef GNUDOS		/* Because of Byte ordering */
extern fourBytes fix_hash(fourBytes);
#define symbhashval(addr)	(fix_hash(((symbpo)(addr))->hhc.hshcde))
#define symbhashvalin(addr)	(((symbpo)(addr))->hhc.hshcde)
#else
#define symbhashval(addr)	(((symbpo)(addr))->hhc.hshcde)
#endif GNUDOS
#define string_val(c)		(symbname(symbvl(c)))
#define string_len(c)		(symblngth(symbvl(c)))
#define string_hash(c)		(symbhash(symbvl(c)))

/* header size is 4 bytes.  (2 for length and hash) */
#define symbSize(len,unit)	(((len) + 4) / (unit) + 1)
#define symMaxLen(cells,unit)	((cells) * (unit) - 5)

/* heap space allocation routines */
#define alloc_cell(addr)	{(addr) = H++;}
#define alloc_symb_struct(addr,length) \
	{register int Zclen = symbSize((length), sizeof(cell)); \
	 mkguard(H, (Zclen+1)); \
	 H++; \
	 mksymb((addr), H); \
	 H += (Zclen); \
	 mkguard(H, -(Zclen+1)); H++;}
#define alloc_symb(addr,length,Zstr) \
	{register int Zclen = symbSize((length), sizeof(cell)); \
	 mkguard(H, (Zclen+1)); \
	 H++; \
	 mksymb((addr), H); \
	 strsymb(H, (length), (Zstr)); \
	 H += (Zclen); \
	 mkguard(H, -(Zclen+1)); H++;}
#define FLOATSIZE	2

#ifdef sparc
#define alloc_float(addr,num) \
	{register FLOAT *Zptr; \
	 if (!((ufourBytes)H & 04)) { mknil(H); H++; } \
	 mkguard(H, FLOATSIZE+1); \
	 H++; \
	 mkflt((addr), H); \
	 Zptr = (FLOAT *)H; \
	 *Zptr = num; \
	 H += FLOATSIZE; \
	 mkguard(H, -(FLOATSIZE+1)); H++;}
#else
#define alloc_float(addr,num) \
	{register FLOAT *Zptr; \
	 mkguard(H, FLOATSIZE+1); \
	 H++; \
	 mkflt((addr), H); \
	 Zptr = (FLOAT *)H; \
	 *Zptr = num; \
	 H += FLOATSIZE; \
	 mkguard(H, -(FLOATSIZE+1)); H++;}
#endif

#define alloc_list(addr,start)	{mklst((addr),H); start = H; H += 2;}
#define alloc_tpl(addr,strt,sz)	{mktpl((addr),H); mkint(H,(sz)); strt = ++H; H += sz;}

/*-----------------------------------------------------------------------*/

/* Miscellaneous declarations for ICP ][ system */

typedef twoBytes	rwmode;
#define read_mode	0
#define write_mode	1

typedef twoBytes	opcode, *codepo;

/* macros for accessing the parameters in the code */
#define envsize(addr)		*(((codepo)(addr))-3)
#define envset(addr)		*((fourBytes *)(((codepo)(addr))-2))

/* macros for converting addresses */
#define rel(a)			((a) + *(a))
#define normal(a)		(fourBytes)((cellpo)(a)-TH->stacks)
#define PC(a)			(fourBytes)((a)-boot)

/* return codes used in primitives */
#define FAIL			0
#define SUCCEED			1
#define SUSPEND			2
#define REQUEUE			3
#define SUSPEND_FOR_EVENT	4
#define WAIT			5
#define AR_ERROR		6

/* alias for SUCCEED */
#define SUCCESS		1	
/*-----------------------------------------------------------------------*/

/* structure for storing statistics */

typedef struct {
	double starttime;
	double lasttime;
	double gc_time;
	int gc_count;
	int gc_acc;
} statistics;

/*-----------------------------------------------------------------------*/

/* structure of hash table entries */

typedef struct index {
    union {
	struct {
	    twoBytes size;
	    twoBytes sym;
	} functkey;
	fourBytes intkey;
    } key;
    twoBytes ioffset;
    twoBytes link;
} *indexpo;

/* macros for accessing index entries */
#define indexarity(addr)	(((indexpo)(addr))->key.functkey.size)
#define indexname(addr)		(((indexpo)(addr))->key.functkey.sym)
#define indexkey(addr)		(((indexpo)(addr))->key.intkey)
#define indexoffset(addr)	(((indexpo)(addr))->ioffset)
#define indexlink(addr)		(((indexpo)(addr))->link)

/*-----------------------------------------------------------------------*/

/* definition of call record */

typedef struct envrec {
    codepo CP;			/* parent program counter */
    struct choicerec *CSB;	/* where to cut to in parent */
    struct envrec *CE;		/* parent environment */
    cellpo HMAX;		/* pre-allocated heap space */
    cell Y[EMPTY];
} *envpo;

/*-----------------------------------------------------------------------*/

/* definition of a choice point record */
 
typedef struct choicerec {
    twoBytes AX;		/* no. or arguments saved */
    twoBytes dummy;		/* to force 4-byte alignment */
    codepo P;			/* next clause to try */
    codepo CP;			/* back up continuation pointer */
    cellpo H;			/* heap pointer */
    struct choicerec *B;	/* previous choice point */
    struct choicerec *CSB;	/* choice point to cut to */
    envpo E;			/* back up environment pointer */
    cellpo TR;			/* trail point */
    cellpo HMAX;		/* pre-allocated heap space */
    cell gcH;			/* g/c requires that H is in a cell */
} *choicepo;

/*-----------------------------------------------------------------------*/
/* definition of run queue */

typedef struct runq {
	struct runq *next, *prev;
	struct thread *th;
	int event;
} *runqpt;

/*-----------------------------------------------------------------------*/
/* definition of event */

typedef struct event_s {
	struct event_s *next;
	char *value;
	int type;
} *eventpt;


/*-----------------------------------------------------------------------*/

/*
 * definition of a thread record
 *
 * A complete thread consists of the various stacks
 * plus space for all the main registers
 */

typedef struct thread {
    struct thread *prev;	/* doubly ...      */
    struct thread *next;	/* ... linked list */
 
    struct io_desc *current_input;	/* current input */
    struct io_desc *current_output;	/* current output */

    runqpt rq;			/* pointer to position in run que */
    bool prolog;		/* prolog or parlog ? */
    twoBytes dummy;		/* to force 4-byte alignment */

    codepo P;			/* program counter */
    codepo CP;			/* continuation program counter */

    cellpo H;			/* top of constructed term stack (heap) */
    cellpo HB;			/* heap back point */
    cellpo HMAX;		/* pre-allocated heap space */
    cellpo CHMAX;		/* shadow HMAX */

    cellpo BLS;			/* base of local stacks */
    choicepo B;			/* last choice point */
    choicepo SB;		/* choice point to cut to */
    choicepo CSB;		/* continuation cut */
    choicepo GC_B;		/* last garbage collected choice point */
    choicepo CATCH;		/* choicepoint pointing to error handler */
 
    envpo E;			/* local environment */
    envpo CE;			/* continuation local environment */

    cellpo TR;			/* trail point */
    statistics stats;		/* statistics */

    fourBytes TSZ;		/* number of cells in the thread */
    cell A[REGISTERS];		/* argument registers */
    cell stacks[EMPTY];		/* the beginning of the space for the stacks */
} *threadpo;
