/* Copyright (C) 1992 Imperial College */
#include <stdio.h>
#include <setjmp.h>
#include <errno.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "macros.h"
#include "ret.h"
#include "hermes.h"

/* maximum number of variables/shared terms */
#define MAXSHARE	0x80

/* tags for transmitting encoded terms */
#define T_S_VAR   0x00
#define T_VAR     0x10
#define T_S_INT   0x20
#define T_INT     0x30
#define T_FLT     0x40
#define T_S_SYMB  0x50
#define T_SYMB    0x60
#define T_NIL     0x70
#define T_LIST    0x80
#define T_S_TPL   0x90
#define T_TPL     0xA0
#define T_S_TAG   0xB0
#define T_TAG     0xC0
#define T_S_REF   0xD0
#define T_REF     0xE0

#define TAGMASK	0xF0
#define VALMASK	0x0F

extern	jmp_buf	icp_interrupt;
extern  char	read_pipe_char();
extern  void	write_pipe_char();

/* these error numbers match the Prolog ones */
#define HEAPERR		613
#define VARERR		508

extern	char		*malloc();
extern	Word		atom_nil;
extern	Word		atom_eof;
extern	Word		funct_port;
extern	bool		h_debug;
extern	int		h_deadlock;
extern	int		wait_pipe;
extern	pipepo		curr_pipe;

/* table of variables/shared terms */
static	Word		shared[MAXSHARE];
static	int		current;


void
init_vars()
{
	current = 0;
}


static int
search_vars(addr)
Word	addr;
{
	register Word	*p;
	p=shared;
	while (++p <= shared+current)
		if (*p == addr)
			return(p - shared);

	/* not found */
	return(0);
}


static int
add_var(addr)
Word	addr;
{
	if (current++ == MAXSHARE)
		longjmp(icp_interrupt, VARERR);
	*(shared+current) = addr;
	return(current);
}


void
h_encoded_write(term, emit)
Word	term;
void	(*emit)();
{
	Word	*p, a;
	Atom	*at;
	int	ttag, n, ref;
	short	a_len;
	double	f;
	char	*c;

	deref(term);

	ttag = u_tag(term);
	switch (ttag)
	{
	case U_REF:
		n = ToRef(term);
		if (ref = search_vars(n)) {
			if (ref < 0x10)
				emit(T_S_REF | (char)ref);
			else {
				emit(T_REF);
				emit(ref);
			}
		}
		else {
			ref = add_var(n);
			if (ref < 0x10)
				emit(T_S_VAR | (char)ref);
			else {
				emit(T_VAR);
				emit(ref);
			}
		}
		break;

	case U_STRUCT:
		p = StructVal(term);
		a = AsAtom(FunctVal(*p)->f_name);
		n = FunctVal(*p)->f_arity + 1;

		if (n < 0x10)
			emit(T_S_TPL | (char)n);
		else {
			emit(T_TPL);
			emit(n >> 010);
			emit(n);
		}
		h_encoded_write(a, emit);
		while (--n)
			h_encoded_write(ToRef(++p), emit);
		break;

	case U_LIST:
		emit(T_LIST);
		p = ListVal(term);
		h_encoded_write(ToRef(p), emit);
		h_encoded_write(ToRef(++p), emit);
		break;

	case U_CONST:
		if (IsNumber(term)) {
			if (IsShort(term) || IsLong(term)) {
				n = int_val(term);
				if (n < 0x10 && n >= 0)
					emit(T_S_INT | (char)n);
				else {
					emit(T_INT);
					emit(n >> 030);
					emit(n >> 020);
					emit(n >> 010);
					emit(n);
				}
			}
			else {
				f = FloatVal(term);
				c = (char *) &f;
				emit(T_FLT);
				emit(c[0]);
				emit(c[1]);
				emit(c[2]);
				emit(c[3]);
				emit(c[4]);
				emit(c[5]);
				emit(c[6]);
				emit(c[7]);
			}
		}

		else if (term == atom_nil)
			emit(T_NIL);

		else {
			if (IsAtom(term))
				at = AtomVal(term);
			else if (IsFunct(term))
				at = FunctVal(term)->f_name;
			a_len = at->a_length;
			c = at->a_string;

			if (a_len < 0x10)
				emit(T_S_SYMB | (char)a_len);
			else {
				emit(T_SYMB);
				emit(a_len >> 010);
				emit(a_len);
			}
			/* N.B. Assumes one-byte characters ! */
			while (a_len--)
				emit(*c++);
			break;
		}

	}
}


c_write_pipe(Args)
Word	*Args;
{
	Word	port;
	iopo	io_ptr;
	pipepo	pipe_ptr;
	strpo	tmp_write_ptr;
	int	errcode;

/* changed next line to prevent write_pipe(X,Y) from hanging */
/*	wait_for_argument(A0); */
	deref(A0);	/* instead of above */

	if (!IsStruct(A0) || *StructVal(A0) != funct_port)
	{	bu_error(A0, "write_pipe/2: arg incorrect");
		return(FAIL);
	}

	port = *(StructVal(A0) + 1);
	deref(port);
	if (!IsShort(port)) 
	{	bu_error(A0, "write_pipe/2: arg incorrect");
		return(FAIL);
	}

	io_ptr = io + ShortVal(port);
	if (io_type(io_ptr) != OUT_PIPE)
	{	bu_error(A0, "write_pipe/2: arg should be an output port");
		return(FAIL);
	}
	if (!th_id(io_ptr))
		th_id(io_ptr) = TH;
	else if (th_id(io_ptr) != TH)
	{	bu_error(A0, "write_pipe/2: not owner of output port");
		return(FAIL);
	}

	pipe_ptr = piped(io_ptr);
	tmp_write_ptr = pipe_ptr->write_ptr;
	if (errcode=setjmp(icp_interrupt)) {
		switch (errcode)
		{
		case HEAPERR:
			bu_error(A0, "write_pipe/2: no heap space to resize pipe");
			return(FAIL);
			break;
		case VARERR:
			bu_error(A1, "write_pipe/2: too many variables in term");
			return(FAIL);
			break;
		}
	}

	init_vars();
	curr_pipe = pipe_ptr;
	h_encoded_write(A1, write_pipe_char);

	pipe_ptr->status &= ~P_EMPTY;
	if (pipe_ptr->status & P_RSUSP) {
		if (h_debug)
			fprintf(user_error,
			"[ wake reader on port(%d) ]\n", pipe_ptr->iport - io);
		h_deadlock = 0;
		if (th_id(pipe_ptr->iport) == parlog_th)
		{	wait_pipe--;
			assign(ToRef(&suspnd(pipe_ptr->iport)), AsUnb(0));
		}
		else
			(void) add_to_runq(th_id(pipe_ptr->iport), FALSE);
		pipe_ptr->status &= ~P_RSUSP;
	}

	return(SUCCESS);
}


h_encoded_read(term, readb)
Word	term;
char	(*readb)();
{
	Word	*s, *HP;
	int	n, arity, index;
	short	a_len;
	char	ch, *c, buf[8], str[A_STRSIZ], *longatom;
	double	*f;

	ch = (*readb)();

	if (ch == EOF) {
		if (errno)
			return(FAIL);
		return( unify(term, atom_eof) );
	}

	switch (ch & TAGMASK)
	{
	case T_S_INT:
		return( unify(term, make_int(ch & VALMASK)) );

	case T_INT:
		n = (*readb)() & 0xFF;
		n = (n << 010) | ((*readb)() & 0xFF);
		n = (n << 010) | ((*readb)() & 0xFF);
		n = (n << 010) | ((*readb)() & 0xFF);
		return( unify(term, make_int(n)) );

	case T_NIL:
		return( unify(term, atom_nil) );

	case T_S_VAR:
		index = ch & VALMASK;
		shared[index] = term;
		return(SUCCESS);

	case T_VAR:
		index = (*readb)();
		shared[index] = term;
		return(SUCCESS);

	case T_S_SYMB:
		a_len = ch & VALMASK;
		c = str;
		n = a_len;
		while (n--)
			*c++ = (*readb)();
		*c = '\0';
		return( unify(term, make_atom(str)) );

	case T_SYMB:
		a_len = (*readb)();
		a_len = (a_len << 010) | ((*readb)() & 0xFF);

		if (a_len < A_STRSIZ)
			c = str;
		else
		{	longatom = malloc(a_len + 1);
			c = longatom;
		}
		n = a_len;
		while (n--)
			*c++ = (*readb)();
		*c = '\0';

		if (a_len < A_STRSIZ)
			return( unify(term, make_atom(str)) );
		n = unify(term, make_atom(longatom) );
		free(longatom);
		return(n);

	case T_LIST:
		s = HP = m_ht->m_top;
		*HP++ = AsUnb(0);
		*HP++ = AsUnb(0);
		m_ht->m_top = HP;
		if (h_encoded_read(ToRef(s), readb) == FAIL)
			return(FAIL);
		if (h_encoded_read(ToRef(s+1), readb) == FAIL)
			return(FAIL);
		return( unify(term, AsList(s)) );

	case T_S_TPL:
		n = ch & VALMASK;
		arity = n - 1;

		s = HP = m_ht->m_top;
		m_ht->m_top += n;
		while (n--)
		{	*HP = AsUnb(0);
			if (h_encoded_read(ToRef(HP++), readb) == FAIL)
				return(FAIL);
		}

		if (!IsAtom(*s))
		{	commit_read(curr_pipe);
			bu_error(*s, "look_pipe/2: invalid functor");
			return(FAIL);
		}
		*s = AsFunct( findfunct(AtomVal(*s), arity) );
		return( unify(term, AsStruct(s)) );

	case T_TPL:
		n = (*readb)();
		n = (n << 010) | ((*readb)() & 0xFF);
		arity = n - 1;

		s = HP = m_ht->m_top;
		m_ht->m_top += n;
		while (n--)
		{	*HP = AsUnb(0);
			if (h_encoded_read(ToRef(HP++), readb) == FAIL)
				return(FAIL);
		}

		if (!IsAtom(*s))
		{	commit_read(curr_pipe);
			bu_error(*s, "look_pipe/2: invalid functor");
			return(FAIL);
		}
		*s = AsFunct( findfunct(AtomVal(*s), arity) );
		return( unify(term, AsStruct(s)) );

	case T_S_TAG:
		index = ch & VALMASK;
		n = h_encoded_read(term, readb);
		shared[index] = term;
		return(n);

	case T_TAG:
		index = (*readb)();
		n = h_encoded_read(term, readb);
		shared[index] = term;
		return(n);

	case T_S_REF:
		return( unify(term, shared[ch & VALMASK]) );

	case T_REF:
		return( unify(term, shared[(*readb)()]) );

	case T_FLT:
		f = (double *)buf;
		for (n=0; n<8; n++)
			buf[n] = (*readb)();
		n = *f;
		if (n == *f && (abs(n) & int_mask))
			return( unify(term, make_int(n)) );
		else
			return( unify(term, make_float(*f)) );
	}
}


commit_read(pipe_ptr)
pipepo	pipe_ptr;
{
	pipe_ptr->status &= ~P_LOCK;
	pipe_ptr->read_ptr = pipe_ptr->look_ptr;
	if (pipe_ptr->read_ptr == pipe_ptr->write_ptr)
		pipe_ptr->status |= P_EMPTY;
}


c_commit_read(Args)
Word	*Args;
{
	Word	port;
	iopo	io_ptr;
	pipepo	pipe_ptr;

	wait_for_argument(A0);
	if (!IsStruct(A0) || *StructVal(A0) != funct_port)
	{	bu_error(A0, "commit_read/1: arg incorrect");
		return(FAIL);
	}

	port = *(StructVal(A0) + 1);
	deref(port);
	if (!IsShort(port)) 
	{	bu_error(A0, "commit_read/1: arg incorrect");
		return(FAIL);
	}

	io_ptr = io + ShortVal(port);
	if (io_type(io_ptr) != IN_PIPE)
	{	bu_error(A0, "commit_read/1: arg should be an input port");
		return(FAIL);
	}
	if (th_id(io_ptr) != TH)
	{	bu_error(A0, "commit_read/1: not owner of input port");
		return(FAIL);
	}

	pipe_ptr = piped(io_ptr);

	if (pipe_ptr->status & P_WSUSP) {
		if (h_debug)
			fprintf(user_error,
			"[ wake writer on port(%d) ]\n", pipe_ptr->oport - io);
		h_deadlock = 0;
		if (th_id(pipe_ptr->oport) == parlog_th)
		{	wait_pipe--;
			assign(ToRef(&suspnd(pipe_ptr->oport)), AsUnb(0));
		}
		else
			(void) add_to_runq(th_id(pipe_ptr->oport), FALSE);
		pipe_ptr->status &= ~P_WSUSP;
	}

	/* can always commit empty reads */
	if (pipe_ptr->status & P_EMPTY)
		return(SUCCESS);

	if (pipe_ptr->status & P_LOCK &&
	   (pipe_ptr->look_ptr > pipe_ptr->read_ptr ||
	   (pipe_ptr->read_ptr > pipe_ptr->write_ptr &&
	    pipe_ptr->write_ptr >= pipe_ptr->look_ptr))) {
		commit_read(pipe_ptr);
		return(SUCCESS);
	}
	else return(FAIL);
}


c_unlock(Args)
Word	*Args;
{
	Word	port;
	iopo	io_ptr;

	wait_for_argument(A0);
	if (!IsStruct(A0) || *StructVal(A0) != funct_port)
	{	bu_error(A0, "unlock/1: arg incorrect");
		return(FAIL);
	}

	port = *(StructVal(A0) + 1);
	deref(port);
	if (!IsShort(port)) 
	{	bu_error(A0, "unlock/1: arg incorrect");
		return(FAIL);
	}

	io_ptr = io + ShortVal(port);

	if (!is_pipe(io_ptr))
	{	bu_error(A0, "unlock/1: arg not a port");
		return(FAIL);
	}

	if (th_id(io_ptr) != TH)
	{	bu_error(A0, "unlock/1: not owner of port");
		return(FAIL);
	}

	piped(io_ptr)->status &= ~P_LOCK;
	return(SUCCESS);
}


c_look_pipe(Args)
Word	*Args;
{
	Word	port;
	iopo	io_ptr;
	pipepo	pipe_ptr;
	int	ret;

/* changed next line to prevent look_pipe(X,Y) from hanging */
/*	wait_for_argument(A0); */
	deref(A0);	/* instead of above */

	if (!IsStruct(A0) || *StructVal(A0) != funct_port)
	{	bu_error(A0, "look_pipe/2: arg incorrect");
		return(FAIL);
	}

	port = *(StructVal(A0) + 1);
	deref(port);
	if (!IsShort(port)) 
	{	bu_error(A0, "look_pipe/2: arg incorrect");
		return(FAIL);
	}

	io_ptr = io + ShortVal(port);
	if (io_type(io_ptr) != IN_PIPE)
	{	bu_error(A0, "look_pipe/2: arg should be an input port");
		return(FAIL);
	}
	if (!th_id(io_ptr))
		th_id(io_ptr) = TH;
	else if (th_id(io_ptr) != TH)
	{	bu_error(A0, "look_pipe/2: not owner of input port");
		return(FAIL);
	}

	pipe_ptr = piped(io_ptr);
	if (pipe_ptr->status & P_LOCK) {
		pipe_ptr->status |= P_RSUSP;
		if (h_debug)
			fprintf(user_error,
			"[ suspend read on port(%d) ]\n", io_ptr - io);
		wait_pipe++;
		VarTbl[Nsuspv++] = ToRef(&suspnd(io_ptr));
		return(SUSPEND);
/*		return(REQUEUE); */
	}

	if (pipe_ptr->status & P_EMPTY) {
		if (!pipe_ptr->oport) {
			if (pipe_ptr->start) {
				free((char *)pipe_ptr->start);
				pipe_ptr->start = NULL;
			}
			return(unify(A1, atom_eof));
		}
		else {
			pipe_ptr->status |= P_RSUSP;
			if (h_debug)
				fprintf(user_error,
				"[ suspend read on port(%d) ]\n", io_ptr - io);
			wait_pipe++;
			VarTbl[Nsuspv++] = ToRef(&suspnd(io_ptr));
			return(SUSPEND);
/*			return(REQUEUE); */
		}
	}

	/* ... otherwise lock it and read */
	pipe_ptr->status |= P_LOCK;
	pipe_ptr->look_ptr = pipe_ptr->read_ptr;
	curr_pipe = pipe_ptr;
	return(h_encoded_read(A1, read_pipe_char));
}


c_read_pipe(Args)
Word	*Args;
{
	int		status;

	status = c_look_pipe(Args);
	if (status == SUCCESS)
		return(c_commit_read(Args));
	return(status);
}


c_empty_pipe(Args)
Word	*Args;
{
	Word	port;
	iopo	io_ptr;
	pipepo	pipe_ptr;

	wait_for_argument(A0);
	if (!IsStruct(A0) || *StructVal(A0) != funct_port)
	{	bu_error(A0, "empty_pipe/1: arg incorrect");
		return(FAIL);
	}

	port = *(StructVal(A0) + 1);
	deref(port);
	if (!IsShort(port)) 
	{	bu_error(A0, "empty_pipe/1: arg incorrect");
		return(FAIL);
	}

	io_ptr = io + ShortVal(port);

	if (!is_pipe(io_ptr))
	{	bu_error(A0, "empty_pipe/1: arg not a port");
		return(FAIL);
	}

	pipe_ptr = piped(io_ptr);
	return ((pipe_ptr->status & P_EMPTY) == P_EMPTY && pipe_ptr->oport);
}
