/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include "objs.h"
#include "instr.h"
#include "funcs.h"
#include "stream.h"

#ifdef HERMES
#include <fcntl.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <vfork.h>
extern	FILE	*user_error;	/* not always stderr ! */
extern	int	tty_pid;	/* Unix process id of jam tty */
extern	int	signal_mask;
extern	int	x_parlog;
#endif
Word	(*bu_table[60])();		/*  builtin predicate table  */

/*  names used in builtins  */
Word	atom_nil;
Word	atom_list;
Word	atom_eq;
Word	atom_lt;
Word	atom_gt;
Word	funct_list;
#ifdef HERMES
Word	funct_port;
#endif

/*  system support  */
Word	atom_system;

/*  I/O  */
Word	atom_user;
Word	atom_user_input;
Word	atom_user_output;
Word	atom_user_error;
Word	atom_read;
Word	atom_write;
Word	atom_append;
Word	atom_eof;
Word	funct_stream;

Stream	*st_user_input;
Stream	*st_user_output;
Stream	*st_user_error;
Stream	*st_curr_input;
Stream	*st_curr_output;
#ifdef HERMES
Stream	*st_stdout;
#endif

/*  load cmd  */
Word	funct_cmd;
Proc	*proc_cmd;
Proc	*proc_load;

/*  debugging etc  */
Word	funct_call1;
Proc	*proc_trace;
Proc	*proc_undefined;
Proc	*proc_interrupt;
Proc	*proc_deadlock;


/*
 *  INITIALISE  --  set up builtin tables, names, predicates, etc.
 */
initialise()
{
	/*
	 *  initialise builtin atoms
	 */
	atom_nil = make_atom("[]");
	atom_list = make_atom(".");
	atom_eq = make_atom("=");
	atom_lt = make_atom("<");
	atom_gt = make_atom(">");

		/*  system support  */
	atom_system = make_atom("system");

		/*  I/O  */
	atom_user = make_atom("user");
	atom_user_input = make_atom("user_input");
	atom_user_output = make_atom("user_output");
	atom_user_error = make_atom("user_error");
	atom_read = make_atom("read");
	atom_write = make_atom("write");
	atom_append = make_atom("append");
	atom_eof = make_atom("end_of_file");

		/*  quotes for null atom  */
	findatom("")->a_props = A_QUOTE;

	/*
	 *  initialise builtin functors
	 */
	funct_list = AsFunct( findfunct(AtomVal(atom_list), 2) );
	funct_stream = AsFunct( findfunct(findatom("$stream"), 2) );
	funct_cmd = AsFunct( findfunct(findatom("<command>"), 0) );
	funct_call1 = AsFunct( findfunct(findatom("$call"), 1) );
#ifdef HERMES
	funct_port = AsFunct( findfunct(findatom("port"), 1) );
#endif

	/*  initialise current module ("system")  */
	*a_curr_module = AtomVal(atom_system);

	/*
	 *  initialise builtin predicates
	 */
	define_builtin("var",1,builtin_o,bu_var,0);
	define_builtin("nonvar",1,builtin_o,bu_nonvar,1);

	define_builtin("atom",1,builtin_i,bu_atom,2);
	define_builtin("integer",1,builtin_i,bu_integer,3);
	define_builtin("float",1,builtin_i,bu_float,4);
	define_builtin("number",1,builtin_i,bu_number,5);
	define_builtin("atomic",1,builtin_i,bu_atomic,6);

	define_builtin("=@=",2,builtin_ii,bu_lex_eq,10);
	define_builtin("\\=@=",2,builtin_ii,bu_lex_ne,11);
	define_builtin("@<",2,builtin_ii,bu_lex_lt,12);
	define_builtin("@>",2,builtin_ii,bu_lex_gt,13);
	define_builtin("@=<",2,builtin_ii,bu_lex_le,14);
	define_builtin("@>=",2,builtin_ii,bu_lex_ge,15);
	define_builtin("$compare",3,builtin_iio,bu_compare,16);

	define_builtin("$term_to_list",2,builtin_io,bu_term_to_list,51);
	define_builtin("$list_to_term",2,builtin_io,bu_list_to_term,52);
	define_builtin("$st_to_list",2,builtin_io,bu_st_to_list,53);
	define_builtin("$list_to_st",2,builtin_io,bu_list_to_st,54);
	define_builtin("$term_to_na",3,builtin_ioo,bu_term_to_na,55);
	define_builtin("$na_to_term",3,builtin_iio,bu_na_to_term,56);
	define_builtin("arg",3,builtin_iio,bu_arg,57);
	define_builtin("$list_to_atom",2,builtin_io,bu_list_to_atom,58);
	define_builtin("clock",1,builtin_o,bu_clock,59);
	/* need to alter size of bu_table[] if more builtins are defined */

	define_builtin("=:=",2,builtin_nn,bu_arith_eq,20);
	define_builtin("=\\=",2,builtin_nn,bu_arith_ne,21);
	define_builtin("<",2,builtin_nn,bu_arith_lt,22);
	define_builtin(">",2,builtin_nn,bu_arith_gt,23);
	define_builtin("=<",2,builtin_nn,bu_arith_le,24);
	define_builtin(">=",2,builtin_nn,bu_arith_ge,25);

	define_function("+",1,fu_plus,30);		/* function_n */
	define_function("-",1,fu_minus,31);		/* function_n */
	define_function("\\",1,fu_not,32);		/* function_n */
	define_function("integer",1,fu_integer,33);	/* function_n */
	define_function("float",1,fu_float,34);		/* function_n */

	define_function("+",2,fu_add,35);		/* function_nn */
	define_function("-",2,fu_subtract,36);		/* function_nn */
	define_function("*",2,fu_multiply,37);		/* function_nn */
	define_function("/",2,fu_fdivide,38);		/* function_nn */
	define_function("//",2,fu_idivide,39);		/* function_nn */
	define_function("mod",2,fu_modulus,40);		/* function_nn */

	define_function("/\\",2,fu_and,41);		/* function_nn */
	define_function("\\/",2,fu_or,42);		/* function_nn */
	define_function("<<",2,fu_lshift,43);		/* function_nn */
	define_function(">>",2,fu_rshift,44);		/* function_nn */

	define_c_predicate("call",1,FNULL,enter_interpret,1);
	define_c_predicate("$call",1,FNULL,enter_interpret,0);
	define_c_predicate("module_call",2,FNULL,enter_interpret,1);
	define_c_predicate("halt",0,FNULL,halt,1);

	define_c_predicate("module",1,c_module,enter_c,1);
	define_c_predicate("current_module",1,c_curr_module,enter_c,1);
	define_c_predicate("$public",1,c_public,enter_io_c,0);
	define_c_predicate("$export",1,c_export,enter_io_c,0);
	define_c_predicate("$import",2,c_import,enter_io_c,0);
	define_c_predicate("$load",1,c_load,enter_io_c,0);
	define_c_predicate("$load1",1,c_load1,enter_io_c,0);

	define_c_predicate("open",3,io_open,enter_io_c,1);
	define_c_predicate("close",1,io_close,enter_io_c,1);
	define_c_predicate("set_input",1,io_set_input,enter_io_c,1);
	define_c_predicate("set_output",1,io_set_output,enter_io_c,1);
	define_c_predicate("current_input",1,io_curr_input,enter_io_c,1);
	define_c_predicate("current_output",1,io_curr_output,enter_io_c,1);
	define_c_predicate("clear_input",1,io_flush,enter_io_c,1);
	define_c_predicate("flush_output",1,io_flush,enter_io_c,1);

	define_c_predicate("get0",1,io_get0,enter_io_c,1);
	define_c_predicate("get0",2,io_st_get0,enter_io_c,1);
	define_c_predicate("get",1,io_get,enter_io_c,1);
	define_c_predicate("get",2,io_st_get,enter_io_c,1);
	define_c_predicate("skip",1,io_skip,enter_io_c,1);
	define_c_predicate("skip",2,io_st_skip,enter_io_c,1);
	define_c_predicate("gets",1,io_gets,enter_io_c,1);
	define_c_predicate("gets",2,io_st_gets,enter_io_c,1);

	define_c_predicate("put",1,io_put,enter_io_c,1);
	define_c_predicate("put",2,io_st_put,enter_io_c,1);
	define_c_predicate("nl",0,io_nl,enter_io_c,1);
	define_c_predicate("nl",1,io_st_nl,enter_io_c,1);
	define_c_predicate("tab",1,io_tab,enter_io_c,1);
	define_c_predicate("tab",2,io_st_tab,enter_io_c,1);
	define_c_predicate("$display",1,io_display,enter_io_c,0);
	define_c_predicate("$display",2,io_st_display,enter_io_c,0);
	define_c_predicate("$write",1,io_write,enter_io_c,0);
	define_c_predicate("$write",2,io_st_write,enter_io_c,0);

	define_c_predicate("$unix",2,c_unix,enter_io_c,0);
	define_c_predicate("$load_foreign",4,c_load_foreign,enter_io_c,0);

	define_c_predicate("$set_op_prec",5,c_set_op_prec,enter_io_c,0);
	define_c_predicate("$get_op_prec",5,c_get_op_prec,enter_c,0);
	define_c_predicate("ground",1,c_ground,enter_c,1);
	define_c_predicate("copy_term",2,c_copy_term,enter_c,1);
	define_c_predicate("create_channel",2,c_create_channel,enter_c,1);
	define_c_predicate("write_channel",2,c_write_channel,enter_c,1);
	define_c_predicate("get_channel",2,c_get_channel,enter_c,1);
	define_c_predicate("close_channel",1,c_close_channel,enter_c,1);

	define_c_predicate("$synch",0,c_synch,enter_c,0);
	define_c_predicate("gc",0,c_gc,enter_c,1);

	define_c_predicate("$spy",1,c_spy,enter_c,0);
	define_c_predicate("$nospy",1,c_nospy,enter_c,0);

	define_c_predicate("$runtime",1,c_runtime,enter_c,0);
	define_c_predicate("$counts",3,c_counts,enter_c,0);
	define_c_predicate("$mem_usage",5,c_mem_usage,enter_c,0);

	define_c_predicate("$getenv",2,c_getenv,enter_io_c,0);
	define_c_predicate("setenv",2,c_setenv,enter_io_c,1);
	define_c_predicate("$concat_atom",2,c_concat_atom,enter_c,0);
	define_c_predicate("stat",7,c_stat,enter_io_c,1);
	define_c_predicate("$defined",1,c_defined,enter_c,1);
	define_c_predicate("varoccurs",2,c_varoccurs,enter_c,1);

#ifdef HERMES
	define_c_predicate("resume",1,c_resume,enter_c,1);
	define_c_predicate("new_thread",1,c_new_thread,enter_c,1);
	define_c_predicate("fork",2,c_fork,enter_c,1);
	define_c_predicate("pipe",2,c_pipe,enter_c,1);
	define_c_predicate("read_pipe",2,c_read_pipe,enter_io_c,1);
	define_c_predicate("write_pipe",2,c_write_pipe,enter_io_c,1);
	define_c_predicate("look_pipe",2,c_look_pipe,enter_io_c,1);
	define_c_predicate("commit_read",1,c_commit_read,enter_c,1);
	define_c_predicate("unlock",1,c_unlock,enter_c,1);
	define_c_predicate("close_port",1,c_close_port,enter_io_c,1);
	define_c_predicate("release_port",1,c_release_port,enter_io_c,1);
	define_c_predicate("debug",1,c_debug,enter_c,1);
	define_c_predicate("empty_pipe",1,c_empty_pipe,enter_c,1);
	define_c_predicate("is_iport",1,c_is_iport,enter_c,1);
	define_c_predicate("is_oport",1,c_is_oport,enter_c,1);
	define_c_predicate("prolog",0,c_prolog,enter_c,1);
	define_c_predicate("timeslice",1,c_timeslice,enter_c,1);
	define_c_predicate("set_port",2,c_set_port,enter_c,1);
	define_c_predicate("get_port",2,c_get_port,enter_c,1);
#endif
	/*
	 *  initialise other "known" predicate names
	 */
	proc_trace = findproc( findfunct(findatom("$debug_goal"),1), ANULL);
	proc_undefined = findproc(findfunct(findatom("$undefined_goal"),1), ANULL);
	proc_interrupt = findproc( findfunct(findatom("$interrupt"),1), ANULL);
	proc_deadlock = findproc( findfunct(findatom("$deadlock"),0), ANULL);

	proc_cmd = findproc( FunctVal(funct_cmd), ANULL);
	proc_load = findproc( findfunct(findatom("$load1"),1), ANULL);

	/*  $undefined_goal should fail unless its (re)defined  */
	*(proc_undefined->p_code) = fail;

#ifdef PARLOGTCP
	tcp_init();
#endif

	initialise_streams();
}


/*
 *  DEFINE PREDICATE functions
 */

define_builtin(name,arity,type,proc,index)
char	*name;
int	arity;
int	type;
int	(*proc)();
int	index;
{
	Atom	*c = findatom(name);
	Funct	*f = findfunct(c,arity);
	Proc	*p = findproc(f,ANULL);
	register Code	*s;

	s = alloc_code(8+arity);
	*s++ = enter_trace;
	*WordP(s) = AsFunct(f);
	s += WordArgSize;
	p->p_code = s;
	p->p_flags = P_READONLY|P_PUBLIC;

	*s++ = (Code) type;
	*s++ = 0;
	if (arity > 1) *s++ = 1;
	if (arity > 2) *s++ = 2;
	*WordP(s) = (Word) proc;
	s += WordArgSize;
	*s++ = commit;
	*s   = proceed;

	bu_table[index] = (Word (*) ()) proc;
}

define_c_predicate(name,arity,proc,enter,public)
char	*name;
int	arity;
int	(*proc)();
int	enter;
int	public;
{
	Atom	*c = findatom(name);
	Funct	*f = findfunct(c,arity);
	Proc	*p = findproc(f,ANULL);
	register Code	*s;

	s = alloc_code(8);
	*s++ = enter_trace;
	*WordP(s) = AsFunct(f);
	s += WordArgSize;
	p->p_code = s;
	p->p_flags = public ? P_READONLY|P_PUBLIC : P_READONLY;

	*s++ = (Code) enter;
	*WordP(s) = (Word) proc;
	s += WordArgSize;
	*s++ = commit;
	*s   = proceed;
}

define_function(name,arity,proc,index)
char	*name;
int	arity;
Word	(*proc)();
int	index;
{
	Atom	*c = findatom(name);
	(void) findfunct(c,arity);

	bu_table[index] = proc;
}


/*
 *  INITIALISE_STREAMS  --  setup builtin I/O stream structures
 */
initialise_streams()
{
#ifdef HERMES
	if (x_parlog) {
		FILE *fd;
		int tty = parlog_tty();

		st_user_input  = make_stream(atom_user_input, atom_read, fdopen(tty, "r"));

		fd = fdopen(tty, "a");
		st_user_output = make_stream(atom_user_output, atom_append, fd);
		st_user_error  = make_stream(atom_user_error, atom_append, fd);
		setbuf(fd, NULL);
		user_error = fd;
	}
	else {
		st_user_input  = make_stream(atom_user_input, atom_read, stdin);
		st_user_output = make_stream(atom_user_output, atom_append, stdout);
		st_user_error  = make_stream(atom_user_error, atom_append, stderr);
		user_error = stderr;
	}
	/* used for tracing */
	st_stdout  = make_stream(atom_user_output, atom_append, stdout);
#else
	st_user_input  = make_stream(atom_user_input, atom_read, stdin);
	st_user_output = make_stream(atom_user_output, atom_append, stdout);
	st_user_error  = make_stream(atom_user_error, atom_append, stderr);
#endif

	st_curr_input = st_user_input;
	st_curr_output = st_user_output;
}
#ifdef HERMES
/*
 *  Open a tty for use by Parlog
 */
#define MASTER	"/dev/ptyXY"
#define SLAVE	"/dev/ttyXY"

parlog_tty()
{
    char	*master_path,
		*slave_path,
		buf[100], host[30], title[40];
    int		i, master, slave, err, old_mask;
    long	params;
    extern char	*malloc();

    /* switch off interrupts */
    old_mask = sigblock(signal_mask);

    master_path = malloc(strlen(MASTER)+1);
    strcpy(master_path, MASTER);

    slave_path = malloc(strlen(SLAVE)+1);
    strcpy(slave_path, SLAVE);

    for(i=0; i <= ('s'-'p') * 16; ++i) {
	master_path[strlen(MASTER)-2] = i / 16 + 'p';
	master_path[strlen(MASTER)-1] = "0123456789abcdef"[i & 15];
	master = open(master_path, O_RDWR);
	if(master >= 0)
	    break;
    }

    if(master < 0) {
	fprintf(stderr, "cannot get free pty for Parlog\n");
	icp_exit(1);
    }

    slave_path[strlen(SLAVE)-2] = master_path[strlen(MASTER)-2];
    slave_path[strlen(SLAVE)-1] = master_path[strlen(MASTER)-1];
    slave = open(slave_path, O_RDWR, 0);

    err = ioctl(master, TIOCLGET, &params);
    if (err) {
	fprintf(stderr, "ioctl error %d in parlog_tty() TIOCLGET\n", err);
	icp_exit(1);
    }
    params |= LCRTERA;
    err = ioctl(master, TIOCLSET, &params);
    if (err) {
	fprintf(stderr, "ioctl error %d in parlog_tty() TIOCLSET\n", err);
	icp_exit(1);
    }

    (void) gethostname(host, 30);
    (void) strcpy(title, "Parlog @ ");
    (void) strcat(title, host);

    sprintf(buf, "-S%s%d", &slave_path[strlen(slave_path)-2], master);

    free(master_path);
    free(slave_path);

    if ((tty_pid=vfork()) == 0) {
	close(slave);
        execlp("xterm", "xterm", "-name", title, buf, 0);
        fprintf(stderr, "xterm exec failed\n");
        _exit(1);
    }

    /* restore interrupts */
    sigsetmask(old_mask);

    close(master);
    (void)read(slave, buf, sizeof(buf)); /* suck xterm junk before echo on */
    return(slave);
}
#endif
