/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "macros.h"
#include "event.h"
#include "ret.h"
#include "timer.h"

/*
 *  this is a parallel garbage collection routine
 *  based on semi-space copying algorithm.
 */
#define	FBIT	0x04000000			/*  forwarding address bit  */

#define	forwarded(x)	((x) & FBIT)		/*  cell has been forwarded */
#define forward(x)	((Word)(x) | FBIT)	/*  mark cell as forwarded  */
#define unforward(x)	((x) &= ~FBIT)		/*  clear forwarded mark    */

#define segmentsize	(sizeof(Mem) / sizeof(Word))

int	h_gen, oldh_gen, s_gen;			/*  g.c. "generation" no.s  */
int	local_counter;				/*  local gc counter	*/
int	segcnt;					/*  segment count	*/

extern	int	*gc_counter;
extern	int	*st_ngcs;
extern	int	Ncalls;
static	int	prev_ncalls;


/*
 *  C_GC  --  builtin predicate to explictly invoke garbage collector
 *		simply sets the event flag to E_GC;
 */
c_gc()
{
	*event |= E_SGC|E_GC;
	return(SUCCESS);
}


/*
 *  HEAP_GC  --  garbage collect heap
 *		scan the stack, copying local heap cells and creating
 *		entries in IPS for remote references. Then scan heap & IPS
 *		copying further cells in old local heap, until all is done.
 */
heap_gc()
{
	Mem	*oldm_hb;

	/*  (initialisation + synchronisation is done in execute_event)  */

#ifdef GCTRACE
	printf("%d#ready for gc, heap size = %d cells, ips size = %d cells\n",
		PR-Pr0, get_used_mem(m_ht), PR->gc_end - PR->gc_top);
#endif
#ifdef TIMER
	tim0 = get_clock();
#endif

	oldm_hb = m_hb;
	oldh_gen = h_gen;
	local_counter = sizeof(Mem);

	m_hb = m_ht = alloc_new_heap_segment();
	segcnt = 1;

	/*  garbage collect  */
	scan_stack();
	scan_heap();

	/*
	 *  add old heap into free list, then request segcnt+4/NPR segments
	 *  to add to new heap.
	 */
	free_segments(oldm_hb);

	(void) request_heap_segments(segcnt + 4/NPR);

	prev_ncalls = Ncalls;
	if (ismaster(PR))
		(*st_ngcs)++;

#ifdef TIMER
	tim1 = get_clock();
#endif

#ifdef GCTRACE
	printf("%d# heap size = %d cells\n", PR-Pr0, get_used_mem(m_ht));
#endif GCTRACE
#ifdef TIMER
	printf("%d# gc done in %9.6f secs\n",
		PR-Pr0, (double)(tim1 - tim0) / 1000000);
#endif
}

scan_stack()
{
	register Word	*p, *vp, *st, *ht;
	Processor *hpr;
	Mem	*ms;
	Word	*he;
	int	tag1, n;
	char	*lck;

	/* 
	 *  scan stack for heap pointers and copy referenced cells to new heap
	 */
	ht = m_ht->m_data;
	he = m_ht->m_ovflw;

	for (ms = m_sb; ms; ms = ms->m_next)
	{	st = ms->m_top;

		for (p = ms->m_data; p < st; p++)
		{	if (*p == 0 || IsConst(*p))
				continue;

			vp = PtrVal(*p);
			if (!HeapPtr(vp))
				continue;

			tag1 = u_tag(*p);

			if (forwarded(*vp))
			{	/*  cell already copied  */

				switch (tag1)
				{
				case U_REF:
					*p = AsRef( value(*vp) );
					continue;

				case U_LIST:
					if (*(vp+1) != *vp + sizeof(Word))
						break;

					*p = AsList( value(*vp) );
					continue;

				case U_STRUCT:
					*p = AsStruct( value(*vp) );
					continue;
				}
			}

			hpr = HeapToPR(vp);
			if (hpr == PR)
			{	/*  local cell  */

				switch (tag1)
				{
				case U_REF:
					*p = ToRef(ht);
					*ht = *vp;
					*vp = forward(ht++);
					break;

				case U_LIST:
					*p = AsList(ht);
					*ht = *vp;
					*vp++ = forward(ht++);
					*ht = *vp;
					*vp++ = forward(ht++);
					break;

				case U_STRUCT:
					*p = AsStruct(ht);
					n = FunctVal(*vp)->f_arity;
					while (n-- >= 0)
					{	*ht = *vp;
						*vp++ = forward(ht++);
					}
					break;
				}

				if (ht > he)
				{	m_ht->m_top = ht;
					m_ht = alloc_heap_segment();
					ht = m_ht->m_data;
					he = m_ht->m_ovflw;
					segcnt++;
				}
			}
			else
			{	/*  remote cell  */
				local_counter--;

				/*  wait if remote ips is full  */
				while (hpr->gc_top > hpr->gc_end)
				{	m_ht->m_top = ht;
					check_my_ips();
					ht = m_ht->m_top;
					he = m_ht->m_ovflw;
				}

				/*
				 *  place a ref to the stack cell in the remote
				 *  IPS. Remote processor will then update this.
				 */
				lck = PRtolck(hpr);
				lock(lck);
				*hpr->gc_top = ToRef(p);
				hpr->gc_top++;
				unlock(lck);
			}
		}
	}

	m_ht->m_top = ht;
}

scan_heap()
{
	register Word	*p, *ht;
	register Mem	*mh;

	/*
	 *  first scan of the new heap.  Locate pointers to terms in old
	 *  heap space and copy them to new heap space.
	 */
	for (mh = m_hb; mh; mh = mh->m_next)
	{	p = mh->m_data;
		ht = mh->m_top;

		while (p < ht)
		{	scan_and_copy(p, ht);
			p = ht;
			ht = mh->m_top;
		}
	}
}


scan_and_copy(from, to)
Word	*from, *to;
{
	register Word	*p, *vp, *ht;
	Processor *hpr;
	Word	*he;
	int	n, tag1;
	char	*lck;
	Word	*prev;

	/*
	 *  scan new heap (or stack) between  from & to  for pointers
	 *  to old heap which must be copied onto new heap.
	 */
	ht = m_ht->m_top;
	he = m_ht->m_ovflw;

	for (p = from; p < to; p++)
	{	if (IsConst(*p))
			continue;

		/*  convert copied forwarding address to a ref  */
		if (forwarded(*p))
		{	unforward(*p);
			continue;
		}

		vp = PtrVal(*p);
		if (vp == WNULL)
			continue;

		/* beginning of Waterloo changes */
		/* by Tony Savor */
		/* handle special case where vp is a process pointer */
		if (ProcessPtr(vp)) {
		    while (vp != WNULL && ProcessPtr(vp)) {
			prev = vp;
			vp = PtrVal(((Process *) vp)->link);
		    }
		    if (vp == WNULL) {
			continue;
		    }
		    ((Process *) prev)->link = (Process *) ht;
		    *ht++ = *vp++;
		    *ht++ = *vp++;
		    continue;	

		} else if (!OldHeapPtr(vp)) {
		        continue;
		}

		/* D. Renaux: modifications to Tony's procedure:
		    - forward the complete hybrid list instead of just one entry,
		      to avoid repeated loops on scan_and_copy
		    - ignore entries with Zero (somebody else unsuspended this process)
		    - use AsUnb(ht) to create the link in the list
                    - check for end of heap segment (ht>he) after forwarding
                 */
		if (ProcessPtr(vp)) {
		    while (vp) {
			if (ProcessPtr(vp)) {
			    prev = vp;
			    vp = PtrVal(((Process *) vp)->link);
			}
			else {
			    /* printf("Found a hybrid list. Forwarding all ...\n"); */
			    if (*(ToPtr(vp[1])) == Zero) {
				vp = PtrVal(*vp);
			    }
			    else {
				if (ProcessPtr(prev))
				    ((Process *) prev)->link = (Process *) AsUnb(ht);
				else
				    (prev[0]) = AsUnb(ht);
				prev = ht;
				*ht++ = *vp++;
				*ht++ = *vp++;	/* D. Renaux: must chk for heap
						 * (ht>he) */
				if (ht > he) {
				    /* printf("added heap segment\n"); */
				    m_ht->m_top = ht;
				    m_ht = alloc_heap_segment();
				    ht = m_ht->m_data;
				    he = m_ht->m_ovflw;
				    segcnt++;
				}
				vp = PtrVal(*prev);
			    }
			}
		    }
		    if (ProcessPtr(prev))
			((Process *) prev)->link = (Process *) AsUnb(0);
		    else
			(prev[0]) = AsUnb(0);
		    continue;
		}
		/* end of Waterloo changes */

		if (!OldHeapPtr(vp))
		    continue;

		/*  p points to cell(s) on old heap starting at vp  */
		tag1 = u_tag(*p);

		if (forwarded(*vp))
		{	/*  cell already copied  */

			switch (tag1)
			{
			case U_REF:
				*p = AsRef( value(*vp) );
				continue;

			case U_LIST:
				if (*(vp+1) != *vp + sizeof(Word))
					break;

				*p = AsList( value(*vp) );
				continue;

			case U_STRUCT:
				*p = AsStruct( value(*vp) );
				continue;

			case U_CONST:
				*p = unforward(*vp);
				continue;
			}
		}

		hpr = HeapToPR(vp);
		if (hpr == PR)
		{	/*  local heap  */

			switch (tag1)
			{
			case U_REF:
				if (IsRef(*p))
				{	/*  reference pointer  */
					*p = ToRef(ht);
					*ht = *vp;	/*  copy reference  */
					*vp = forward(ht++);
				}
				else
				{	/*
					 *  var points to suspension note; n.b.
					 *  cant be forwarded - only 1 reference
					 */

					/* beginning of Waterloo changes */
					/* D. Renaux: bug fixing of hanger */
					/*
					 * find first non-Zero susp. node or process ptr
					 */

					while (!ProcessPtr(vp) && UnbVal(*vp) && *(ToPtr(vp[1])) == Zero)
						vp = ToPtr(UnbVal(*vp));
					if (ProcessPtr(vp))
						*p = AsUnb(vp);
					else {
					/* end of Waterloo changes */

						*p = AsUnb(ht);
						*ht++ = *vp++;
						*ht++ = *vp++;
						break;
					}
				}
				break;

			case U_LIST:
				*p = AsList(ht);
				*ht = *vp;		/* copy head */
				*vp++ = forward(ht++);
				*ht = *vp;		/* copy tail */
				*vp++ = forward(ht++);
				break;

			case U_STRUCT:
				*p = AsStruct(ht);
				n = FunctVal(*vp)->f_arity;
				while (n-- >= 0)	/* copy term */
				{	*ht = *vp;
					*vp++ = forward(ht++);
				}
				break;

			case U_CONST:
				*p = *vp;
				break;
			}

			if (ht > he)
			{	m_ht->m_top = ht;
				m_ht = alloc_heap_segment();
				ht = m_ht->m_data;
				he = m_ht->m_ovflw;
				segcnt++;
			}
		}
		else
		{	/*  remote heap  */

			if (local_counter > 1)
				local_counter--;
			else
			{	lck = ptrtolck(gc_counter);
				lock(lck);
				(*gc_counter) += 8;
				unlock(lck);
				local_counter += 7;
			}

			/*  wait if remote ips is full  */
			while (hpr->gc_top > hpr->gc_end)
			{	m_ht->m_top = ht;
				check_my_ips();
				ht = m_ht->m_top;
				he = m_ht->m_ovflw;
			}

			/*
			 *  place a ref to the cell in the remote IPS.
			 *  The remote processor will then update this.
			 */
			lck = PRtolck(hpr);
			lock(lck);
			*hpr->gc_top = ToRef(p);
			hpr->gc_top++;
			unlock(lck);
		}
	}

	m_ht->m_top = ht;
}


check_my_ips()
{
	register Word	ips;
	char	*lck;

	/*
	 *  called when remote IPS is full; if local IPS is full process a cell
	 *  while waiting (to avoid livelock). Note: local IPS entries point to
	 *  local cells (and hence do not create IPS entries when copied).
	 */
	if (PR->gc_top > PR->gc_end)
	{	lck = PRtolck(PR);
		lock(lck);
		ips = *--PR->gc_top;
		unlock(lck);
		local_counter++;

		scan_and_copy( ToPtr(ips), ToPtr(ips)+1 );
	}
}


/*
 *  STACK_GC  --  garbage collect argument stack
 *		scan the process stack, copying arguments to local new stack
 */
stack_gc()
{
	register Process *ps;
	register Word	*st;
	Process	*pb, *pt;
	Mem	*mp, *oldm_sb;
	Word	*se, *ep, em;

	/*  (synchronisation is done in execute_event)  */

#ifdef GCTRACE
	printf("%d#ready for sgc, stack size = %d cells\n",
		PR-Pr0, get_used_mem(m_st));
#endif
#ifdef TIMER
	tim0 = get_clock();
#endif

	oldm_sb = m_sb;
	m_sb = m_st = alloc_new_stack_segment();
	segcnt = 1;

	/*
 	 *  scan process stack looking for argument and environment
	 *  pointers and copy the reachable cells to the local new stack
	 *  n.b. the old stk pointers may point to remote stacks
	 */
	st = m_st->m_data;
	se = m_st->m_ovflw;

	for (mp = m_pb; mp; mp = mp->m_next)
	{	pb = (Process *) mp->m_data;
		pt = (mp == m_pt) ? p_top : (Process *) mp->m_top;

		for (ps = pb; ps < pt; ps++)
		{	/*  copy process arguments  */
			if (ps->args != WNULL)
			{	copyargs(ps->args, st, ps->nargs);
				ps->args = st;
				st += ps->nargs;
			}

			/*  copy process environment  */
			if (ps->env != WNULL)
			{	ep = ps->env;
				em = (Word ) ps->env;
				ps->env = st;
				while (*ep != em)
					*st++ = *ep++;
				*st++ = (Word) ps->env;
			}

			if (st > se)
			{	m_st->m_top = st;
				m_st = alloc_stack_segment();
				st = m_st->m_data;
				se = m_st->m_ovflw;
			}
		}
	}

	m_st->m_top = st;

	/*
	 *  add old stack into free list, then request segcnt segments to
	 *  add to new stack (thus minimum requested stack size = 2 segments)
	 */
	free_segments(oldm_sb);

	(void) request_stack_segments(segcnt);

	if (ismaster(PR))
		(*st_ngcs)++;

#ifdef TIMER
	tim1 = get_clock();
#endif

#ifdef GCTRACE
	printf("%d# stack size = %d cells\n", PR-Pr0, get_used_mem(m_st));
#endif
#ifdef TIMER
	printf("%d# sgc done in %9.6f secs\n",
		PR-Pr0, (double)(tim1 - tim0) / 1000000);
#endif
}
