/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <strings.h>
#include <sys/ioctl.h>
#include <setjmp.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "macros.h"
#include "ret.h"
#include "stream.h"

jmp_buf	inbuf;

extern	Word	atom_eof, atom_read, atom_write, atom_append, atom_nil;
extern	Word	atom_user, atom_user_input, atom_user_output, atom_user_error;

extern	int	Ncalls;
extern	FILE	*open_ic_file();
#ifdef HERMES
extern	Stream	*st_stdout;
extern	int	wait_user;
extern	int	h_deadlock;
#endif

Word	stream_to_term();
int	blocking = 0;


/*
 *  EXECUTE_IO  --  execute i/o primitives that have been queued.
 *		    this can only be called by master processor;
 *		    all primitives assumed to have single clause
 *		    which executes instructions: [enter_io_c,commit,proceed]
 */
execute_io()
{
	register Process *queue, *P;
	Process	*new_q_front, *new_q_back;
	int	i, refcnt;
	Word	*SP;
	char	*lck;

	new_q_front = PNULL;
	SP = m_st->m_top;

	lck = ptrtolck(io_q_front);
	lock(lck);
	queue = *io_q_front;
	*io_q_front = *io_q_back = PNULL;
	unlock(lck);

	while (queue != PNULL)
	{	PS = queue;
		queue = queue->link;

		if (dead(PS->root))
			i = FAIL;
		else
			i = (*BuiltinP(PS->cont+1))(PS->args);

		if (i == SUCCESS)
		{	/*  execute a commit + proceed  */
			dealloc_stack(PS->args, PS->nargs);
#ifdef TRACE	/* was DEBUG */
			if (traceflag)
				printf("i/o primitive succeeds\n");
#endif

			Ncalls++;
			P = PS->parent;
			dealloc_ps(PS);

			if (!P)
				continue;
		
			if (P->refcount > 1)
			{	lck = ptrtolck(P);
				lock(lck);
				refcnt = --(P->refcount);
				unlock(lck);

				if (refcnt > 0)
					continue;
			}

			/*  fall thru to requeue parent  */
		}
		else if (i == REQUEUE)
		{	PS->link = PNULL;
			if (new_q_front == PNULL)
				new_q_front = new_q_back = PS;
			else
			{	new_q_back->link = PS;
				new_q_back = PS;
			}
			continue;
		}
		else if (i == SUSPEND)
		{	(void) suspend(ToPtr(*VarTbl));
			Nsuspv = 0;
			continue;
		}
		else	/*  fail  */
		{	dealloc_stack(PS->args, PS->nargs);

			P = PS->parent;
			dealloc_ps(PS);

			if (!P)			/*  top level fail  */
				err(1, "top level goal failed\n");
			else
			{
				P->cont = P->nextcl;
				P->nextcl = 0;	/* added by dac 9/1/92 */
			}
				/*  fall thru to requeue parent  */
		}

		if (dead(P))
		{
			refcnt = --(P->refcount);
			if (refcnt == 0)
				dealloc_ps(P);
			continue;
		}

		/*  requeue parent process  */
		enqueue_process(P);
	}


	/*  add any requeued processes to i/o queue  */
	if (new_q_front)
	{	lck = ptrtolck(io_q_front);
		lock(lck);
		if (*io_q_front != PNULL)
			(*io_q_back)->link = new_q_front;
		else
			*io_q_front = new_q_front;
		*io_q_back = new_q_back;
		unlock(lck);
	}

	m_st->m_top = SP;
	return;
}


/*
 *  STREAM PREDICATES
 */

io_open(Args)
Word	*Args;
{
	char	*filename, *mode;
	FILE	*fp;

	wait_for_argument(A0);
	if (!IsAtom(A0))
	{	bu_error(A0, "open/3: 1st arg incorrect");
		return(FAIL);
	}
	filename = string_val(A0);

	wait_for_argument(A1);
	if (A1 == atom_read)
		mode = "r";
	else if (A1 == atom_write)
		mode = "w";
	else if (A1 == atom_append)
		mode = "a";
	else
	{	bu_error(A1, "open/3: 2nd arg incorrect");
		return(FAIL);
	}

	fp = open_ic_file(filename, mode, (A1 == atom_read));
	if (fp == NULL)
	{	bu_error(A0, "cannot open file");
		return(FAIL);
	}

	return( unify(A2, stream_to_term(make_stream(A0,A1,fp))) );
}

io_close(Args)
Word	*Args;
{
	Stream	*st;
	int	i;

	wait_for_argument(A0);

	st = term_to_stream(A0);
	if (st == NULL)
	{	bu_error(A0, "close/1: arg is not a stream");
		return(FAIL);
	}

	/*  closing user streams has no effect  */
	if (st == st_user_input || st == st_user_output || st == st_user_error)
		return(SUCCESS);

	if (st == st_curr_input)
		st_curr_input = st_user_input;
	else if (st == st_curr_output)
		st_curr_output = st_user_output;

	i = fclose(st->st_fp);
	if (i == EOF)
	{	bu_error(st->st_name, "cannot close file");
		return(FAIL);
	}

	return(SUCCESS);
}

io_set_input(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL || st->st_mode != atom_read)
	{	bu_error(A0, "set_input/1: bad stream");
		return(FAIL);
	}

	st_curr_input = st;
	return(SUCCESS);
}

io_set_output(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL ||
	    (st->st_mode != atom_write && st->st_mode != atom_append))
	{	bu_error(A0, "set_input/1: bad stream");
		return(FAIL);
	}

	st_curr_output = st;
	return(SUCCESS);
}

io_curr_input(Args)
Word	*Args;
{
	return( unify(A0, stream_to_term(st_curr_input)) );
}

io_curr_output(Args)
Word	*Args;
{
	return( unify(A0, stream_to_term(st_curr_output)) );
}

io_flush(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL)
		return(FAIL);

	(void) fflush(st->st_fp);
	return(SUCCESS);
}



/*
 *  READ PREDICATES
 */

io_get0(Args)
Word	*Args;
{
	return( st_getc(st_curr_input, A0, 0) );
}

io_st_get0(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL || st->st_mode != atom_read)
	{	bu_error(A0, "get0/2: bad stream");
		return(FAIL);
	}

	return( st_getc(st, A1, 0) );
}

io_get(Args)
Word	*Args;
{
	return( st_getc(st_curr_input, A0, -1) );
}

io_st_get(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL || st->st_mode != atom_read)
	{	bu_error(A0, "get/2: bad stream");
		return(FAIL);
	}

	return( st_getc(st, A1, -1) );
}

io_skip(Args)
Word	*Args;
{
	int	c;

	wait_for_argument(A0);

	if (!IsShort(A0) || (c = ShortVal(A0)) < 0 || c > 127)
	{	bu_error(A0, "skip/1: arg incorrect");
		return(FAIL);
	}

	return( st_getc(st_curr_input, A0, c) );
}

io_st_skip(Args)
Word	*Args;
{
	Stream	*st;
	int	c;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL || st->st_mode != atom_read)
	{	bu_error(A0, "skip/2: bad stream");
		return(FAIL);
	}

	wait_for_argument(A1);

	if (!IsShort(A1) || (c = ShortVal(A1)) < 0 || c > 127)
	{	bu_error(A1, "skip/2: 2nd arg incorrect");
		return(FAIL);
	}

	return( st_getc(st, A1, c) );
}


io_gets(Args)
Word	*Args;
{
	return( st_gets(st_curr_input, A0) );
}

io_st_gets(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL || st->st_mode != atom_read)
	{	bu_error(A0, "gets/2: bad stream");
		return(FAIL);
	}

	return( st_gets(st, A1) );
}



/*
 *  WRITE PREDICATES
 */

io_put(Args)
Word	*Args;
{
	int	ch;

	wait_for_argument(A0);

	if (!IsShort(A0) || (ch = ShortVal(A0)) < 0 || ch > 127)
	{	bu_error(A0, "put/1: arg incorrect");
		return(FAIL);
	}

	if (putc(ch, st_curr_output->st_fp) == EOF)
		return(FAIL);
	return(SUCCESS);
}

io_st_put(Args)
Word	*Args;
{
	Stream	*st;
	int	ch;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL)
	{	bu_error(A0, "put/2: bad stream");
		return(FAIL);
	}

	wait_for_argument(A1);

	if (!IsShort(A1) || (ch = ShortVal(A1)) < 0 || ch > 127)
	{	bu_error(A1, "put/2: 2nd arg incorrect");
		return(FAIL);
	}

	if (putc(ch, st->st_fp) == EOF)
		return(FAIL);
	return(SUCCESS);
}

io_nl()
{
	if (putc('\n', st_curr_output->st_fp) == EOF)
		return(FAIL);
	return(SUCCESS);
}

io_st_nl(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL)
	{	bu_error(A0, "nl/1: bad stream");
		return(FAIL);
	}
	
	if (putc('\n', st->st_fp) == EOF)
		return(FAIL);
	return(SUCCESS);
}

io_tab(Args)
Word	*Args;
{
	int	n;

	wait_for_argument(A0);

	if (!IsShort(A0) || (n = ShortVal(A0)) < 0 || n > 127)
	{	bu_error(A0, "tab/1: arg incorrect");
		return(FAIL);
	}

	while (n-- > 0)
		(void) putc(' ', st_curr_output->st_fp);

	return(SUCCESS);
}

io_st_tab(Args)
Word	*Args;
{
	Stream	*st;
	int	n;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL)
	{	bu_error(A0, "tab/2: bad stream");
		return(FAIL);
	}

	wait_for_argument(A1);

	if (!IsShort(A1) || (n = ShortVal(A1)) < 0 || n > 127)
	{	bu_error(A1, "tab/2: 2nd arg incorrect");
		return(FAIL);
	}

	while (n-- > 0)
		(void) putc(' ', st->st_fp);

	return(SUCCESS);
}


io_display(Args)
Word	*Args;
{
	return( write_term_flush(st_curr_output, A0, 0) );
}

io_st_display(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL ||
	    (st->st_mode != atom_write && st->st_mode != atom_append))
	{	bu_error(A0, "$display/2: bad stream");
		return(FAIL);
	}

	return( write_term_flush(st, A1, 0) );
}

io_write(Args)
Word	*Args;
{
	return( write_term_flush(st_curr_output, A0, 1) );
}

io_st_write(Args)
Word	*Args;
{
	Stream	*st;

	wait_for_argument(A0);

	if ((st = term_to_stream(A0)) == NULL ||
	    (st->st_mode != atom_write && st->st_mode != atom_append))
	{	bu_error(A0, "$write/2: bad stream");
		return(FAIL);
	}

	return( write_term_flush(st, A1, 1) );
}


/*
 *  SUPPORT FUNCTIONS
 */
Word
stream_to_term(s)
Stream	*s;
{	
	if (s == st_user_input)
		return(atom_user_input);
	if (s == st_user_output)
		return(atom_user_output);
	if (s == st_user_error)
		return(atom_user_error);

	return(make_int((int)s));
}

Stream	*
term_to_stream(w)
Word	w;
{
	if (IsShort(w))
		return((Stream *) ShortVal(w));
	if (IsLong(w))
		return((Stream *) LongVal(w));

	/*  special cases  */
	if (w == atom_user || w == atom_user_input)
		return(st_user_input);
	if (w == atom_user_output)
		return(st_user_output);
	if (w == atom_user_error)
		return(st_user_error);

	return(NULL);
}


Stream	*
make_stream(name,mode,fp)
Word	name, mode;
FILE	*fp;
{
	Stream	*st = (Stream *) alloc_code(sizeof(Stream)/sizeof(short));

	st->st_fp = fp;
	st->st_name = name;
	st->st_mode = mode;

	return(st);
}


st_getc(st, arg, skip)
register Stream	*st;
Word	arg;
int	skip;
{
	register FILE	*fp = st->st_fp;
	int	ch;
	int	nbytes = BUFSIZ;

	if (st->st_mode == atom_eof)
	{	bu_error(st->st_name, "attempt to read past end of file");
		return(FAIL);
	}

	/*
	 *  if reading user_input and in non-blocking state determine how many
	 *  chars can be read before reading them; if none available: requeue
	 */
	if (st == st_user_input)
	{	if (PR->pr_state == PR_INPUT)
		{	if (setjmp(inbuf))
			{	PR->pr_state = PR_EXECUTE;
				blocking = 0;
#ifdef TRACE	/* was DEBUG */
				if (traceflag)
					printf("interrupted read\n");
#endif
#ifdef HERMES
				wait_user = 1;
#endif
				return(REQUEUE);
			}
			blocking = 1;
		}
		else
		{	if (fp->_cnt > 0)
				nbytes = fp->_cnt;
			else
			{	(void) ioctl(fileno(fp), FIONREAD, &nbytes);
				if (nbytes == 1)
				{
					ungetc((ch=getc(fp)), fp);
					if (ch == '\n') {
						fputs("|: ", st_user_output->st_fp);
						fflush(st_user_output->st_fp);
					}
				}
				else if (nbytes == 0)
				{
#ifdef HERMES
					wait_user = 1;
#endif
					return(REQUEUE);
				}
			}
		}
	}

#ifdef HERMES
	h_deadlock = 0;
#endif

#ifdef TRACE	/* was DEBUG */
	if (traceflag && blocking) printf("blocking on read\n");
#endif
	ch = getc(fp);
	blocking = 0;

	if (skip)
	{	while (ch != EOF)
		{	if (--nbytes < 0)
			{
#ifdef HERMES
				wait_user = 1;
#endif
				return(REQUEUE);
			}

			if (skip < 0 && ch > 32)	/*  get/1  */
				break;
			if (skip == ch)			/*  skip/1  */
				break;

			ch = getc(fp);
		}
	}

	if (ch == EOF)
	{	if (st == st_user_input)
			clearerr(fp);
		else
			st->st_mode = atom_eof;
	}

	return( unify(arg, ToShort(ch)) );
}


st_gets(st, arg)
register Stream	*st;
Word	arg;
{
	register FILE	*fp = st->st_fp;
	register int	ch;
	register Word	*hp, *car, *cdr;
	Word	list;
	int	nbytes = BUFSIZ;

	if (st->st_mode == atom_eof)
	{	bu_error(st->st_name, "attempt to read past end of file");
		return(FAIL);
	}

	/*
	 *  this assumes that user_input is in line mode, not cbreak/raw.
	 *  if one or more characters can be read, then the whole line
	 *  is read without checking whether it will block.
	 */
	if (st == st_user_input)
	{	if (PR->pr_state == PR_INPUT)
		{	if (setjmp(inbuf))
			{	PR->pr_state = PR_EXECUTE;
				blocking = 0;
#ifdef TRACE	/* was DEBUG */
				if (traceflag)
					printf("interrupted read\n");
#endif
#ifdef HERMES
				wait_user = 1;
#endif
				return(REQUEUE);
			}
			blocking = 1;
		}
		else
		{	if (fp->_cnt > 0)
				nbytes = fp->_cnt;
			else
			{	(void) ioctl(fileno(fp), FIONREAD, &nbytes);
				if (nbytes == 1)
				{
					ungetc((ch=getc(fp)), fp);
					if (ch == '\n') {
						fputs("? ", st_user_output->st_fp);
						fflush(st_user_output->st_fp);
					}
				}
				else if (nbytes == 0)
				{
#ifdef HERMES
					wait_user = 1;
#endif
					return(REQUEUE);
				}
			}
		}
	}

#ifdef HERMES
	h_deadlock = 0;
#endif

#ifdef TRACE	/* was DEBUG */
	if (traceflag && blocking) printf("blocking on read\n");
#endif
	ch = getc(fp);
	blocking = 0;

	hp = m_ht->m_top;
	car = hp++;
	cdr = hp++;

	while (ch != EOF && ch != '\n')
	{	*car = ToShort(ch);
		*cdr = AsList(hp);
		car = hp++;
		cdr = hp++;

		/*  check for '.'<space> - possibly end of term  */
		if (ch == '.')
		{	ch = getc(fp);
			if (ch <= 32)
				break;
		}
		else
			ch = getc(fp);
	}

	*car = ToShort(ch);
	*cdr = atom_nil;

	if (ch == EOF)
	{	if (st == st_user_input)
			clearerr(fp);
		else
			st->st_mode = atom_eof;
	}

	list = AsList(m_ht->m_top);
	m_ht->m_top = hp;

	if (m_ht->m_top > (Word *)(m_ht + 1))
		err(0, "st_gets: heap overflow\n");

	return( unify(arg, list) );
}


write_term_flush(st, term, quote)
Stream	*st;
Word	term;
int	quote;
{	int	result;

	result = write_term(st, term, quote);
	fflush(st->st_fp);
	return(result);
}


write_term(st, term, quote)
Stream	*st;
Word	term;
int	quote;
{
	FILE	*fp = st->st_fp;
	char	fnbuf[40], *fnp;
	Word	*p;
	Atom	*a;
	int	ttag, n;
	extern	Mem	*shmem_base;
	char	*quote_atom();

	deref(term);

	ttag = u_tag(term);
	switch (ttag)
	{
	case U_REF:
		if (InCurrHeapSeg(term))
			n = ToPtr(term) - m_ht->m_data;
		else
			n = ToPtr(term) - (Word *) shmem_base;
		fprintf(fp, "_%d", n);
		break;

	case U_STRUCT:
		p = StructVal(term);
		a = FunctVal(*p)->f_name;
		n = FunctVal(*p)->f_arity;

		fputs( quote ? quote_atom(a) : a->a_string, fp );

		(void) putc('(', fp);
		while (--n > 0)
		{	(void) write_term(st, ToRef(++p), quote);
			(void) putc(',', fp);
		}
		(void) write_term(st, ToRef(++p), quote);
		(void) putc(')', fp);
		break;

	case U_LIST:
		(void) putc('[', fp);
		p = ListVal(term);
		(void) write_term(st, ToRef(p), quote);

		term = ToRef(++p);
		deref(term);
		while (IsList(term))
		{	(void) putc(',', fp);
			p = ListVal(term);
			(void) write_term(st, ToRef(p), quote);

			term = ToRef(++p);
			deref(term);
		}

		if (term != atom_nil)
		{	(void) putc('|', fp);
			(void) write_term(st, term, quote);
		}
		(void) putc(']', fp);
		break;

	case U_CONST:
		if (IsShort(term))
			fprintf(fp, "%d", ShortVal(term));
		else if (IsAtom(term))
		{	a = AtomVal(term);
			fputs( quote ? quote_atom(a) : a->a_string, fp );
		}
		else if (IsFunct(term))
		{	a = FunctVal(term)->f_name;
			fprintf(fp,"%s/%d",quote ? quote_atom(a) : a->a_string,
					   FunctVal(term)->f_arity);
		}
		else if (IsLong(term))
			fprintf(fp, "%d", LongVal(term));
		else
		{	/*  ensure floats are not printed as integers  */
			(void) sprintf(fnbuf, "%g", FloatVal(term));
			for (fnp = fnbuf; *fnp; fnp++)
			{	if (*fnp == '.' || *fnp == 'e')
					break;
			}
			if (*fnp == '\0')
				(void) strcpy(fnp, ".0");
			fputs(fnbuf, fp);
		}
		break;
	}

	if (ferror(fp))
	{	bu_error(st->st_name, "write failed");
		return(FAIL);
	}
	return(SUCCESS);
}

char	*
quote_atom(a)
Atom	*a;
{
	static	char	qstr[A_STRSIZ*2];
	register char	*qs = qstr, ch;
	register char	*s = a->a_string;

	if (a->a_props == 0)
		return(s);

	*qs++ = '\'';
	while ((ch=*s++))
	{	if (ch == '\'')
		{	*qs++ = '\'';
			*qs++ = '\'';
		}
		else switch (ch)
		{
		case '\b':	/* backspace */
			*qs++ = '\\';
			*qs++ = 'b';
			break;
		case '\t':	/* tab */
			*qs++ = '\\';
			*qs++ = 't';
			break;
		case '\n':	/* linefeed */
			*qs++ = '\\';
			*qs++ = 'n';
			break;
		case '\v':	/* vertical tab */
			*qs++ = '\\';
			*qs++ = 'v';
			break;
		case '\f':	/* formfeed */
			*qs++ = '\\';
			*qs++ = 'f';
			break;
		case '\r':	/* carriage return */
			*qs++ = '\\';
			*qs++ = 'r';
			break;
		case '\007':	/* alarm (bell) */
			*qs++ = '\\';
			*qs++ = 'a';
			break;
		case '\033':	/* escape */
			*qs++ = '\\';
			*qs++ = 'e';
			break;
		case '\177':	/* delete */
			*qs++ = '\\';
			*qs++ = 'd';
			break;
		case '\\':	/* backslash */
			*qs++ = '\\';
			*qs++ = '\\';
			break;
		default:
			if (ch >= ' ')
				*qs++ = ch;
			else 	/* control character */
			{	*qs++ = '\\';
				*qs++ = '^';
				*qs++ = '@'+ch;
			}
			break;
		}
	}
	*qs++ = '\'';
	*qs = '\0';

	return(qstr);
}


/*
 *  print_term should now be obsolete, but remains for the print b.i.p.;
 *  may be useful for bu_error and functions in trace.c ?
 */
void
print_term(term)
Word	term;
{
	(void) write_term_flush(st_stdout, term, 1);
}
