/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <sys/time.h>
#include <sys/resource.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "macros.h"
#include "instr.h"
#include "ret.h"

#ifdef TRACE
#include <stdio.h>
#include "synch.h"
#include "event.h"
#include "trace.h"

extern	Atom	**a_htable;
extern	Word	(*bu_table[])();
extern	int	Ncalls;
extern	Mem	*shmem_base;
extern  int     *code_size;
extern	int	*sys_base;
extern	Code	*shd_code_end, *prv_code_end;
extern	void	print_term();

int	nextstop = 0;

/*
 *  Table of instructions, indexed by opcode
 */
struct	instruction	instructions[] =
{
	"halt",		 	"",	1,		/*  0 */

	"get_x_variable",	"xa",	5,		/*  1 */
	"get_x_value",		"xa",	5,		/*  2 */
	"get_y_variable",	"ya",	5,		/*  3 */
	"get_y_value",		"ya",	5,		/*  4 */
	"get_constant",		"ac",	5,		/*  5 */
	"get_nil",		"a",	5,		/*  6 */
	"get_list_var_var",	"axx",	5,		/*  7 */
	"get_list_var_val",	"axx",	5,		/*  8 */
	"get_list_val_var",	"axx",	5,		/*  9 */
	"get_list_val_val",	"axx",	5,		/* 10 */

	"wait_x_value",		"xa",	5,		/* 11 */
	"wait_constant",	"ac",	5,		/* 12 */
	"wait_nil",		"a",	5,		/* 13 */
	"wait_structure",	"as",	5,		/* 14 */
	"wait_list",		"a",	5,		/* 15 */
	"wait_variable",	"a",	5,		/* 16 */
	"wait_2_variables",	"aa",	5,		/* 17 */
	"unassigned (18)",	"",	1,		/* 18 */
	"unassigned (19)",	"",	1,		/* 19 */
	"unassigned (20)",	"",	1,		/* 20 */

	"put_x_variable",	"xa",	5,		/* 21 */
	"put_x_value",		"xa",	5,		/* 22 */
	"put_y_variable",	"ya",	5,		/* 23 */
	"put_y_value",		"ya",	5,		/* 24 */
	"put_constant",		"ac",	5,		/* 25 */
	"put_nil",		"a",	5,		/* 26 */
	"put_structure",	"as",	5,		/* 27 */
	"put_list",		"a",	5,		/* 28 */
	"unassigned (29)",	"",	1,		/* 29 */
	"unassigned (30)",	"",	1,		/* 30 */

	"push_x_variable",	"x",	5,		/* 31 */
	"push_x_value",		"x",	5,		/* 32 */
	"push_y_variable",	"y",	5,		/* 33 */
	"push_y_value",		"y",	5,		/* 34 */
	"push_constant",	"c",	5,		/* 35 */
	"push_nil",		"",	5,		/* 36 */
	"push_structure",	"s",	5,		/* 37 */
	"push_list",		"",	5,		/* 38 */
	"unassigned (39)",	"",	1,		/* 39 */
	"unassigned (40)",	"",	1,		/* 40 */

	"write_x_variable",	"x",	5,		/* 41 */
	"write_x_value",	"x",	5,		/* 42 */
	"write_constant",	"c",	5,		/* 43 */
	"write_nil",		"",	5,		/* 44 */
	"write_void",		"n",	1,		/* 45 */

	"read_x_variable",	"x",	5,		/* 46 */
	"read_x_value",		"x",	5,		/* 47 */
	"read_constant",	"c",	5,		/* 48 */
	"read_nil",		"",	5,		/* 49 */
	"read_void",		"n",	5,		/* 50 */

	"call",			"np",	1,		/* 51 */
	"call_last",		"np",	1,		/* 52 */
	"call_promoted",	"np",	1,		/* 53 */
	"call_promoted_last",	"np",	1,		/* 54 */
	"process_switch",	"",	1,		/* 55 */
	"proceed",		"",	2,		/* 56 */
	"try",			"l",	2,		/* 57 */
	"trust",		"l",	2,		/* 58 */
	"otherwise",		"",	2,		/* 59 */
	"otherwise_restore",	"",	2,		/* 60 */
	"begin_guard",		"",	2,		/* 61 */
	"end_guard",		"",	2,		/* 62 */
	"commit",		"",	2,		/* 63 */
	"fail",			"",	1,		/* 64 */

	"allocate",		"n",	4,		/* 65 */
	"deallocate",		"",	4,		/* 66 */

	"switch_on_term",	"xllll",3,		/* 67 */
	"wait_switch_on_term",	"xlll", 3,		/* 68 */

	"unassigned (69)",	"",	1,		/* 69 */
	"unassigned (70)",	"",	1,		/* 70 */

	"builtin_o",		"xb",	2,		/* 71 */
	"builtin_i",		"xb",	2,		/* 72 */
	"builtin_io",		"xxb",	2,		/* 73 */
	"builtin_ii",		"xxb",	2,		/* 74 */
	"builtin_nn",		"xxb",	2,		/* 75 */
	"builtin_ioo",		"xxxb",	2,		/* 76 */
	"builtin_iio",		"xxxb",	2,		/* 77 */
	"function_n",		"xbxl",	2,		/* 78 */
	"function_nn",		"xxbxl",2,		/* 79 */
	"increment",		"xxl",	2,		/* 80 */
	"decrement",		"xxl",	2,		/* 81 */
	"suspend_function",	"",	5,		/* 82 */

	"unassigned (83)",	"",	1,		/* 83 */
	"unassigned (84)",	"",	1,		/* 84 */

	"enter_c",		"l",	2,		/* 85 */
	"enter_io_c",		"l",	2,		/* 86 */
	"enter_interpret",	"",	2,		/* 87 */
	"enter_trace",		"",	2,		/* 88 */
	"enter_undefined",	"",	1,		/* 89 */
};
#define	NINSTRS	sizeof(instructions) / sizeof(struct instruction)

/*
 *  instruction tracing
 */
int	verifyflag = 0;				/*  instr verification  */
int	traceflag = 0;				/*  tracing level	*/
int	instr_count[NINSTRS];			/*  instruction stats	*/


verify_instr(c,sp,hp)
register Code *c;
Word	*sp, *hp;
{
	register int	i;
	register Word	w;
	char	*args;
	Code	*cp;
	Code	op;

	if (sp < m_st->m_data || sp >= (Word *) (m_st + 1))
		verify_msg("stack ptr out of bounds (%x)\n", sp);
	if (hp < m_ht->m_data || hp >= (Word *) (m_ht + 1))
		verify_msg("heap ptr out of bounds (%x)\n", hp);

	if (c < (Code *)sys_base || c >= (Code *)*mem_free) 
		verify_msg("code ptr out of bounds (%x)\n", c);

	op = *c++;
	if (op < 0 || op >= NINSTRS)
		verify_msg("unknown instruction (%d)\n", op);

	args = instructions[op].i_argtypes;

	while (*args)
	{	switch (*args)
		{
		case A_REGISTER:
			i = *c++;
			if (i < 0 || i > NARGS)
				verify_msg("reg index out of bounds (%d)\n", i);

			w = Reg[i];
			if (op < put_x_variable && !IsConst(w) &&
			    !HeapPtr(addr_to_segment(value(w))))
				verify_msg("reg ptrval not to heap (%x)\n", w);
			break;

		case X_REGISTER:
			i = *c++;
			if (i < 0 || i > NARGS)
				verify_msg("reg index out of bounds (%d)\n", i);
			break;

		case Y_REGISTER:
			i = *c++;
			if (PS->env == WNULL)
				verify_msg("no environment! (Y%d)\n", i);

			if (!StackPtr(addr_to_segment(PS->env+i)))
				verify_msg("env index out of bounds (%d)\n", i);

			w = PS->env[i];
			if (op == get_y_value && !IsConst(w) &&
			    !HeapPtr(addr_to_segment(value(w))))
				verify_msg("env ptrval not to heap (%x)\n", w);
			break;

		case INTEGER:
			c++;
			break;

		case CONSTANT:
		case STRUCTURE:
			w = *WordP(c);
			c += WordArgSize;
			if (!IsConst(w))
				verify_msg("bad const arg (%x)\n", w);
			break;

		case PROCEDURE:
			w = *WordP(c);
			if (c < (Code *)sys_base || c >= (Code *)*mem_free) 
				verify_msg("proc arg not to code (%x)\n", w);

			cp = (*ProcP(c))->p_code;
			c += WordArgSize;
			if (*cp != enter_trace && *cp != enter_undefined)
				cp -= TraceOffset;

			w = *WordP(cp+1);
			if (!IsFunct(w))
				verify_msg("bad proc arg (%x)\n", w);
			break;

		case BUILTIN:
			w = *WordP(c);
			c += WordArgSize;
			for  (i=0; i<60; i++)
			{	if (w == (Word) bu_table[i])
					break;
			}
			if (i==60)
				verify_msg("bad builtin arg (%x)\n", w);
			break;

		case LABEL:
			w = *WordP(c);
			c += WordArgSize;
			if (op < enter_c &&
			    addr_to_segment(w) != addr_to_segment(c))
				verify_msg("bad label arg (%x)\n", w);
			break;
		}

		args++;
	}
}

verify_ps()
{
	if (PS == PNULL)
		verify_msg("null ps\n", 0);

	if (!ProcessPtr(PS))
		verify_msg("ps not in proc stk (%x)\n", PS);

	if (PS->parent && !ProcessPtr(PS->parent))
		verify_msg("bad parent field (%x)\n", PS->parent);

	if (PS->refcount < 0)
		verify_msg("-ve ref cnt field (%d)\n", PS->refcount);

	if (PS->nargs > NARGS || PS->nargs < -1)
		verify_msg("bad nargs field (%d)\n", PS->nargs);

	if (PS->args && !StackPtr(PS->args))
		verify_msg("bad args field (%x)\n", PS->args);
}

trace_instr(c)
Code	*c;
{
	Code	op = *c;
	Code	*oldc;
	int	b, w, i, n;
	Word	*wp;
	char	*args, line[80], rtype;
	Process	*p;
	Funct	*f, *code_to_funct();
	Code	*cp;
	Process	*proc_base = (Process *)shmem_base;

	if (Ncalls < nextstop)
		return;

	/*
	 *  print current instruction and arguments
	 */
	printf("%d#%d executes %5d: %-16s ", (PR - Pr0), PS - proc_base,
					     c, instructions[op].i_opname);

Top:
	oldc = c++;
	args = instructions[op].i_argtypes;
	b = 1;
	w = 1;

	while (*args)
	{	if ((b + w) > 2)
			printf(",");

		switch(*args)
		{
		case A_REGISTER:
		case X_REGISTER:
		case Y_REGISTER:
			printf(" %c%d", *args, *c++);
			break;
			
		case INTEGER:
			printf(" %d", *c++);
			break;
			
		case CONSTANT:
		case STRUCTURE:
			printf(" ");
			print_term( *WordP(c) );
			c += WordArgSize;
			break;

		case PROCEDURE:
			cp = (*ProcP(c))->p_code;
			c += WordArgSize;
			if (*cp != enter_trace && *cp != enter_undefined)
				cp -= TraceOffset;

			f = FunctVal( *WordP(cp+1) );

			printf(" %s/%d", f->f_name->a_string, f->f_arity);
			break;

		case BUILTIN:
			/*
			 *  should create a table of <c_address, name>
			 *  entries so we can print name of builtin's
			 *  when in trace mode
			 */
			w = *WordP(c);
			c += WordArgSize;
			for  (i=0; i<60; i++)
			{	if (w == (Word) bu_table[i])
				{	printf(" bu_%d", i);
					break;
				}
			}
			break;

		case LABEL:
			printf(" %5d", *WordP(c));
			c += WordArgSize;
			break;
			
		}

		args++;
	}

	/*
	 *  prompt for next action, then do it
	 */
	printf(" \t? ");
	(void) gets(line);
	switch(line[0])
	{
	case '\0':
	case '\n':
	case ' ':
		return;		/*  should break out of while loop  */

	case 'q':
		*event = E_TERM;
		printf("%d# execution aborted\n", (PR - Pr0));
		(void) execute_event();
		icp_exit(0);

	case 'a':
		n = PS->nargs;
		if (sscanf(line, "a %d", &i) == 1)
			n = i;
		printf("args: ");
		for (i=0; i < n; i++)
		{	if (Reg[i] == 0)
				printf("(null)");
			else
				print_term(Reg[i]);

			printf((i==n-1) ? "\n" : ",");
		}
		break;

	case 'e':
		if (PS->env == WNULL)
		{	printf("no environment!\n");
			break;
		}
		printf("env: ");
		wp = PS->env;
		while (*wp != (Word)PS->env)
		{	if (*wp == 0)
				printf("(null)");
			else
				print_term(*wp);

			printf((*++wp == (Word)PS->env) ? "\n" : ",");
		}
		break;

	case 'p':
		if (sscanf(line, "p %d", &i) == 1)
		{	p = proc_base + i;
			/* align to process boundary */
			p = (Process *)((int)p + (int)((MNULL)->m_data));
		}
		else
			p = PS;
		printf("Process #%d: ", (p - proc_base));
		if (p->cont && p->args >= 0)
		{	f = code_to_funct(p->cont);
			printf("(%s/%d)", f->f_name->a_string, f->f_arity);
		}
		printf("\n");
		if (p->parent)
			printf("\tparent: %d\n", (p->parent - proc_base));
		printf("\targs:   %5x,%d\n", p->args, p->nargs);
		printf("\tenv:    %5x\n", p->env);
		printf("\tcont:   %5d\n", p->cont);
		printf("\tnextcl: %5d\n", p->nextcl);
		printf("\trefcnt: %d\n", p->refcount);
		if (p->link)
			printf("\tlink:   %d\n", (p->link - proc_base));
		if (p->root)
			printf("\troot:   %d\n", (p->root - proc_base));
		break;

	case 'r':
		(void) sscanf(line, "r %c%d", &rtype, &i);
		if (rtype == 'y')
			print_term(PS->env[i]);
		else
			print_term(Reg[i]);
		printf("\n");
		break;

	case 'n':
		if (sscanf(line, "n %d", &i) == 1)
		{	nextstop = Ncalls + i;
			return;
		}
		else
			printf("ncalls = %d\n", Ncalls);
		break;

	case 'w':
		for (p = PS; p != PNULL; p = p->parent)
		{	f = code_to_funct(p->cont);
			printf("%s/%d\n", f->f_name->a_string, f->f_arity);
		}
		break;

	case 'c':
		trace_instr_count();
		break;

	case 's':
		trace_symbols();
		break;

	case 't':
		(void) sscanf(line, "t %d", &traceflag);
		break;
	}
	c = oldc;
	printf("%d#%d at       %5d: %-16s ", (PR - Pr0), (PS - proc_base),
					   c, instructions[op].i_opname);
	goto Top;		/* hack - should be a while loop  */

	/* unlock trace routine */
}

trace_execute()
{
	Process	*proc_base = (Process *)shmem_base;

	if (Ncalls < nextstop)
		return;

	printf("%d#%d scheduled\n", PR - Pr0, PS - proc_base);
}

trace_suspend()
{
	int	i;
	Process	*proc_base = (Process *)shmem_base;

	if (Ncalls < nextstop)
		return;

	printf("#%d suspends on ", PS - proc_base);
	for (i=0; i<Nsuspv;)
	{	print_term(VarTbl[i]);
		printf(++i==Nsuspv ? "\n" : ", ");
	}
}

trace_fail()
{
	Process	*proc_base = (Process *)shmem_base;
	Funct	*f;

	if (Ncalls < nextstop)
		return;

	if (PS->cont && PS->args >= 0)
	{	f = code_to_funct(PS->cont);
		printf("#%d fails (%s/%d)\n",
			PS - proc_base, f->f_name->a_string, f->f_arity);
	}
	else
		printf("#%d fails\n", PS - proc_base);
}

trace_wake(p)
Process	*p;
{
	Process *proc_base = (Process *)shmem_base;

	if (Ncalls < nextstop)
		return;

	printf("%d# wake #%d\n", PR - Pr0, p - proc_base);
}


trace_instr_count()
{
	register int	i, total;

	for (i=0, total=0; i<NINSTRS; i++)
	{	if (instr_count[i] > 0)
		{	printf("%-20s: %6d\n",  instructions[i].i_opname,
						instr_count[i]);
			total += instr_count[i];
		}
	}
	printf("%d instructions executed\n", total);
}


trace_symbols()
{
	int	i;
	Atom	*a;
	Funct	*f;

	printf("atom table:\n");
	for (i=0; i < H_TBLSIZ; i++)
	{	a = a_htable[i];

		while (a)
		{	printf("%3d: atom %s\n", i, a->a_string);

			f = a->a_funct;
			while (f)
			{	printf("     funct %s/%d",
					f->f_name->a_string, f->f_arity);
				if (f->f_proc)
					printf(" proc at %d\n", f->f_proc);
				else
					printf("\n");

				f = f->f_next;
			}

			a = a->a_next;
		}
	}
}

verify_stack()
{
	register Word	*p, *st;
	Mem	*ms;

	for (ms = m_sb; ms; ms = ms->m_next)
	{	st = ms->m_top;
		for (p = ms->m_data; p < st; p++)
			verify_word(*p);
	}
}

verify_word(term)
Word	term;
{
	char	*a;
	int	n;
	double	fl;
	Word	*p;

	if (term == 0 || IsUnb(term))
		;
	else if (IsRef(term))
		verify_word((Word) *ToPtr(term));
	else if (IsConst(term))
	{	if (IsShort(term))
			n = ShortVal(term);
		else if (IsAtom(term))
			a = string_val(term);
		else if (IsFunct(term))
		{	a = (FunctVal(term)->f_name)->a_string;
			n = FunctVal(term)->f_arity;
		}
		else if (IsLong(term))
			n = LongVal(term);
		else if (IsFloat(term))
			fl = FloatVal(term);
		else
			printf("ERROR: invalid constant in arg stack\n");
	}
	else if (IsList(term))
	{	p = ListVal(term);
		verify_word(ToRef(p));
		verify_word(ToRef(++p));
	}
	else if (IsStruct(term))
	{	p = StructVal(term);
		if (IsFunct(*p))
		{	a = (FunctVal(*p)->f_name)->a_string;
			n = FunctVal(*p)->f_arity;
			while (n-- > 0)
				verify_word(ToRef(++p));
		}
		else
			printf("ERROR: invalid functor : %lx\n", *p);
	}
	else
		printf("ERROR: invalid term in arg stack\n");
}
#endif TRACE


/*
 *  SPY POINTS
 */
c_spy(Args)
Word	*Args;
{
	wait_for_argument(A0);
	return( set_spypoint(A0, 1) );
}

c_nospy(Args)
Word	*Args;
{
	wait_for_argument(A0);
	return( set_spypoint(A0, 0) );
}

set_spypoint(arg, on)
Word	arg;
int	on;
{
	Funct	*f;
	Proc	*p;

	if (IsAtom(arg))
		f = findfunct( AtomVal(arg), 0 );
	else if (IsStruct(arg))
		f = FunctVal( *StructVal(arg) );
	else
		return(FAIL);

	p = findproc(f, ANULL);
	if (p->p_code[0] == enter_undefined)
		return(FAIL);

	if (on)
	{	/*  check spy point is off  */
		if (p->p_code[0] != enter_trace)
			p->p_code -= TraceOffset;
	}
	else
	{	/*  check spy point is on  */
		if (p->p_code[0] == enter_trace)
			p->p_code += TraceOffset;
	}

	return(SUCCESS);
}


/*
 *  STATISTICS PREDICATES
 */
#define procsize	(sizeof(Process) / sizeof(Word))

extern	Mem	*shmem_base;		/*  shared memory base  */
extern	Code	**code_top, **code_end;	/*  code pointers	*/
extern  int     *code_size;		/*  code size		*/
extern	int	Ncalls;			/*  no. calls counter   */
extern	int	Nsuspensions;		/*  no. susps counter   */
extern	int	*st_ngcs;		/*  no. of g.c's        */
extern	int	*st_utime;		/*  user cpu time       */
extern	int	*st_last_utime;		/*  last user cpu time  */

Word	make_pair();


/*
 *  statistics  --  gather local statistics into shared "registers"
 *		    called from execute_event()
 */
statistics()
{
	if (ismaster(PR))
	{	*st_last_utime = *st_utime;
		*st_utime = user_time();
	}

	PR->st_ncalls = Ncalls;
	PR->st_nsusps = Nsuspensions;
	PR->st_h_used = get_used_mem(m_ht);
	PR->st_h_free = get_free_mem(m_ht);
	PR->st_s_used = get_used_mem(m_st);
	PR->st_s_free = get_free_mem(m_st);
	PR->st_p_used = get_used_mem(m_pt) / procsize - (p_end - p_top);
	PR->st_p_free = get_free_mem(m_pt) / procsize + (p_end - p_top);
	PR->st_c_free = c_end - c_top;
}


c_runtime(Args)
Word	*Args;
{
	int	ut, lt;

	/*  $runtime([CPU time since start, CPU time since last call])  */

	ut = *st_utime;
	lt = *st_utime - *st_last_utime;

	return( unify(A0, make_pair(make_int(ut), make_int(lt))) );
}

c_counts(Args)
Word	*Args;
{
	Processor *pr;
	int	ncalls = 0;
	int	nsusps = 0;

	/*  $counts(no of calls, no of suspensions, no of gcs)  */

	for (pr = Pr0; pr < PrN; pr++)
	{	ncalls += pr->st_ncalls;
		nsusps += pr->st_nsusps;
	}
	return( unify(A0, make_int(ncalls)) &&
		unify(A1, make_int(nsusps)) &&
		unify(A2, make_int(*st_ngcs)) );
}

c_mem_usage(Args)
Word	*Args;
{
	Processor *pr;
	int	memused, hused = 0, sused = 0, pused = 0, cused = 0;
	int	memfree, hfree = 0, sfree = 0, pfree = 0, cfree = 0;

	/*
	 *  $mem_usage(heap, stack, proc, code, total)
	 *		where each argument is a list of [mem used, mem free]
	 */

	for (pr = Pr0; pr < PrN; pr++)
	{	hused += pr->st_h_used;
		hfree += pr->st_h_free;
		sused += pr->st_s_used;
		sfree += pr->st_s_free;
		pused += pr->st_p_used;
		pfree += pr->st_p_free;
		cfree += pr->st_c_free;
	}
	cused = *code_size - cfree;
	cfree += *code_end - *code_top;

	memused = (char *) *mem_free - (char *) shmem_base;
	memfree = *mem_nfree * sizeof(Mem);

	return( unify(A0, make_pair(AsShort(hused), AsShort(hfree))) &&
		unify(A1, make_pair(AsShort(sused), AsShort(sfree))) &&
		unify(A2, make_pair(AsShort(pused), AsShort(pfree))) &&
		unify(A3, make_pair(AsShort(cused), AsShort(cfree))) &&
		unify(A4, make_pair(AsShort(memused), AsShort(memfree))) );
}


/*
 *  SUPPORT routines
 */

/*  convert to milliseconds  */
#define ttoi(t)	(t.tv_sec * 1000) + (t.tv_usec / 1000)

/*
 *  user_time  --  return user cpu time in milliseconds
 */
user_time()
{
	struct	rusage	r_usage;


	(void) getrusage(RUSAGE_SELF, &r_usage);
	return( ttoi(r_usage.ru_utime) );
}

/*
 *  make_pair  --  build a list pair with e1 and e2, thus: [e1, e2]
 */
Word
make_pair(e1, e2)
Word	e1, e2;
{
	register Word	*hp;
	extern	Word	atom_nil;

	hp = m_ht->m_top;
	hp[0] = e1;
	hp[1] = AsList(hp + 2);
	hp[2] = e2;
	hp[3] = atom_nil;
	m_ht->m_top = hp + 4;

	return(AsList(hp));
}
