/* Copyright (C) 1992 Imperial College */
/*--------------------------------------------------------------*
 *								*
 *			Arithmetic Operations			*
 *								*
 *--------------------------------------------------------------*/

#include "primitives.h"
#include "arith.h"

#define arithThrow(n) \
	{if (ERR_PT) { \
		return(AR_ERROR); \
	} else \
		throw(n);}

#ifndef GNUDOS
#define PI	3.14159265358979323846
#endif
#define checknum(reg,num,varerror,other)	\
			if (IsInt(reg)) num = (FLOAT)intvl(reg);	\
			else if (IsFloat(reg)) num = floatvl(reg);	\
			else arithThrow(IsVar(reg) ? varerror : other);

#ifdef sparc
FLOAT
floatvl(f)
cellpo f;
{
    register
    cellpo	c = (cellpo)vl(f);
    FLOAT	fp;
    fourBytes	*ptr;

    if ((unsigned long)c & 07) {	/* not aligned to 8-byte boundary */
	ptr = (fourBytes *) &fp;
	*ptr++ = (fourBytes) *c++;
	*ptr = (fourBytes) *c;
	return(fp);
    }
    else 
	return(* (FLOAT *) c);
}
#endif

bool
num_result(num, reg)
FLOAT	num;
cellpo	reg;
{
    cellpo	argmt = reg;
    fourBytes	rounded;

    delnk(argmt);

    switch (tg(argmt)) {
	case int_ref:
	    return(numequal(num, intvl(argmt)));

	case float_ref:
	    return(numequal(num, floatvl(argmt)));

	case var_ref:
	    mkreset(argmt);

	    /* convert to an integer, if close enough */
	    rounded = round(num);
	    if (numequal(num, rounded)) {
		mkint1(argmt, rounded);
		return(SUCCEED);
	    }

	    /* result is a floating point - need heap space */
	    if (gc_test(3L, 3)) {
		argmt = reg;
		delnk(argmt);
	    }

	    alloc_float(argmt, num);
	    return(SUCCEED);

	default:
	    arithThrow(218);
    }
}

bool
pr_add()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint1(arg3, (intvl(arg1) + intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) + intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_increment()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];

    delnk(arg1);
    delnk(arg2);

    if (IsInt(arg1)) {
	if (IsVar(arg2)) {
	    mkreset(arg2);
	    mkint1(arg2, (intvl(arg1)+1));
	    return(SUCCEED);
	}
	else if (IsInt(arg2))
	    return(intvl(arg1)+1 == intvl(arg2));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) ? 100 : 200);
}

bool
pr_decrement()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];

    delnk(arg1);
    delnk(arg2);

    if (IsInt(arg1)) {
	if (IsVar(arg2)) {
	    mkreset(arg2);
	    mkint1(arg2, (intvl(arg1)-1));
	    return(SUCCEED);
	}
	else if (IsInt(arg2))
	    return(intvl(arg1)-1 == intvl(arg2));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) ? 100 : 200);
}

bool
pr_subtract()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint1(arg3, (intvl(arg1) - intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) - intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_multiply()
{ 
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint1(arg3, (intvl(arg1) * intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) * intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_divide()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];
    FLOAT	num1, num2, num3;
    fourBytes	rounded;

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    if (num2 == 0.0)		/* divide by 0 error */
	arithThrow(600);

    num3 = num1 / num2;
    rounded = round(num3);

    if (IsInt(arg3))
	return(rounded == intvl(arg3));
    else if (IsVar(arg3)) {
	mkreset(arg3);
	    mkint1(arg3, rounded);
	    return(SUCCEED);
    }
    else arithThrow(200);
}

bool
pr_mod()
{ 
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    if (IsInt(arg1) && IsInt(arg2)) {
	cellpo arg3 = &A[3];
	delnk(arg3);

	if (!intvl(arg2))	/* divide by 0 error */
	    arithThrow(600);

	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint1(arg3, (intvl(arg1) % intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) % intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    if (num2 == 0.0)		/* divide by 0 error */
	arithThrow(600);

    num3 = fmod(num1, num2);

    return(num_result(num3, &A[3]));
}

bool
pr_lshift()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint1(arg3, (intvl(arg1) << intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) << intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_rshift()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint(arg3, (intvl(arg1) >> intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return(intvl(arg1) >> intvl(arg2) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_not()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];

    delnk(arg1);
    delnk(arg2);

    if (IsInt(arg1)) {
	if (IsVar(arg2)) {
	    mkreset(arg2);
	    mkint(arg2, (~intvl(arg1)));
	    return(SUCCEED);
	}
	else if (IsInt(arg2))
	    return(~intvl(arg1) == intvl(arg2));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) ? 100 : 200);
}

bool
pr_and()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint(arg3, (intvl(arg1) & intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return((intvl(arg1) & intvl(arg2)) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

bool
pr_or()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);
    delnk(arg2);
    delnk(arg3);

    if (IsInt(arg1) && IsInt(arg2)) {
	if (IsVar(arg3)) {
	    mkreset(arg3);
	    mkint(arg3, (intvl(arg1) | intvl(arg2)));
	    return(SUCCEED);
	}
	else if (IsInt(arg3))
	    return((intvl(arg1) | intvl(arg2)) == intvl(arg3));
	else arithThrow(200);
    }

    arithThrow(IsVar(arg1) || IsVar(arg2) ? 100 : 200);
}

/*------------------------------------------------------------*/

bool
pr_eq()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 == num2);
}

bool
pr_ne()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 != num2);
}

bool
pr_lt()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 < num2);
}

bool
pr_le()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 <= num2);
}

bool
pr_gt()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 > num2);
}

bool
pr_ge()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 101, 202);
    checknum(arg2, num2, 101, 202);

    return(num1 >= num2);
}

/*------------------------------------------------------------*/

int
cmp(term1, term2)
cellpo	term1, term2;
{
    register fourBytes val = 0;

    delnk(term1);
    delnk(term2);

    if (!sameTag(term1, term2)) {

	/* special case for comparing integers and floats */
	if (IsInt(term1) && IsFloat(term2))
	    return(((FLOAT)intvl(term1) > floatvl(term2)) ? 1 : -1);
	else if (IsFloat(term1) && IsInt(term2))
	    return(((FLOAT)intvl(term2) < floatvl(term1)) ? 1 : -1);
	    
	return((tg(term1) > tg(term2)) ? 1 : -1);
    }

    /* same tags */
    if (vl(term1) == vl(term2))
	return(0);

    switch (tg(term1)) {
    case int_ref:
	val = intvl(term1) - intvl(term2);
	break;
    case float_ref:
    {
	FLOAT t1 = floatvl(term1),
	      t2 = floatvl(term2);
	val = (t1 > t2) ? 1 : (t1 < t2) ? -1 : 0;
	break;
    }
    case symb_ref:
    {
	symbpo s1 = symbvl(term1),
	       s2 = symbvl(term2);
	utwoBytes length  = min(symblngth(s1), symblngth(s2));
	strpo symb1 = symbname(s1),
	      symb2 = symbname(s2);

	while (!val && length--)
	    val = *symb1++ - *symb2++;
	if (!val)
	    val = symblngth(s1) - symblngth(s2);
	break;
    }
    case nil_ref:
	val = 0;
	break;
    case var_ref:
	val = vl(term1) - vl(term2);
	break;
    case list_ref:
	if (!(val = cmp(hd(term1), hd(term2))))
	    val = cmp(tl(term1), tl(term2));
	break;
    case tpl_ref:
    {
	register int i = -1;
	val = arity(term1) - arity(term2);
	while (!val && ++i < arity(term1))
	    val = cmp(arg(term1, i), arg(term2, i));
	break;
    }
    default:
	(void) fprintf(stderr, "unknown tag in compare\n");
    }
    return((val == 0) ? 0 : (val > 0) ? 1 : -1);
}

/* term comparison */
/* A[1] is assumed to be a variable or -1, 0 or 1. */
bool
pr_cmp()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2],
		arg3 = &A[3];

    delnk(arg1);

    if (IsVar(arg1)) {
	mkreset(arg1);
	mkint(arg1, cmp(arg2, arg3));
	return(SUCCEED);
    }
    else return(intvl(arg1) == cmp(arg2, arg3));
}



/*--------------------------------------------------------------*
 *								*
 *			Floating Point				*
 *								*
 *--------------------------------------------------------------*/

bool
pr_fadd()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    num3 = num1 + num2;

    return(num_result(num3, &A[3]));
}

bool
pr_fsub()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    num3 = num1 - num2;

    return(num_result(num3, &A[3]));
}

bool
pr_fmul()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    num3 = num1 * num2;

    return(num_result(num3, &A[3]));
}

bool
pr_fdiv()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);

    if (num2 == 0.0)		/* divide by 0 error */
	arithThrow(600);

    num3 = num1 / num2;

    return(num_result(num3, &A[3]));
}



/*--------------------------------------------------------------*
 *								*
 *	T R I G O N O M E T R I C     F U N C T I O N S         *
 *								*
 *--------------------------------------------------------------*/

bool
pr_sin()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = sin(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_cos()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = cos(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_tan()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = tan(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_asin()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = asin(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_acos()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = acos(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_atan()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = atan(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_atan2()
{
    register
    cellpo	arg1 = &A[1],
    		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);
    num3 = atan2(num1, num2);

    return(num_result(num3, &A[3]));
}

bool
pr_sinh()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = sinh(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_cosh()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = cosh(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_tanh()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = tanh(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_exp()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = exp(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_log()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = log(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_log10()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = log10(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_sqrt()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = sqrt(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_pow()
{
    register
    cellpo	arg1 = &A[1],
    		arg2 = &A[2];
    FLOAT	num1, num2, num3;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    checknum(arg2, num2, 100, 218);
    num3 = pow(num1, num2);

    return(num_result(num3, &A[3]));
}

bool
pr_int()
{
    register
    cellpo	arg1 = &A[1],
		arg2 = &A[2];
    FLOAT	num1;
    fourBytes	rounded;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    rounded = round(num1);

    if (IsInt(arg2))
	return(rounded == intvl(arg2));
    else if (IsVar(arg2)) {
	mkreset(arg2);
	mkint1(arg2, rounded);
	return(SUCCEED);
    }
    else arithThrow(213);
}

bool
pr_abs()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = fabs(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_sign()
{
    register
    cellpo	arg1 = &A[1],
    		arg2 = &A[2];
    FLOAT	num1;
    fourBytes	num2;

    delnk(arg1);
    delnk(arg2);

    checknum(arg1, num1, 100, 218);
    num2 = num1 == 0.0 ? 0L : num1 < 0.0 ? -1L : 1L;

    if (IsInt(arg2))
	return(num2 == intvl(arg2));
    else if (IsVar(arg2)) {
	mkreset(arg2);
	mkint(arg2, num2);
	return(SUCCEED);
    }
    else arithThrow(213);
}

bool
pr_ceil()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = ceil(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_floor()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = floor(num1);

    return(num_result(num2, &A[2]));
}

bool
pr_pi()
{
    register
    cellpo	arg1 = &A[1];

    delnk(arg1);
    if (IsVar(arg1)) {
	mkreset(arg1);
	/* result is a floating point - need heap space */
	if (gc_test(3L, 3)) {
	    arg1 = &A[1];
	    delnk(arg1);
	}
	alloc_float(arg1, PI);
	return(SUCCEED);
    }
    else arithThrow(209);
}

bool
pr_rand()
{
    register
    cellpo	arg1 = &A[1];

    delnk(arg1);
    if (IsVar(arg1)) {
	mkreset(arg1);
	mkint(arg1, rand() & 0x7FFF);
	return(SUCCEED);
    }
    else arithThrow(209);
}

bool
pr_deg2rad()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = num1 * PI / 180.0;

    return(num_result(num2, &A[2]));
}

bool
pr_rad2deg()
{
    register
    cellpo	arg1 = &A[1];
    FLOAT	num1, num2;

    delnk(arg1);

    checknum(arg1, num1, 100, 218);
    num2 = num1 * 180.0 / PI;

    return(num_result(num2, &A[2]));
}

bool pr_ar_int()
{
	cellpo arg1 = &A[1], arg2 = &A[2];
	long val;

	delnk(arg1);
	delnk(arg2);

	if (IsInt(arg1))
		val = intvl(arg1);
	else if (IsFloat(arg1))
		val = (long) floatvl(arg1);
	else
		arithThrow(219);
	if (IsVar(arg2)) {
		mkreset(arg2);
		mkint(arg2, val);
	} else if (IsInt(arg2))
		return(intvl(arg2) == val);
	else
		return(FAIL);
	return(SUCCEED);
}

bool pr_ar_float()
{
	cellpo arg1, arg2;
	FLOAT val;

	(void) gc_test(3L, 2);

	arg1 = &A[1]; delnk(arg1);
	arg2 = &A[2]; delnk(arg2);

	if (IsInt(arg1))
		val = (FLOAT) intvl(arg1);
	else if (IsFloat(arg1))
		val = floatvl(arg1);
	else
		arithThrow(219);

	if (IsVar(arg2)) {
		mkreset(arg2);
		alloc_float(arg2, val);
	} else if (IsFloat(arg2))
		return(floatvl(arg2) == val);
	else
		return(FAIL);
	return(SUCCEED);
}
