/* Copyright (C) 1992 Imperial College */
/* ********************************************************************

ICP CODE GENERATOR
	Takes a list (in register 1) of instructions as prolog
	terms, and outputs 'binary' icp code.
	
    2/8/90	pjs
    	Constant and label tables are now hashed. As a result, the compiler 
    	is 10% faster. (The compiler now compiles in 4'30" - previous result
	was 5'02")

    23/6/90	dac
	Used block output routines (fwrite, memcpy) instead of a
	character at a time.  ( at least 10 times faster )
	Added many register declarations to improve performance.
    27/4/90	pjs
	Fixed bug in hash tables.  We forgot that 'mod' can return
	negative numbers.

    16/3/90	pjs
	added error handling

    3/3/90	pjs
	pc is defined as (nw - boc). The  fact that pc is  in excess
	of 4 does not affect the computation of  offsets and  allows
	UNDEF (i.e.'No fixup list') to be any integer between -3 and
	3 (We choose 0). That pc is really nw - (boc + 4) need  only
	be remembered when generating  the  Entry  Points  Table.

    22/2/90	dac
	The symbol '?fail?' is constructed in local space rather
	than the heap.  This avoids garbage collection.

    15/2/90	pjs
	const changed to con.

   ******************************************************************** */

#include "primitives.h"
#include "opcodes.h"

/* external declarations */

#if defined(GNUDOS)
extern	bool samesymb(symbpo, symbpo);
#elif defined(ANSI)
extern	bool samesymb(symbpo, symbpo);
extern	FILE *fopen(char *filename, char *mode);
#else
extern	bool samesymb();
extern	FILE *fopen();
extern	char *memcpy();
extern	char *memset();
#endif

extern	int debugLevel;
extern  symbpo fail_sym;
extern  symbpo load_sym;

static	jmp_buf errjump;

#define UP		1
#define DOWN		0


/* number of constants and labels in a code block */
#define MAX_C		300	
#define MAX_NL		150	

#define MAX_PL		150		/* initial size */
#define RESIZE_PL	100		/* resize increment */

#define MAX_CODE 	0x8000
#define CG_HEAP 	0x4000

#define GEN_REL		1
#define GEN_SEG		0

#define STATIC		0
#define DYNAMIC		1

#define SLASH		47

#define STRLEN		16

#define LOAD_FACTOR	0.8
#define ADDRESS_SIZE	132
#define TABLE_SIZE	138

#define ENTRY_SIZE 	4
#define FREE	 	-1
#define NO_LINK 	-2

#define pred_p(addr)	(IsTpl(addr) && \
			arity(addr) == 3 && \
			*string_val(functor(addr)) == SLASH)

#define pc		(twoBytes)(nw - boc)

/* The OpCode Table * ************************************************* */

struct instruction {
    fourBytes	hashcode;
    twoBytes	opcode_no;
    twoBytes	opcode_size;
    twoBytes	no_rands;
    twoBytes	link;
    uchar	string[STRLEN];
    uchar	operands[STRLEN];
} opcodes[] = {
  /*   0 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   1 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   2 */  { 0x702e5, 64, 8, 4,   0, "dynamic        ", "slll" },
  /*   3 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   4 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   5 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   6 */  { 0x702e9, 84, 1, 0,   0, "clr_err        ", "" },
  /*   7 */  { 0x702ea, 40, 4, 2, 123, "get_int        ", "rl" },
  /*   8 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*   9 */  { 0x4019c,  7, 5, 3, 135, "call           ", "Lsl" },
  /*  10 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  11 */  { 0x702ee, 75, 1, 0,   0, "set_nil        ", "" },
  /*  12 */  { 0x702ef, 43, 3, 2,   0, "get_tpl        ", "rs" },
  /*  13 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  14 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  15 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  16 */  { 0x702f3,  3, 2, 1, 137, "execute        ", "L" },
  /*  17 */  { 0x702f4, 83, 2, 1,   0, "set_err        ", "L" },
  /*  18 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  19 */  { 0xa0446, 54, 1, 0, 114, "unify_list     ", "" },
  /*  20 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  21 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  22 */  { 0x401a9,  0, 1, 0,   0, "halt           ", "" },
  /*  23 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  24 */  { 0x702fb, 31, 2, 1, 112, "put_nil        ", "r" },
  /*  25 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  26 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  27 */  { 0x401ae,  1, 1, 0,   0, "succ           ", "" },
  /*  28 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  29 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  30 */  { 0x80345,  9, 1, 0,   0, "allocate       ", "" },
  /*  31 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  32 */  { 0x70303, 30, 4, 2,   0, "put_int        ", "lr" },
  /*  33 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  34 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  35 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  36 */  { 0x8034b, 65, 8, 4,   0, "fdynamic       ", "slll" },
  /*  37 */  { 0x70308, 12, 2, 1, 133, "retryme        ", "L" },
  /*  38 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  39 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  40 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  41 */  { 0xb04a0, 80, 5, 1,   0, "unify_float    ", "f" },
  /*  42 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  43 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  44 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  45 */  { 0xb04a4, 47, 2, 1,   0, "unify_x_val    ", "r" },
  /*  46 */  { 0xb04a5, 48, 2, 1,   0, "unify_y_val    ", "y" },
  /*  47 */  { 0x300fa, 89, 1, 0,   0, "gc0            ", "" },
  /*  48 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  49 */  { 0x70314, 13, 1, 0, 122, "trustme        ", "" },
  /*  50 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  51 */  { 0xb04aa, 45, 2, 1,   0, "unify_x_var    ", "r" },
  /*  52 */  { 0x8035b, 42, 2, 1, 128, "get_list       ", "r" },
  /*  53 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  54 */  { 0xf0641, 49, 2, 1, 121, "unify_loc_x_val", "r" },
  /*  55 */  { 0xf0642, 50, 2, 1,   0, "unify_loc_y_val", "y" },
  /*  56 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  57 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  58 */  { 0xb04b1, 51, 2, 1,   0, "unify_const    ", "c" },
  /*  59 */  { 0x50212,  8, 5, 3,   0, "callv          ", "ssl" },
  /*  60 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  61 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  62 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  63 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  64 */  { 0x80367, 76, 1, 0,   0, "set_list       ", "" },
  /*  65 */  { 0x50218, 82, 2, 1,   0, "go_to          ", "L" },
  /*  66 */  { 0x80369,  4, 2, 1,   0, "executev       ", "s" },
  /*  67 */  { 0x8036a, 28, 2, 1, 126, "put_void       ", "r" },
  /*  68 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  69 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  70 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  71 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  72 */  { 0x8036f, 60, 0, 0,   0, "brkinstr       ", "" },
  /*  73 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  74 */  { 0x903b5, 79, 6, 2,   0, "get_float      ", "rf" },
  /*  75 */  { 0x50222, 57, 1, 0,   0, "pop_t          ", "" },
  /*  76 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  77 */  { 0x80374, 32, 2, 1,   0, "put_list       ", "r" },
  /*  78 */  { 0x903b9, 37, 3, 2,   0, "get_x_val      ", "rr" },
  /*  79 */  { 0x903ba, 38, 3, 2,   0, "get_y_val      ", "ry" },
  /*  80 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  81 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  82 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  83 */  { 0x6026e,  6, 5, 3,   0, "callme         ", "Lsl" },
  /*  84 */  { 0x903bf, 35, 3, 2,   0, "get_x_var      ", "rr" },
  /*  85 */  { 0x903c0, 36, 3, 2,   0, "get_y_var      ", "ry" },
  /*  86 */  { 0x903c1, 22, 3, 2, 125, "clr_y_var      ", "yr" },
  /*  87 */  { 0xc0512, 27, 3, 2,   0, "put_unsafe_y   ", "yr" },
  /*  88 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  89 */  { 0x00000,  0, 0, 0,  -1, "               ", "" },
  /*  90 */  { 0x50231, 11, 3, 2, 118, "tryme          ", "sL" },
  /*  91 */  { 0x903c6, 39, 3, 2, 117, "get_const      ", "rc" },
  /*  92 */  { 0xb046e, 88, 3, 2,   0, "gc_allocate    ", "ss" },
  /*  93 */  { 0x40180, 87, 1, 0,   0, "nop3           ", "" },
  /*  94 */  { 0x4017f, 86, 1, 0,   0, "nop2           ", "" },
  /*  95 */  { 0xa040e, 10, 1, 0, 136, "deallocate     ", "" },
  /*  96 */  { 0x6027b, 19, 2, 1, 119, "indexc         ", "r" },
  /*  97 */  { 0x903cc, 68, 2, 1,   0, "set_y_var      ", "y" },
  /*  98 */  { 0x903cd, 53, 1, 0,   0, "unify_nil      ", "" },
  /*  99 */  { 0xd0562, 71, 2, 1, 108, "set_loc_x_val  ", "r" },
  /* 100 */  { 0xd0563, 72, 2, 1,   0, "set_loc_y_val  ", "y" },
  /* 101 */  { 0x3014d, 85, 1, 0,   0, "nop            ", "" },
  /* 102 */  { 0x60281, 20, 2, 1,   0, "indexi         ", "r" },
  /* 103 */  { 0x903d2, 25, 3, 2, 115, "put_x_val      ", "rr" },
  /* 104 */  { 0x903d3, 26, 3, 2, 131, "put_y_val      ", "yr" },
  /* 105 */  { 0x903c1, 81, 5, 1,   0, "set_float      ", "f" },
  /* 106 */  { 0x903d5, 52, 3, 1,   0, "unify_int      ", "l" },
  /* 107 */  { 0x50242, 16, 2, 1,   0, "trust          ", "L" },
  /* 108 */  { 0x903ce, 78, 6, 2,   0, "put_float      ", "fr" },
  /* 109 */  { 0x903d8, 23, 3, 2,   0, "put_x_var      ", "rr" },
  /* 110 */  { 0x903d9, 24, 3, 2,   0, "put_y_var      ", "yr" },
  /* 111 */  { 0x903da, 55, 2, 1,   0, "unify_tpl      ", "s" },
  /* 112 */  { 0x702fb, 77, 2, 1,  94, "set_tpl        ", "s" },
  /* 113 */  { 0x6028c, 21, 4, 3,  93, "indext         ", "rLL" },
  /* 114 */  { 0x702f6, 74, 3, 1,   0, "set_int        ", "l" },
  /* 115 */  { 0x903d2, 73, 2, 1,   0, "set_const      ", "c" },
  /* 116 */  { 0x903df, 29, 3, 2,   0, "put_const      ", "cr" },
  /* 117 */  { 0x903c6, 70, 2, 1,   0, "set_y_val      ", "y" },
  /* 118 */  { 0x903c5, 69, 2, 1,   0, "set_x_val      ", "r" },
  /* 119 */  { 0x903cb, 67, 2, 1,   0, "set_x_var      ", "r" },
  /* 120 */  { 0x60293, 56, 1, 0,   0, "push_t         ", "" },
  /* 121 */  { 0x8035d, 66, 2, 1,   0, "set_void       ", "s" },
  /* 122 */  { 0x80358, 63, 1, 0,   0, "f_push_t       ", "" },
  /* 123 */  { 0x4019a, 62, 4, 2,  92, "data           ", "sl" },
  /* 124 */  { 0x30147,  2, 2, 1, 134, "jmp            ", "L" },
  /* 125 */  { 0x60271, 61, 2, 1, 105, "escape         ", "s" },
  /* 126 */  { 0x200ca, 59, 3, 2,   0, "gc             ", "ss" },
  /* 127 */  { 0x3014c, 58, 1, 0,   0, "cut            ", "" },
  /* 128 */  { 0xb04ab, 46, 2, 1,   0, "unify_y_var    ", "y" },
  /* 129 */  { 0xa043c, 44, 2, 1, 127, "unify_void     ", "s" },
  /* 130 */  { 0x702e2, 41, 2, 1, 101, "get_nil        ", "r" },
  /* 131 */  { 0x903d3, 34, 2, 1, 130, "put_slash      ", "r" },
  /* 132 */  { 0x3014f,  5, 5, 3,   0, "jsr            ", "Lsl" },
  /* 133 */  { 0x70308, 33, 3, 2,   0, "put_tpl        ", "sr" },
  /* 134 */  { 0xa042b, 18, 9, 8,   0, "arg_switch     ", "rLLLLLLL" },
  /* 135 */  { 0x4019c, 17, 1, 0, 129, "fail           ", "" },
  /* 136 */  { 0x50236, 15, 2, 1,   0, "retry          ", "L" },
  /* 137 */  { 0x3015f, 14, 3, 2,   0, "try            ", "sL" }
};

#define emitNULL { \
	register strpo operands1 = opcodes[op_code].operands; \
	register short no_rands1 = opcodes[op_code].no_rands; \
	register int offset = 1, rem = 0; \
	while (no_rands1--) { \
		if (*operands1 == 'f') { \
			rem = (nw-boc+offset) % 4; \
			switch (rem) { \
				case 1: \
					emit(nop3); \
					emit(nop); \
					emit(nop); \
					break; \
				case 2: \
					emit(nop2); \
					emit(nop); \
					break; \
				case 3: \
					emit(nop); \
					break; \
			} \
			break; \
		} else if (*operands1 == 'l') { \
			rem = (nw-boc+offset) % 2; \
			if (rem) \
				emit(nop); \
			break; \
		} \
		offset++; \
		operands1++; \
	} \
}

#define emitNULL1(type, offset) { \
	register rem = 0; \
	if (type == 'f') { \
		rem = (nw-boc+offset) % 4; \
		switch (rem) { \
			case 1: \
				emit(nop3); \
				emit(nop); \
				emit(nop); \
				break; \
			case 2: \
				emit(nop2); \
				emit(nop); \
				break; \
			case 3: \
				emit(nop); \
				break; \
		} \
	} else if (type == 'l') { \
		rem = (nw-boc+offset) % 2; \
		if (rem) \
			emit(nop); \
	} \
}

#ifdef GNUDOS
void emit_short(twoBytes);
void emit_int(int);
void emit_float(FLOAT);
#endif

twoBytes
lookup_opcode(instr)
cellpo instr;
{
    register
    twoBytes	i_op;
    symbpo	op_name		= symbvl(functor(instr));
    fourBytes	h 		= symbhashval(op_name);
    strpo	print_name 	= symbname(op_name);
    utwoBytes	length 		= symblngth(op_name);
    bool	no_match	= TRUE;
    i_op = h % ADDRESS_SIZE + 1;

    while (	(h != opcodes[i_op].hashcode ||
		(no_match=strncmp(print_name,opcodes[i_op].string,(int)length)))
	   && opcodes[i_op].link > 0)
	i_op = opcodes[i_op].link;

    if (no_match)
	longjmp(errjump,501);
    return(i_op);
}


/* **************************************************************** */

typedef struct c_and_nl_entry {
    symbpo	sym;
    twoBytes	addr;
    short	link;
} co, *cpo;

#define nl	co
#define nlpo	cpo

typedef struct pl_entry {
    symbpo	sym;
    twoBytes	addr;
    twoBytes	args;
} pl, *plpo;

typedef struct pub_entry {
    symbpo	s;
    twoBytes	n;
} *pubpo;


#define symbl(x)		(x)->sym
#define address(x)		(x)->addr
#define head_fix(x)		(x)->addr
#define ar(x)			(x)->args

#define set_symbl(x,y)		(x)->sym  = (y)
#define set_address(x,y)	(x)->addr = (y)
#define set_head_fix(x,y)	(x)->addr = (y)
#define set_ar(x,y)		(x)->args = (y)

/* UNDEF means a new undefined label */
#define UNDEF			0
#define is_fix(n)		((n) < UNDEF)
#define is_defined(addr)	(address(addr) > UNDEF)
#define fixup_const(c)		fixup(head_fix(c),NOT_A_PRED)
#define fixup_label(l)		fixup(head_fix(l),NOT_A_PRED)
#define fixup_pred(p,cntxt)	fixup(head_fix(p),cntxt)
#define LOCAL_PRED		0
#define EXTERNAL_PRED		1
#define NOT_A_PRED		-1

static codepo nw, boc, maxcode;
static cpo	xt_c,  st_c;
static nlpo	xt_nl, st_nl;
static plpo	xt_pl, st_pl, top_pl;
static twoBytes	fail_fixup;
static pubpo	Public, Pubend;
static short	max_c, max_nl, max_pl, cdivisor, nldivisor;
static cellpo	CgStart, CgEnd, CgH;
static bool	CgMode,CodeType;
static cell	c;
static cellpo	fail_cellpo = &c;

symbpo copy(in)
symbpo in;
{
    register
    symbpo out     = (symbpo)CgH;
    twoBytes len   = symblngth(in);
    twoBytes clen  = symbSize(len, sizeof(cell));
    if(CgH + clen >= CgEnd)
	longjmp(errjump,605);
    symblngth(out) = len;
    symbhash(out)  = symbhash(in);
    (void)strcpy(symbname(out),symbname(in));
    CgH += clen;
    return(out);
}

void
emit(word)
twoBytes word;
{
#ifdef GNUDOS
	(void)emit_short(word); /* emit is taken to mean store short! */
#else
    *nw = word;
    if (nw++ >= maxcode)
	longjmp(errjump,604);
#endif
}
#ifdef GNUDOS
#include "dos/emit_dos.c"
#endif

void reinit_nl()
{
    (void)memset((char*)st_nl,0,(SIZE_TYPE)(max_nl*sizeof(struct c_and_nl_entry)));
    xt_nl = st_nl + max_nl - 1;
}

bool init_cg()
{
    mksymb(fail_cellpo, fail_sym);

    boc = (codepo) alloc(MAX_CODE,2);
    maxcode = boc + MAX_CODE; 
    
    st_c = (cpo) alloc(MAX_C, sizeof(struct c_and_nl_entry)); 
    cdivisor = MAX_C - (MAX_C >> 2);
    max_c = MAX_C;    
    
    st_nl = (cpo) alloc(MAX_NL, sizeof(struct c_and_nl_entry));  
    nldivisor = MAX_NL - (MAX_NL >> 2);
    max_nl = MAX_NL;
    
    st_pl = (plpo) alloc(MAX_PL, sizeof(struct pl_entry)); 
    max_pl = MAX_PL;
    top_pl = st_pl + MAX_PL; 
    
    CgStart = (cellpo) alloc(CG_HEAP, sizeof(cell));
    CgEnd = &(CgStart[CG_HEAP]);

    return(boc && st_c && st_nl && st_pl && CgStart);
}

void cg_init()
{
    fail_fixup = 0;
    
    (void)memset((char*)st_nl,0,(SIZE_TYPE)max_nl*sizeof(struct c_and_nl_entry));
    xt_nl = st_nl + max_nl - 1;

    (void)memset((char*)st_c,0,(SIZE_TYPE)max_c*sizeof(struct c_and_nl_entry));
    xt_c = st_c + max_c - 1;
  
    (void)memset((char*)st_pl,0,(SIZE_TYPE)max_pl*sizeof(struct pl_entry));
    xt_pl = st_pl;
 
/* not necessary
    (void)memset((char*)boc,0,(SIZE_TYPE)MAX_CODE*2);
*/
    nw = boc;
    
    emit(0); /* skip four spaces for header */
    emit(0);
    emit(0);
    emit(0);
    
    if(CgMode == GEN_SEG) {
	(void)memset((char*)CgStart,0,(SIZE_TYPE)CG_HEAP*sizeof(cell));
	CgH = CgStart;
    }
}

void resize_table(stpo,xtpo,maxpo,divisorpo)
nlpo 	 *stpo, *xtpo;
short	 *maxpo, *divisorpo;
{
    nlpo 	p, 
    		st=*stpo, 
    		newst, 
    		newxt, 
    		probe, 
    		entry;
    short 	max_nl=*maxpo, 
    		newmax = *maxpo = max_nl * 2, 
    		newdivisor;
    *stpo = newst = (nlpo) alloc(newmax, sizeof(struct c_and_nl_entry));
	(void) memset((char *) newst, 0, (int) (newmax*sizeof(struct c_and_nl_entry)));
    if (!newst)
    	longjmp(errjump,606);
    newxt = newst + newmax - 1;
    newdivisor = *divisorpo = newmax - (newmax >> 2);

    for (p=st; max_nl--; p++) {
    	if (p->sym) {
	    probe = newst + symbhashval(p->sym) % newdivisor;
    	    if (probe < newst)
	    	probe += newdivisor;
	    for (;;) {	
		if (!probe->link) {	/* if end of chain, add symbol to new table */
		    if (probe->sym) {
			while (newxt->sym)
			    newxt--;
			entry = newxt;
			entry->sym = p->sym;
			entry->addr = p->addr;
			probe->link = entry - probe;
			break;
	            }
	            else {
	            	probe->sym = p->sym;
	            	probe->addr = p->addr;
	            	break;
	            }
	    	}
	        probe += probe->link; /* otherwise follow the chain */
            }
	}
    }

    while (newxt->sym)
	newxt--;
    free((char*)st);
    *xtpo = newxt;
}

void
resize_pl_table()
{
    register
    plpo st_pl1, xt_pl1, p;

    max_pl += RESIZE_PL;
    st_pl1 = xt_pl1 = (plpo) alloc(max_pl, sizeof(struct pl_entry));
    if (!st_pl1)
    	longjmp(errjump,606);
    top_pl = st_pl1 + max_pl;
    p = st_pl;
    for (; st_pl < xt_pl ;)
	*xt_pl1++ = *st_pl++;
    st_pl = st_pl1;
    xt_pl = xt_pl1;
    free((char*)p);
}

nlpo alloc_con()
{
    while (xt_c->sym && xt_c >= st_c)
	xt_c--;
    if (xt_c < st_c) {
	resize_table(&st_c,&xt_c,&max_c,&cdivisor);
	return(NULL);
    }
    return(xt_c);
}

nlpo alloc_nl()
{
    while (xt_nl->sym && xt_nl >= st_nl)
	xt_nl--;
    if (xt_nl < st_nl) {
	resize_table(&st_nl,&xt_nl,&max_nl,&nldivisor);
	return(NULL);
    }
    return(xt_nl);
}

bool lookup_label(s,lab)
symbpo s;
nlpo   *lab;
{
    register
    nlpo	probe;
    
    probe = st_nl + symbhashval(s) % nldivisor;

    if (probe < st_nl)
	probe += nldivisor;

    for (;;) {
	if (probe->sym && samesymb(s, probe->sym)) {
	    *lab=probe; /* already in table */
	    return(TRUE);
	}
	if (!probe->link) { /* end of chain */
	    *lab=probe;
	    return(FALSE);
	}	
	probe += probe->link;	/* follow chain */
    }
}

bool lookup_const(s,c)
symbpo s;
nlpo   *c;
{
    register
    nlpo	probe;
    
    probe = st_c + symbhashval(s) % cdivisor;

    if (probe < st_c)
	probe += cdivisor;

    for (;;) {
	if (probe->sym && samesymb(s, probe->sym)) {
	    *c=probe; /* already in table */
	    return(TRUE);
	}
	if (!probe->link) { /* end of chain */
	    *c=probe;
	    return(FALSE);
	}	
	probe += probe->link;	/* follow chain */
    }
}

nlpo
enter_label(l)
cellpo l;
{
    nlpo lprobe, lentry;
    symbpo s = symbvl(l);
    if (!lookup_label(s,&lprobe)) {
	if (!(lentry = lprobe->sym ? alloc_nl() : lprobe))
	    return(enter_label(l));
	lprobe->link = lentry - lprobe;
	set_symbl(lentry, CgMode == GEN_SEG ? copy(s) : s);
	set_address(lentry,UNDEF);
	return(lentry);
    }
    return(lprobe);
}

nlpo
enter_const(c)
symbpo c;
{
    nlpo cprobe, centry;
    if (!lookup_const(c,&cprobe)) {
	if (!(centry= cprobe->sym ? alloc_con() : cprobe))
	    return(enter_const(c));
	cprobe->link = centry - cprobe;
	set_symbl(centry, CgMode == GEN_SEG ? copy(c) : c);
	set_head_fix(centry,UNDEF);
	return(centry);
    }
    return(cprobe);
}

plpo
nextFreePred()
{
    if (xt_pl == top_pl)
	resize_pl_table();
    return(xt_pl++);
}

plpo
lookup_pred(s,args)
symbpo s;
twoBytes args;
{
    register
    plpo pred = st_pl; 
    while ( (pred < xt_pl) && (! (samesymb(s,symbl(pred)) && (args == ar(pred))))) {
	pred++;
    }
    return(pred);
}

plpo
enter_pred(p)
cellpo p;
{
    register
    plpo	pred ;
    register
    cellpo 	temp;
    symbpo 	s;
    twoBytes	args;

    delnk(p);

    /* get predicate name */
    temp = arg(p,1);
    delnk(temp);
    s = symbvl(temp) ;

    /* get arity */
    temp = arg(p,2);
    delnk(temp);
    args = (twoBytes)intvl(temp);

    pred = lookup_pred(s,args);
    if (pred == xt_pl) {
	pred = nextFreePred();
	set_symbl(pred, CgMode == GEN_SEG ? copy(s) : s);
	set_address(pred,UNDEF);
	set_ar(pred,args);
    }
    return(pred);
}

twoBytes
assemble_const(c)
symbpo c;
{
    register
    cpo		con = enter_const(c);
    register
    twoBytes	last_fix = head_fix(con);
    set_head_fix(con,- pc);
    return (is_fix(last_fix) ? (-last_fix - pc) : UNDEF);
}

twoBytes
assemble_label(l)
cellpo l;
{
    register
    nlpo	label = enter_label(l);
    register
    twoBytes	last_fix;

    if (is_defined(label))
	return(address(label) - pc);
    last_fix = head_fix(label);
    set_head_fix(label,- pc);
    return (is_fix(last_fix) ? (-last_fix - pc) : UNDEF);
}

twoBytes
assemble_pred(p)
cellpo p;
{
    register
    plpo	label = enter_pred(p);
    register
    twoBytes	last_fix;

    if (is_defined(label))
	return(address(label) - pc);

    last_fix = head_fix(label);
    set_head_fix(label,- pc);
    return (is_fix(last_fix) ? (-last_fix - pc) : UNDEF);
}

void
fixup(first_fix, cntxt)
twoBytes first_fix;
twoBytes cntxt;
{
    register
    codepo prev_fix, curr_fix;

    if (is_fix(first_fix)) {
	prev_fix = boc - first_fix;
	do
	{
	    curr_fix = prev_fix;
	    prev_fix += *prev_fix;
	    *curr_fix = (twoBytes)(nw - curr_fix);
	    if (cntxt == LOCAL_PRED) {
		if (*(curr_fix - 1) == call)
		    *(curr_fix - 1) = jsr;
		else *(curr_fix - 1) = jmp;
	    }
	} while (prev_fix != curr_fix);
    }
}

 /* Hashing Table ****************************************************** */

typedef struct indx {
    union {
	fourBytes integer;
	struct {
	    twoBytes args;
	    symbpo sym;
	} functor_key;
    } key;
    cellpo alternative;
    twoBytes link;
} *indxpo;

#define index_arity(addr)	(addr) -> key.functor_key.args
#define index_const(addr)	(addr) -> key.functor_key.sym
#define index_integer(addr)	(addr) -> key.integer
#define index_altern(addr)	(addr) -> alternative
#define index_link(addr)	(addr) -> link

enum entry_type { intgr, con, tpl };
twoBytes	no_of_entries, hash_table_size;
indxpo		first_entry, next_entry, top_of_hash_table;

twoBytes
length(list)
cellpo list;
{
    register twoBytes n = 0;
    delnk(list);
    while (IsList(list)) {
	list = tl(list);
	delnk(list);
	n++;
    }
    return(n);
}

void enter_in_hash_table();
indxpo next_entry_for();

void
construct_hash_table(instr, type)
cellpo instr;
enum entry_type type;
{
    register
    cellpo index, entry;
    indxpo i;
    index = (type == tpl) ? arg(instr,5) : arg(instr,3);
    delnk(index);

    no_of_entries     = length(index);
    hash_table_size   = no_of_entries / LOAD_FACTOR;
    first_entry       = (indxpo) alloc(hash_table_size, sizeof(struct indx));
    if (!first_entry)
    	longjmp(errjump,606);
    top_of_hash_table = first_entry + hash_table_size;
    next_entry        = top_of_hash_table - 1;

    for (i=first_entry; i<top_of_hash_table; i++)
	index_link(i) = FREE;

    while (IsList(index)) {
	entry = hd(index);
	delnk(entry);
	index = tl(index);
	delnk(index);
	enter_in_hash_table(entry,type);
    }
}

void
write_out_hash_table(instr, type)
cellpo instr;
enum entry_type type;
{
    register indxpo i;
    register twoBytes link;

    cellpo default_label = arg(instr,2);
    delnk(default_label);

    emit((hash_table_size + 1) * 4);
    emit(0);
    emit(assemble_label(default_label));
    emit(no_of_entries);

    for (i = first_entry; i < top_of_hash_table; i++) {
	if (index_link(i) == FREE) {
	    emit(0); emit(0); emit(0); emit(0);
	}
	else {
	    switch (type) {
		case intgr: {
		    register
		    fourBytes n = index_integer(i);
#ifdef GNUDOS
		    emit_int((int)n);
#else
		    emit((twoBytes) (n >> 16));
		    emit((twoBytes) (n & 0xFFFF));
#endif
		    break;
		}
		case con:
		    emit(0);
		    emit(assemble_const(index_const(i)));
		    break;
		case tpl:
		    emit(index_arity(i));
		    emit(assemble_const(index_const(i)));
		    break;
	    }
	    emit(assemble_label(index_altern(i)));
	    link = index_link(i);
	    if (link >= 0)
		emit((link - ((twoBytes)(i - first_entry))) * 4 - 3);
	    else if (link == NO_LINK)
		emit(0);
	    else emit(FREE);
	}
    }
   free((char*)first_entry);
}

void enter_in_hash_table(entry, type)
cellpo entry;
enum entry_type type;
{
    register
    cellpo temp;
    cellpo label;
    indxpo index;

    label = arg(entry,1);
    delnk(label);
    (void) enter_label(label);

    switch (type) {
	case intgr: {
	    register fourBytes n;
	    temp = functor(entry);
	    delnk(temp);
	    n = intvl(temp);
	    index = next_entry_for((twoBytes)(n % no_of_entries));
	    index_integer(index) = n;
	    break;
	}
	case con:
	    temp = functor(entry);
	    delnk(temp);
	    index = next_entry_for((twoBytes)(symbhashval(symbvl(temp))
					      % no_of_entries));
	    index_const(index) = symbvl(temp);
	    break;
	case tpl: {
	    register cellpo t, a;
	    short args;
	    t = functor(entry);
	    delnk(t);
	    temp = arg(t,1);
	    a = arg(t,2);
	    delnk(temp);
	    delnk(a);
	    args = intvl(a);
	    index = next_entry_for((twoBytes)(symbhashval(symbvl(temp))
					      * args % no_of_entries));
	    index_arity(index) = args;
	    index_const(index) = symbvl(temp);
	    break;
	}
    }
    index_altern(index) = label;
}

indxpo
next_entry_for(index)
twoBytes index;
{
    register
    indxpo i;

    if (index < 0)
	index += no_of_entries;

    i = first_entry + index;

    while (index_link(i) > 0)
	i = first_entry + index_link(i);
    if (index_link(i) == NO_LINK) {
	while (index_link(next_entry) != FREE)
	    next_entry--;
	index_link(i) = next_entry - first_entry;
	index_link(next_entry) = NO_LINK;
	return(next_entry);
    }
    index_link(i) = NO_LINK;
    return(i);
}

/* ******************************************************************* */

void
found_label(instr)
cellpo	instr;
{
    register
    nlpo label = enter_label(instr);
    if (is_defined(label))
	longjmp(errjump,502);
    fixup_label(label);
    set_address(label,pc);
}

void
found_pred(instr)
cellpo	instr;
{
    register plpo label = enter_pred(instr);
    register cellpo callee;

    if (is_defined(label))
	longjmp(errjump,502);
    callee = arg(instr,1);
    delnk(callee);
    (void) enter_const(symbvl(callee));
    fixup_pred(label, LOCAL_PRED);
    set_address(label,pc);
}

cellpo found_instr(instr,stream)
cellpo	instr;
cellpo	stream;
{
    register
    cellpo	cell1, cell2;
    register
    fourBytes	n;
    short	op_code   = lookup_opcode(instr);
    short	opcode_no = opcodes[op_code].opcode_no;

    switch (opcode_no) {
	case execute:
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    if (is_defined(enter_pred(cell1)))
		emit(jmp);
	    else emit(execute);
	    emit(assemble_pred(cell1));
	    cell2 = arg(cell1,1);
	    delnk(cell2);
	    (void) enter_const(symbvl(cell2));
	    break;
	case call:
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    emitNULL;
	    if (is_defined(enter_pred(cell1)))
		emit(jsr);
	    else
		emit(call);
	    emit(assemble_pred(cell1));
	    cell2 = arg(cell1,1);
	    delnk(cell2);
	    (void) enter_const(symbvl(cell2));
	    cell1 = arg(instr, 2);
	    delnk(cell1);
	    emit((twoBytes) intvl(cell1));
	    cell1 = arg(instr, 3);
	    delnk(cell1);
	    n = intvl(cell1);
#ifdef GNUDOS
	    emit_int((int)n);
#else
	    emit((twoBytes)(n >> 16));
	    emit((twoBytes)(n & 0xFFFF));
#endif
	    break;
	case indexc:
	    if ((nw-boc) % 2)
		emit(nop);
	    emit(indexc);
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    emit((short)intvl(cell1));
	    construct_hash_table(instr,con);
	    write_out_hash_table(instr,con);
	    break;
	case indext:
	    if ((nw-boc) % 2)
		emit(nop);
	    emit(indext);
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    emit((twoBytes)intvl(cell1));
	    cell1 = arg(instr,3);
	    delnk(cell1);
	    emit(assemble_label(cell1));
	    cell1 = arg(instr,4);
	    delnk(cell1);
	    emit(assemble_label(cell1));
	    construct_hash_table(instr,tpl);
	    write_out_hash_table(instr,tpl);
	    break;
	case indexi:
	    if ((nw-boc) % 2)
		emit(nop);
	    emit(opcode_no);
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    emit((twoBytes)intvl(cell1));
	    construct_hash_table(instr,intgr);
	    write_out_hash_table(instr,intgr);
	    break;
	case dynamic:
	    emitNULL;
	    emit(opcode_no);
	    cell1 = arg(instr,1);
	    delnk(cell1);
	    emit((twoBytes)intvl(cell1));
	    emit(0); emit(0);	/* NEXT */
	    emit(0); emit(0);	/* BIRTH */
	    emit(0); emit(0);	/* DEATH */
	    break;
	case gc: {		/* substitute gc_allocate instr'n
				   if followed by allocate() */
	    cellpo str = tl(stream);
	    cellpo next;
	    delnk(str);
	    if (NotList(str))
		goto def_inst;
	    next = hd(str);
	    delnk(next);
	    if (IsTpl(next) &&
			strcmp(string_val(functor(next)), "allocate") == 0) {
		op_code = 92;	/* HACK: hard-wired entry of gc_allocate */
		opcode_no = gc_allocate;
		stream = str;
	    }
	    goto def_inst;
	}

#define emit_low_argR(val) \
	{register int val1; \
	cell1 = arg(instr,1); \
	delnk(cell1); \
	if ((val1=intvl(cell1)) <= 4 && val1) \
		emit(val + val1 - 1); \
	else \
		goto def_inst;}
#define emit_low_argY(val) \
	{register int val1; \
	cell1 = arg(instr,1); \
	delnk(cell1); \
	if ((val1=intvl(cell1)) < 4) \
		emit(val + val1); \
	else \
		goto def_inst;}

#define emit_low_argRR(val) \
	{register int val1, val2; \
	cell1 = cell2 = arg(instr,1); \
	delnk(cell1); \
	cell2++; \
	delnk(cell2); \
	if (((val1=intvl(cell1)) <= 4) && val1 && ((val2=intvl(cell2)) <= 4) && val2) \
		emit(val + (val1 - 1)*4 + (val2 - 1)); \
	else \
		goto def_inst;}

#define emit_low_argRY(val) \
	{register int val1, val2; \
	cell1 = cell2 = arg(instr,1); \
	delnk(cell1); \
	cell2++; \
	delnk(cell2); \
	if ((val1=intvl(cell1)) <= 4 && val1) { \
		if ((val2=intvl(cell2)) < 4) \
			emit(val + (val1 - 1)*4 + val2); \
		else { \
			emit(val + 16 + (val1 - 1)); \
			emit((twoBytes)intvl(cell2)); \
		} \
	} else \
		goto def_inst;}

#define emit_low_argRZ(type, val) \
	{register int val1; \
	cell1 = cell2 = arg(instr,1); \
	delnk(cell1); \
	cell2++; \
	delnk(cell2); \
	if ((val1=intvl(cell1)) <= 4 && val1) { \
		emitNULL1(type, 1); \
		emit(val + (val1 - 1)); \
		emit_fn(type, cell2); \
	} else \
		goto def_inst;}

#define emit_low_argYR(val) \
	{register int val1, val2; \
	cell1 = cell2 = arg(instr,1); \
	cell2++; \
	delnk(cell2); \
	if ((val2=intvl(cell2)) <= 4 && val2) { \
		delnk(cell1); \
		if ((val1=intvl(cell1)) < 4) \
			emit(val + val1*4 + (val2-1)); \
		else { \
			emit(val + 16 + (val2 - 1)); \
			emit((twoBytes)intvl(cell1)); \
		} \
	} else \
		goto def_inst;}

#define emit_low_argZR(type, val) \
	{register int val1; \
	cell1 = cell2 = arg(instr,1); \
	cell2++; \
	delnk(cell2); \
	if ((val1=intvl(cell2)) <= 4 && val1) { \
		emitNULL1(type, 1); \
		delnk(cell1); \
		emit(val + (val1 - 1)); \
		emit_fn(type, cell1); \
	} else \
		goto def_inst;}

#include "cg_low.c"

def_inst:
	default: {
	    register strpo operands = opcodes[op_code].operands;
	    register short no_rands = opcodes[op_code].no_rands;

	    emitNULL;
	    emit(opcode_no);
	    cell2 = arg(instr,1);
	    while (no_rands--) {
		cell1 = cell2++;
		delnk(cell1);
		switch (*operands++) {
		    case 'r':
		    case 'y':
		    case 's':
			emit((twoBytes)intvl(cell1));
			break;
		    case 'l':
			n = intvl(cell1);
#ifdef GNUDOS
			emit_int((int)n);
#else
			emit((twoBytes)(n >> 16));
			emit((twoBytes)(n & 0xFFFF));
#endif
			break;
		    case 'c':
			emit(assemble_const(symbvl(cell1)));
			break;
		    case 'L':
			emit(assemble_label(cell1));
			break;
		    case 'f': {
			FLOAT f = floatvl(cell1);
#ifdef GNUDOS
			emit_float((FLOAT)f);
#else
			utwoBytes *flt = (utwoBytes *)&f;
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt);
#endif
			break;
		    }
		}
	    }
	}
    }
    return(stream);
}

/* ******************************************************************* */

bool is_public(foo, args)
symbpo 		foo;
twoBytes	args;
{
    register
    pubpo p = Public;

    /* '<LOAD>'/0 is always public */
    if (args == 0 && samesymb(load_sym, foo))
	return(TRUE);

    while ((p < Pubend) && ((args != p->n) || !samesymb(foo,p->s)))
	p++;
    return(p < Pubend);
}

bool process_public(public)
cellpo public;
{
    register
    cellpo	pred,
		foo,
		n;
    pubpo	p;

    if (IsNil(public)) {	/* empty public list */
	Public = Pubend = NULL;
	return(SUCCEED);
    }

    Public = p = (pubpo) alloc(length(public), sizeof(struct pub_entry));
    if (!Public)
    	return(FAIL);

    while (IsList(public)) {
	pred = hd(public);
	delnk(pred);
	if (IsSymb(pred)) {
	    p->s=symbvl(pred);
	    p++->n=0;
	} else {
	    foo = arg(pred,1);
	    delnk(foo);
	    p->s = symbvl(foo);
	    n = arg(pred,2);
	    delnk(n);
	    p++->n = (twoBytes)intvl(n);
	}
	public = tl(public);
	delnk(public);
    }
    Pubend = p;
    return(SUCCEED);
}

bool
gen_code(stream)
cellpo	stream;
{
    register
    cellpo	instr;           /* The Current Instruction */
    nlpo	entry;
    
    enter_label(fail_cellpo)->addr = fail_fixup;

    delnk(stream);

    while (IsList(stream)) {
    	instr = hd(stream);
	delnk(instr);

	if (IsSymb(instr))
	    found_label(instr);

	else if (pred_p(instr))
	    found_pred(instr);

	else stream = found_instr(instr,stream);

	stream = tl(stream);
	delnk(stream);
    }

    if (NotNil(stream))	/* error in instruction stream */
	longjmp(errjump,500);
	
    (void) lookup_label(fail_sym,&entry);
    fail_fixup = entry->addr;
       
    reinit_nl(); 

    return(SUCCEED);
}

gen_externals()
{
    register
    plpo	i;
    twoBytes 	n;
    for (i = st_pl; i < xt_pl; i++) {
	if (! is_defined(i)) {
	    fixup_pred(i,EXTERNAL_PRED);
	    emit(0);
	    emit(0);
	    emit(ar(i));
	    n = assemble_const(symbl(i));
	    emit(n);
	}
    }
}

void
gen_entry_point() /* Generate xtra entry point for dynamic predicate */
{
    emit(0); /* The address of the predicate is always zero */
    emit(Public->n);
    emit(assemble_const(Public->s));
}

void
gen_entry_points()
{
    register plpo	p;
    register symbpo	s;
    register twoBytes 	n;

    if (Public == Pubend)	/* public list is empty */
	for (p = st_pl; p < xt_pl; p++) {
	    if (is_defined(p)) {
		emit(address(p) - 4);
		emit(ar(p));
		emit(assemble_const(symbl(p)));
	    }
	}
    else
	for (p = st_pl; p < xt_pl; p++) {
	    n = ar(p);
	    s = symbl(p);
	    if (is_defined(p) && is_public(s,n)) {
		emit(address(p) - 4);
		emit(n);
		emit(assemble_const(s));
	    }
	}
}

gen_constants()
{
    register cpo i;
    register twoBytes *pr_name;
    register utwoBytes length;
    for (i = st_c + max_c - 1; i >= st_c; i--) {
    	if(i->sym) {
	    if ((nw-boc) % 2)
		emit(nop);
	    fixup_const(i);
	    length  = symbSize(symblngth(symbl(i)), sizeof(twoBytes));
	    pr_name = (twoBytes*) (symbl(i));
	    while(length--)
	    	emit(*pr_name++);
	}
    }
}

codegen_cleanup()
{
    free((char *)Public);
}

/* ******************************************************************* */

/* A[1] is assumed to be 0, 1 or 2 */

bool pr_cg_init()
{
    cellpo 	reg1 = &A[1];
    twoBytes 	error;

    delnk(reg1);

    switch (intvl(reg1)) {
	case 0 :
	    CgMode = GEN_SEG;
	    CodeType = STATIC;
	    break;
	case 1 :
	    CgMode = GEN_REL;
	    CodeType = STATIC;
	    break;
	case 2 :
	    CgMode = GEN_REL;
	    CodeType = DYNAMIC;
	    break;
    }

    if (error = setjmp(errjump))
	throw(error)
    else cg_init();

    return(SUCCEED);
}


bool pr_cg()
{
    register
    cellpo	stream	= &A[1];	/* The Instruction Stream */
    int		errcode;
        
    if (errcode = setjmp(errjump)) {
	codegen_cleanup();
	throw(errcode);
    }
    else return(gen_code(stream));
}

bool pr_cg_fixup()
{
    register cellpo	public	= &A[1], /* The Public Predicates */
    			size	= &A[2]; /* The size of the code segment generated */

    /* When generating code for a dynamic predicate, the public */
    /* list must contain the predicate's name */

    delnk(public);

    if (!process_public(public))
    	throw(606);

    /* insert the fail instruction if '?fail?' was referenced */
    if (is_fix(fail_fixup)) {
	fixup(fail_fixup,NOT_A_PRED);
	emit(fail);
    }

    if ((nw-boc) % 2)	/* on odd address ? */
	emit(nop);

    /* the fourth cell points to beginning of externals */
    boc[3] = pc - 4;
    gen_externals();

    /* the second cell points to beginning of entrypoints */
    boc[1] = pc - 4;
    
    if (CodeType == DYNAMIC)
	gen_entry_point(); /* Generate extra entry point for dynamic pred */
    gen_entry_points();

    /* the third cell points to beginning of constants */
    boc[2] = pc - 4;
    gen_constants();

    /* the first cell is length of code block */
    boc[0] = pc - 1;	/* 4 cells for header - 1 */
    
    delnk(size);
    mkreset(size);
    mkint(size,pc);
   
    codegen_cleanup();
   
    return(SUCCEED);
   
}

bool pr_cg_out()
{
    /* write the generated code segment to the current output */

/* for debugging

    FILE	*inspect;
    register codepo   p;
    debug(3,inspect = fopen("inspect","w"));
    for (p = boc; p < nw; p++) {
	debug(3,fprintf(inspect,"%d\n",*p));
	(void) fprintf(inspect,"%d\n",*p);
    }
    debug(3,(void)fclose(inspect));
*/

    /* much faster to write the whole block than one character at a time */
    if (charout == pr_write_stream) {
	/* fast stream output */
	(void)fwrite((char *)boc, 2, nw-boc, fdes(current_output));
	
    }
    else if (charout == pr_write_ram) {
	/* fast memory output */
	register strpo mem = ramd(current_output)->write;
	(void)memcpy(mem, (char*)boc, 2*(nw-boc));
	ramd(current_output)->write += 2*(nw-boc);
    }
    
    return(SUCCEED);
}

emit_fn(type, cell1)
cellpo	cell1;
char type;
{
	fourBytes n;
	FLOAT f;
	utwoBytes *flt;

	switch (type) {
		case 'r':
		case 'y':
		case 's':
			emit((twoBytes)intvl(cell1));
			break;
		case 'l':
			n = intvl(cell1);
#ifdef GNUDOS
			emit_int((int)n);
#else
			emit((twoBytes)(n >> 16));
			emit((twoBytes)(n & 0xFFFF));
#endif
			break;
		case 'c':
			emit(assemble_const(symbvl(cell1)));
			break;
		case 'L':
			emit(assemble_label(cell1));
			break;
		case 'f':
			f = floatvl(cell1);
#ifdef GNUDOS
			emit_float((FLOAT)f);
#else
			flt = (utwoBytes *)&f;
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt++);
			emit((twoBytes)*flt);
#endif
			break;
	}
}
