/* Copyright (C) 1992 Imperial College */
/* debugging functions for IC-Prolog V2 */
/*
    27/6/90	pjs
	call and execute now display predicate names rather than addresses
    27/3/90	dac
	added check to terminate listing of illegal instructions
*/

#include "primitives.h"
#include "gc.h"
#include "opcodes.h"
#include "symtab.h"
#include "dynamic.h"
#include "termio.h"

extern noref();
extern refs();
extern codepo	boot;
extern choicepo	CATCH;
extern int	debugLevel, oldLevel;
extern fourBytes bit[];
extern cell	symbolHeap[];

extern void icp_exit();

#define LISTING		20	/* number of instr'ns to be displayed */
#define TRACELIST	20	/* default number of instr'ns displayed */
#define MAXDEPTH	10	/* deepest structure displayed */
#define MAXNAME		20	/* longest print name displayed */
#define MAXPRTDUMP	0x100L	/* maximum memory dump size */
#define BREAKPOINTS	10	/* number of breakpoints */



/* verbosity of display */
static bool	verbose = FALSE;
static int	maxdepth = MAXDEPTH;
static int	listingCount = LISTING;

/* length of each instruction */
short		instrLen[NUMINSTR];

/* global variable which contains the broken codes */
opcode		brkcode[BREAKPOINTS];
fourBytes	brkpoint[BREAKPOINTS];
static short	numBreaks = 0;

/* circular list to hold most recently executed instructions */
fourBytes	trace[AUDITSIZE];
short		tracePtr = 0;

/* details about current call/environment */
short		numRegs = DISPLREG;
short		numLocals = 0;
fourBytes	currentLocals = 0L;

void printName();



opcode
get_brkcode(addr)
fourBytes	addr;
{
    short	i = 0;
    while (i < numBreaks && brkpoint[i] != addr)
	i++;
    return((i == BREAKPOINTS) ? -1 : brkcode[i]);
}

init_instr()
{
    int		i;
    for (i=0; i<AUDITSIZE; i++)
	trace[i] = -1;

    instrLen[halt]		= 1;
    instrLen[succ]		= 1;
    instrLen[jmp]		= 2;
    instrLen[execute]		= 2;
    instrLen[executev]		= 2;
    instrLen[jsr]		= 5;
    instrLen[callme]		= 5;
    instrLen[call]		= 5;
    instrLen[callv]		= 5;
    instrLen[allocate]		= 1;
    instrLen[deallocate]	= 1;
    instrLen[tryme]		= 3;
    instrLen[retryme]		= 2;
    instrLen[trustme]		= 1;
    instrLen[try]		= 3;
    instrLen[retry]		= 2;
    instrLen[trust]		= 2;
    instrLen[fail]		= 1;
    instrLen[arg_switch]	= 9;
    instrLen[indexc]		= 0;	/* These instructions have variable length because */
    instrLen[indexi]		= 0;	/* they contain a hash table.  These special cases */
    instrLen[indext]		= 0;	/* are handled in procedure 'oneInstruction'.      */
    instrLen[clr_y_var]		= 3;
    instrLen[put_x_var]		= 3;
    instrLen[put_y_var]		= 3;
    instrLen[put_x_val]		= 3;
    instrLen[put_y_val]		= 3;
    instrLen[put_unsafe_y]	= 3;
    instrLen[put_void]		= 2;
    instrLen[put_const]		= 3;
    instrLen[put_int]		= 4;
    instrLen[put_nil]		= 2;
    instrLen[put_list]		= 2;
    instrLen[put_tpl]		= 3;
    instrLen[put_slash]		= 2;
    instrLen[get_x_var]		= 3;
    instrLen[get_y_var]		= 3;
    instrLen[get_x_val]		= 3;
    instrLen[get_y_val]		= 3;
    instrLen[get_const]		= 3;
    instrLen[get_int]		= 4;
    instrLen[get_nil]		= 2;
    instrLen[get_list]		= 2;
    instrLen[get_tpl]		= 3;
    instrLen[unify_void]	= 2;
    instrLen[unify_x_var]	= 2;
    instrLen[unify_y_var]	= 2;
    instrLen[unify_x_val]	= 2;
    instrLen[unify_y_val]	= 2;
    instrLen[unify_loc_x_val]	= 2;
    instrLen[unify_loc_y_val]	= 2;
    instrLen[unify_const]	= 2;
    instrLen[unify_int]		= 3;
    instrLen[unify_nil]		= 1;
    instrLen[unify_list]	= 1;
    instrLen[unify_tpl]		= 2;
    instrLen[push_t]		= 1;
    instrLen[pop_t]		= 1;
    instrLen[cut]		= 1;
    instrLen[gc]		= 3;
    instrLen[brkinstr]		= 0;
    instrLen[escape]		= 2;
    instrLen[data]		= 4;
    instrLen[f_push_t]		= 1;
    instrLen[dynamic]		= 8;
    instrLen[fdynamic]		= 8;
    instrLen[set_void]		= 2;
    instrLen[set_x_var]		= 2;
    instrLen[set_y_var]		= 2;
    instrLen[set_x_val]		= 2;
    instrLen[set_y_val]		= 2;
    instrLen[set_loc_x_val]	= 2;
    instrLen[set_loc_y_val]	= 2;
    instrLen[set_const]		= 2;
    instrLen[set_int]		= 3;
    instrLen[set_nil]		= 1;
    instrLen[set_list]		= 1;
    instrLen[set_tpl]		= 2;
    instrLen[put_float]		= 6;
    instrLen[get_float]		= 6;
    instrLen[unify_float]	= 5;
    instrLen[set_float]		= 5;
    instrLen[go_to]		= 2;
    instrLen[set_err]		= 2;
    instrLen[clr_err]		= 1;
    instrLen[nop]		= 1;
    instrLen[nop2]		= 1;
    instrLen[nop3]		= 1;
#include "instrlen_low.c"
}

#ifdef DEBUG
/* This is a very hacky procedure for determining the currently valid local */
/* variables.  The reason for this is that until a call instruction is      */
/* executed, the size of the current environment is not known.              */
void
guessLocals(P)
    codepo P;
{
    opcode I;
    while ((I = *P) > HIGH_CALL) {
	if (I >= NUMINSTR)
	    longjmp(icp_interrupt, 506);
	else if (I == fail)
	    break;
	else if (I == brkinstr) {
	    opcode br_code = get_brkcode(PC(P));
	    if (br_code <= HIGH_CALL)
		break;
	    else P += instrLen[br_code];
	}
	else P += instrLen[I];
    }

    if (I < LOW_CALL || I > HIGH_CALL) {	/* no call instructions found */
	numLocals = 8;
	currentLocals = 0xFFL;
    }
    else {	/* found a call instruction */
	P += 5;
	numLocals = envsize(P);
	currentLocals = envset(P);
    }
}
#endif

/*
 * return a string describing the instruction at given 
 * address and increments pointer to next instruction
 */
#define	oprnd(n)	*(P + n)
#define	lbl(n)		n + oprnd(n)
#define	adr(n)		PC(P) + lbl(n)

codepo
oneInstruction(P)
    codepo	P;
{
    opcode	instr = *P;

    switch (instr) {
    case halt:			/* stop execution */
	(void) printf("halt\n");
	break;

    case succ:			/* succeed clause */
	(void) printf("succ\n");
	break;

    case jmp:			/* local last call */
	(void) printf("jmp $%ld\n", adr(1));
	break;

    case execute: {		/* external last call */
	codepo PP = P + oprnd(1) + 4;
	(void) printf("execute ");
	printName((symbpo)rel(PP), stdout);
	(void) printf("/%d\n", *(PP-1));
	break;
    }

    case executev:		/* meta last call */
	(void) printf("executev %d\n", oprnd(1));
	break;

    case jsr:			/* call a local procedure */
	(void) printf("jsr $%ld,%d,0x%08lx\n", adr(1), oprnd(2), *(fourBytes *)(P+3));
	break;

    case callme:		/* inline jsr */
	(void) printf("callme $%ld,%d,0x%08lx\n", adr(1), oprnd(2), *(fourBytes *)(P+3));
	break;

    case call: {		/* call an external procedure */
	codepo PP = P + oprnd(1) + 4;
	(void) printf("call ");
	printName((symbpo)rel(PP), stdout);
	(void) printf("/%d,%d,0x%08lx\n", *(PP-1), oprnd(2), *(fourBytes *)(P+3));
	break;
    }

    case callv:			/* meta call */
	(void) printf("callv %d,%d,0x%08lx\n", oprnd(1), oprnd(2), *(fourBytes *)(P+3));
	break;

    case allocate:		/* allocate a call record and local environment */
	(void) printf("allocate\n");
	break;

    case deallocate:		/* deallocate the call record */
	(void) printf("deallocate\n");
	break;

    case tryme:			/* try a clause */
	(void) printf("tryme %d,$%ld\n", oprnd(1), adr(2));
	break;

    case retryme:		/* try subsequent clause */
	(void) printf("retryme $%ld\n", adr(1));
	break;

    case trustme:		/* last clause to try */
	(void) printf("trustme\n");
	break;

    case try:			/* try a clause */
	(void) printf("try %d,$%ld\n", oprnd(1), adr(2));
	break;

    case retry:			/* try a subsequent clause */
	(void) printf("retry $%ld\n", adr(1));
	break;

    case trust:			/* last clause to try */
	(void) printf("trust $%ld\n", adr(1));
	break;

    case fail:			/* fail execution */
	(void) printf("fail\n");
	break;

    case arg_switch:		/* switch on argument register */
	(void) printf("arg_switch A[%d],$%ld,$%ld,$%ld,$%ld,$%ld,$%ld,$%ld\n", oprnd(1), 
		adr(2), adr(3), adr(4), adr(5), adr(6), adr(7), adr(8));
	break;

    case indexc: {		/* index constant symbol access */
	codepo	last;
	(void) printf("indexc A[%d],", oprnd(1));
	P += 2;
	(void) printf("$%ld\n", adr(2));	/* default label */
	last = rel(P);
	for (P+=4; P<last; P+=4) {	/* length of each index entry is 4 words */
	    if (indexoffset(P)) {
		(void) printf("\t    ");
		printName((symbpo)(P + lbl(1)), stdout);
		(void) printf(" -> $%ld\n", adr(2));
	    }
	}
	break;
    }

    case indexi: {		/* index integer access */
	codepo	last;
	(void) printf("indexi A[%d],", oprnd(1));
	P += 2;
	(void) printf("$%ld\n", adr(2));	/* default label */
	last = rel(P);
	for (P+=4; P<last; P+=4) 	/* length of each index entry is 4 words */
	    if (indexoffset(P))
		(void) printf("\t    %ld -> $%ld\n", indexkey(P), adr(2));
	break;
    }

    case indext: {		/* index on first argument of tuple */
	codepo	last;
	(void) printf("indext A[%d],$%ld,$%ld,$%ld\n", oprnd(1), adr(6), adr(2), adr(3));
	P += 4;
	last = rel(P);
	for (P+=4; P<last; P+=4) {	/* length of each index entry is 4 words */
	    if (indexoffset(P)) {
		(void) printf("\t    ");
		printName((symbpo)(P + lbl(1)), stdout);
		(void) printf("/%d -> $%ld\n", indexarity(P), adr(2));
	    }
	}
	break;
    }

	/* put instructions */

    case clr_y_var:                    /* initialises a global var in local */
        (void) printf("clr_y_var Y[%d],A[%d]\n", oprnd(1), oprnd(2));
        break;

    case put_x_var:		/* put a new variable in argument register and temporary */
	(void) printf("put_x_var A[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_y_var:		/* put a new variable in argument register and local */
	(void) printf("put_y_var Y[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_x_val:		/* copy argument register to argument register */
	(void) printf("put_x_val A[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_y_val:		/* copy contents of local into argument register */
	(void) printf("put_y_val Y[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_unsafe_y:		/* last occ. of an unsafe variable in arg reg. */
	(void) printf("put_unsafe_y Y[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_void:		/* put a void variable into argument register */
	(void) printf("put_void A[%d]\n", oprnd(1));
	break;

    case put_const:		/* put a constant into argument register */
	(void) printf("put_const ");
	printName((symbpo)(P + lbl(1)), stdout);
	(void) printf(",A[%d]\n", oprnd(2));
	break;

    case put_int:		/* put an integer into argument register */
	(void) printf("put_int %ld,A[%d]\n", *(fourBytes *)(P + 1), oprnd(3));
	break;

    case put_nil:		/* put nil into argument register */
	(void) printf("put_nil A[%d]\n", oprnd(1));
	break;

    case put_list:		/* put list pair into argument register */
	(void) printf("put_list A[%d]\n", oprnd(1));
	break;

    case put_tpl:		/* put a tuple into argument register */
	(void) printf("put_tpl %d,A[%d]\n", oprnd(1), oprnd(2));
	break;

    case put_slash:		/* store slashback in argument register */
	(void) printf("put_slash A[%d]\n", oprnd(1));
	break;

	/* get instructions */

    case get_x_var:		/* move argument register */
	(void) printf("get_x_var A[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case get_y_var:		/* move arg to local env */
	(void) printf("get_y_var A[%d],Y[%d]\n", oprnd(1), oprnd(2));
	break;

    case get_x_val:		/* unify subsequent occ. of x var. */
	(void) printf("get_x_val A[%d],A[%d]\n", oprnd(1), oprnd(2));
	break;

    case get_y_val:		/* subsequent occ. of arg in local */
	(void) printf("get_y_val A[%d],Y[%d]\n", oprnd(1), oprnd(2));
	break;

    case get_const:		/* unify arg register with a constant symbol */
	(void) printf("get_const A[%d],", oprnd(1));
	printName((symbpo)(P + lbl(2)), stdout);
	(void) printf("\n");
	break;

    case get_int:		/* unify arg register with an integer */
	(void) printf("get_int A[%d],%ld\n", oprnd(1), *(fourBytes *)(P + 2));
	break;

    case get_nil:		/* unify arg register with nil */
	(void) printf("get_nil A[%d]\n", oprnd(1));
	break;

    case get_list:		/* unify arg register with a list pair */
	(void) printf("get_list A[%d]\n", oprnd(1));
	break;

    case get_tpl:		/* unify arg register with a tuple(structure) */
	(void) printf("get_tpl A[%d],%d\n", oprnd(1), oprnd(2));
	break;

	/* unify instructions */

    case unify_void:		/* skip some terms */
	(void) printf("unify_void %d\n", oprnd(1));
	break;

    case unify_x_var:		/* new var in structure against arg register */
	(void) printf("unify_x_var A[%d]\n", oprnd(1));
	break;

    case unify_y_var:		/* new var in structure against local var */
	(void) printf("unify_y_var Y[%d]\n", oprnd(1));
	break;

    case unify_x_val:		/* unify var structure against arg register */
	(void) printf("unify_x_val A[%d]\n", oprnd(1));
	break;

    case unify_y_val:		/* unify var in structure against local var */
	(void) printf("unify_y_val Y[%d]\n", oprnd(1));
	break;

    case unify_loc_x_val:	/* unify local value against arg register */
	(void) printf("unify_loc_x_val A[%d]\n", oprnd(1));
	break;

    case unify_loc_y_val:	/* unify local value against local var */
	(void) printf("unify_loc_y_val Y[%d]\n", oprnd(1));
	break;

    case unify_const:		/* unify against constant symbol */
	(void) printf("unify_const ");
	printName((symbpo)(P + lbl(1)), stdout);
	(void) printf("\n");
	break;

    case unify_int:		/* unify against integer */
	(void) printf("unify_int %ld\n", *(fourBytes *)(P + 1));
	break;

    case unify_nil:		/* unify against nil */
	(void) printf("unify_nil\n");
	break;

    case unify_list:		/* unify against list */
	(void) printf("unify_list\n");
	break;

    case unify_tpl:		/* unify against tuple */
	(void) printf("unify_tpl %d\n", oprnd(1));
	break;

    case push_t:		/* push term pointer */
	(void) printf("push_t\n");
	break;

    case pop_t:			/* pop term pointer */
	(void) printf("pop_t\n");
	break;

    case cut:			/* trim choice points */
	(void) printf("cut\n");
	break;

    case gc:			/* garbage collection */
	(void) printf("gc %d,%d\n", oprnd(1), oprnd(2));
	break;

    case gc0:			/* set up shadow HMAX */
	(void) printf("gc0\n");
	break;

    case brkinstr: {		/* break point instruction */
	opcode br_code = get_brkcode(PC(P));
	codepo PP = P;
	(void) printf("*** breakpoint ***     ");
	*P = br_code;
	P = oneInstruction(P);
	*PP = brkinstr;
	break;
    }

    case escape:		/* esacape into service functions */
	(void) printf("escape %d\n", oprnd(1));
	break;

    case data:			/* data for inline jsr */
	(void) printf("DATA %d,0x%08lx\n", oprnd(1), *(fourBytes *)(P + 2));
	break;

    case f_push_t:		/* first push term pointer */
	(void) printf("f_push_t\n");
	break;

    case dynamic:
	(void) printf("dynamic %d,$%ld,(%lu,%lu]\n", *(P+1),
		PC(P)+nextcl(P), birth(P), death(P));
	break;

    case fdynamic:
	(void) printf("fdynamic %d,$%ld,(%lu,%lu]\n", *(P+1),
		PC(P)+nextcl(P), birth(P), death(P));
	break;

	/* set instructions */

    case set_void:		/* skip some terms */
	(void) printf("set_void %d\n", oprnd(1));
	break;

    case set_x_var:		/* new var in structure against arg register */
	(void) printf("set_x_var A[%d]\n", oprnd(1));
	break;

    case set_y_var:		/* new var in structure against local var */
	(void) printf("set_y_var Y[%d]\n", oprnd(1));
	break;

    case set_x_val:		/* unify var structure against arg register */
	(void) printf("set_x_val A[%d]\n", oprnd(1));
	break;

    case set_y_val:		/* unify var in structure against local var */
	(void) printf("set_y_val Y[%d]\n", oprnd(1));
	break;

    case set_loc_x_val:		/* unify local value against arg register */
	(void) printf("set_loc_x_val A[%d]\n", oprnd(1));
	break;

    case set_loc_y_val:		/* unify local value against local var */
	(void) printf("set_loc_y_val Y[%d]\n", oprnd(1));
	break;

    case set_const:		/* unify against constant symbol */
	(void) printf("set_const ");
	printName((symbpo)(P + lbl(1)), stdout);
	(void) printf("\n");
	break;

    case set_int:		/* unify against integer */
	(void) printf("set_int %ld\n", *(fourBytes *)(P + 1));
	break;

    case set_nil:		/* unify against nil */
	(void) printf("set_nil\n");
	break;

    case set_list:		/* unify against list */
	(void) printf("set_list\n");
	break;

    case set_tpl:		/* unify against tuple */
	(void) printf("set_tpl %d\n", oprnd(1));
	break;

    case put_float:		/* put a float into argument register */
	(void) printf("put_float %.16lg,A[%d]\n", *(FLOAT *)(P + 1), oprnd(5));
	break;

    case get_float:		/* unify arg register with a float */
	(void) printf("get_float A[%d],%.16lg\n", oprnd(1), *(FLOAT *)(P + 2));
	break;

    case unify_float:		/* unify against a float */
	(void) printf("unify_float %.16lg\n", *(FLOAT *)(P + 1));
	break;

    case set_float:		/*  unify against a float */
	(void) printf("set_float %.16lg\n", *(FLOAT *)(P + 1));
	break;

    case go_to:			/*  go to error label */
	(void) printf("go_to $%ld\n", adr(1));
	break;

    case set_err:		/*  set error label */
	(void) printf("set_err $%ld\n", adr(1));
	break;

    case clr_err:		/*  clear error */
	(void) printf("clr_err\n");
	break;
    case nop:
	(void) printf("nop\n");
	break;
    case nop2:
	(void) printf("nop2\n");
	break;
    case nop3:
	(void) printf("nop3\n");
	break;
    case gc_allocate:		/* g/c + allocate */
	(void) printf("gc_allocate %d,%d\n", oprnd(1), oprnd(2));
	break;


#include "debug_low.c"

    default:			/* illegal instruction */
	(void) printf("illegal instruction\n");
	return(P);
	break;

    }

    return(P + instrLen[instr]);
}

/*
 * diss - disassembles instructions from an address
 */
codepo
diss(P, N)
    codepo	P;
    int		N;
{
    codepo tempP;
    while (N--) {
	(void) printf("%5ld : ", PC(P));
	if ((tempP  = oneInstruction(P)) > P)
	    P = tempP;
	else break;
    }
    return(P);
}

/*------------------------------------------------------------*/

extern cell A[REGISTERS];

safeptr(p, TH)
    cellpo	p;
    threadpo	TH;
{
    return ((p >= (cellpo) TH && (!TH || p < TH->stacks + TH->TSZ)) ||
	    (p >= A && p < A + REGISTERS));
}

void
printName(symb, fp)
symbpo	symb;
FILE	*fp;
{
    strpo	name;
    utwoBytes	len;
    int		quoting;

    if (safeptr((cellpo) symb, TH)
		|| (symb >= (symbpo)boot && symb < (symbpo)0x1000000)
		/* globally defined system constants */
		|| (symb >= (symbpo)symbolHeap && symb < (symbpo)boot)) {
	name = symbname(symb);
	len = min(symblngth(symb), MAXNAME);

	/* determine whether quoting is needed */
	quoting = 1;
	if (chtype(*name) == lower) {
	    quoting = 0;
	    while (--len && !quoting)
		switch (chtype(*++name)) {
		    case lower:
		    case upper:
		    case number:
			break;
		    default:
			quoting = 1;
		}
	}

	name = symbname(symb);
	len = min(symblngth(symb), MAXNAME);

	if (quoting)
	    (void) putc('\'', fp);
	    
	while (len--) {
	    (void) putc(*name, fp);
	    name++;
	}
	if (symblngth(symb) > MAXNAME)
	    (void) fprintf(fp, " ... ");

	if (quoting)
	    (void) putc('\'', fp);
    }
    else
	(void) fprintf(fp, "*** invalid constant ***");
}

/* print a cell for debugging purposes */
auxprtcell(c, TH, depth, fp)
    cellpo	c;
    threadpo	TH;
    int		depth;
    FILE	*fp;
{
    int		length = maxdepth, markflag = 0, size = 1;

if (depth < 1) 			/* deep structure, likely to be a loop */
    (void) fprintf(fp, " ... ");
else {

    if (marked(c)) {
	(void) fprintf(fp, "M");
	markflag = 1;
    }
    if (first(c))
	(void) fprintf(fp, "H");

    switch (pureTag(c)) {
    case var_ref:
	if (!vl(c))
	    (void) fprintf(fp, "_");
	else if (c == (cellpo) vl(c))
	    (void) fprintf(fp, "_%lx", normal(c));
	else if (safeptr(c = (cellpo) vl(c), TH)) {
	    if (verbose && c != (cellpo) vl(c))
		(void) fprintf(fp, "->%lx:", normal(c));
	    (void) auxprtcell(c, TH, depth - 1, fp);
	}
	else
		(void) fprintf(fp, "$$$");
	break;

    case int_ref:
	(void) fprintf(fp, "%ld", intvl(c));
	break;

    case float_ref: {
	cellpo t = (cellpo) vl(c);
	if (verbose)
	    (void) fprintf(fp, "F/0x%lx:", normal(t));
	if (markflag && t >= GC_B->H && t < H) {	/* in new heap */
	    if (marked(t))
		(void) fprintf(fp, "(M)");
	    t++;
	}
	/* assuming 24-bit addresses */
	if (safeptr(t, TH) || (t >= (cellpo)boot && t < (cellpo)0x1000000)) {
	    (void) fprintf(fp, "%.16lg", floatvl((cellpo)&t));
	}
	else
		(void) fprintf(fp, "$$$");
	break;
    }

    case guard_ref: {
	fourBytes len;
	cellpo end_marker;
	len = intvl(c);
	if (len < 1 || len > 20) {	/* check for plausibility ! */
	    (void) fprintf(fp, "G:%ld",len);
	    break;
	}
	end_marker = c+len;
	if (intvl(end_marker) == -len) {
	    c++;
	    if (len == (FLOATSIZE+1) &&
		    symblngth(c) > symMaxLen(FLOATSIZE, sizeof(cell))) {
		(void) fprintf(fp, "G%d:%.16lg", len, floatvl((cellpo)&c));
		size = len;
		break;
	    }
	    (void) fprintf(fp, "G%d:", len);
	    printName((symbpo)c, fp);
	    size = len;
	}
	else
		(void) fprintf(fp, "G:%ld",len);
	break;
    }

    case symb_ref:
	c = (cellpo) vl(c);
	if (verbose)
	    (void) fprintf(fp, "C/0x%lx:", normal(c));
	if (markflag && c >= GC_B->H && c < H) {	/* in new heap */
	    if (marked(c))
		(void) fprintf(fp, "(M)");
	    c++;
	}
	/* assuming 24-bit addresses */
	if (safeptr(c, TH)
		|| (c >= (cellpo)boot && c < (cellpo)0x1000000)
		/* globally defined system constants */
		|| (c >= symbolHeap && c < (cellpo)boot))
	    printName((symbpo)c, fp);
	else
		(void) fprintf(fp, "$$$");
	break;

    case nil_ref:
	(void) fprintf(fp, "[]");
	break;

    case list_ref:
	if (safeptr(c = (cellpo) vl(c), TH)) {
	    if (verbose)
		(void) fprintf(fp, "L/0x%lx:", normal(c));
	    (void) putc('[', fp);
	    (void) auxprtcell(c, TH, depth - 1, fp);
	    c++;
	    {	/* special delnk which masks g/c bits */
	    register cellpo val;
	    while (IsVar(c) && (val=(cellpo)vl(c)) != (c))
		c = val;
	    }
	    while (pureTag(c) == list_ref && --length) {
		(void) fprintf(fp, ",");
		(void) auxprtcell(c = (cellpo) vl(c), TH, depth - 1, fp);
		c++;
		{	/* special delnk which masks g/c bits */
		register cellpo val;
		while (IsVar(c) && (val=(cellpo)vl(c)) != (c))
		c = val;
		}
	    }
	    if (!length)
		(void) fprintf(fp, " ... ");
	    else if (pureTag(c) != nil_ref) {
		(void) fprintf(fp, "|");
		(void) auxprtcell(c, TH, depth - 1, fp);
	    }
	    (void) putc(']', fp);
	}
	else
		(void) fprintf(fp, "$$$");
	break;

    case tpl_ref:
	{
	    fourBytes count;
	    if (safeptr(c = (cellpo) vl(c), TH)) {
		if (verbose)
		    (void) fprintf(fp, "T/0x%lx:", normal(c));
	 	count = intvl(c);
		c++;
		(void) auxprtcell(c++, TH, depth - 1, fp);	/* print functor */
		(void) putc('(', fp);
		count--;
		while (count-- && --length) {
		    (void) auxprtcell(c++, TH, depth - 1, fp);
		    if (count)
			(void) fprintf(fp, ",");
		}
		if (!length)
		    (void) fprintf(fp, " ... ");
		(void) putc(')', fp);
	    }
	    else
		(void) fprintf(fp, "$$$");
	    break;
	}
    default:
	(void) fprintf(fp, "???");
    }
}
    return(size);
}

prtcell(c, TH)
    cellpo	c;
    threadpo	TH;
{
    if (safeptr(c, TH))
	return(auxprtcell(c, TH, maxdepth, stdout));
    else {
	(void) fprintf(stdout, "$$$");
	return(1);
    }
}

/*------------------------------------------------------------*/

/* dmpA dumps the A-registers */
dmpA(A, TH)
    cellpo	A;
    threadpo	TH;
{
    int		i;
    cellpo	p = &A[0];

    for (i = 0; i < numRegs; i++) {
	(void) printf("A[%.2d]=", i);
	(void) prtcell(p++, TH);
	(void) printf((i & 03) == 03 ? "\n" : "\t");
    }
    if (i & 03)
	(void) printf("\n");
}

/* dmpreg dumps the main machine registers */
dmpreg(P)
    codepo	P;
{
    int		i;
    cellpo	p;

    (void) printf("P =");
    (void)diss(P, 1);
    (void) printf("CP=");
    if (CP)
	(void)diss(CP, 1);
    else
	(void) printf("undefined\n");
    (void) printf("H =0x%05lx, HB=0x%05lx, ", normal(H), normal(HB));
    (void) printf("HMAX=0x%05lx, S=0x%05lx:", normal(HMAX), normal(S));
    if (safeptr(S, TH))
	(void) prtcell(S, TH);
    else
	(void) printf("$$$");
    (void) printf("\n");
    (void) printf("E =0x%05lx, CE=0x%05lx, TR=0x%05lx\n",
	    normal(E), normal(CE), normal(TR));
    (void) printf("B =0x%05lx, SB=0x%05lx, CSB=0x%05lx, GC_B=0x%05lx, CATCH=0x%05lx\n",
	    normal(B), normal(SB), normal(CSB), normal(GC_B), normal(CATCH));

    /* display the A-registers */
    dmpA(A, TH);

    /* display the current environment */
    p = E->Y;
    if (safeptr((cellpo)E, TH)) {
	for (i = 0; i < numLocals; i++, p++) {
	    (void) printf("Y[%.2d]=", i);
	    if (currentLocals & bit[i])
		(void) prtcell(p, TH);
	    else (void) printf("???");
	    (void) printf((i & 03) == 03 ? "\n" : "\t");
	}
	if (i & 03)
	    (void) printf("\n");
    }
}

prtheap(H, TH, n)
    cellpo	H;
    threadpo	TH;
    fourBytes	n;
{
    fourBytes	i = normal(H) - n;
    cellpo	p = TH->stacks;
    int		size;

    i = max(0,i);
    while (i < normal(H)) {
	(void) printf("[%05lx] = ", i);
	size = prtcell(p+i, TH);
	(void) printf((i & 03) == 03 ? "\n" : "\t");
	i += size;
    }
    if (i & 03)
	(void) printf("\n");
}

prtenv(E, CP, TH, indent)
    envpo	E;
    codepo	CP;
    threadpo	TH;
    bool	indent;
{
    int		i, n=envsize(CP);
    cellpo 	p;

    (void) printf("\n");
    if (indent) (void) printf("\t");
    (void) printf("%lx: ", normal(p = &(E->Y[0])));
    for (i=0; i<n; i++, p++) {
	(void) printf("Y[%.2d]=", i);
	(void) prtcell(p, TH);
	(void) printf("   ");
    }
    (void) printf("\n");

    if (indent) (void) printf("\t");
    (void) printf("CP=%6ld, CE=0x%05lx, CSB=0x%05lx, HMAX=0x%05lx\n",
	    PC(E->CP), normal(E->CE), normal(E->CSB), normal(E->HMAX));
}

prtchoice(B, TH)
    choicepo	B;
    threadpo	TH;
{
    int		i, n=B->AX;
    cellpo	p;

    (void) printf("\n\t\t\tH=0x%05lx, TR=0x%05lx, E=0x%05lx, B=0x%05lx\n",
	    normal(B->H), normal(B->TR), normal(B->E), normal(B->B));
    (void) printf("\t\t\tP=%6ld, CP=%6ld, CSB=0x%05lx, HMAX=0x%05lx\n",
	    PC(B->P), PC(B->CP), normal(B->CSB), normal(B->HMAX));

    (void) printf("\t\t\t%lx: ", normal(p = (cellpo)B - n));
    for (i=0; i<n; i++, p++) {
	(void) printf("A[%.2d]=", i);
	(void) prtcell(p, TH);
	(void) printf("   ");
    }
    (void) printf("\n");
}

prtstack(B, E, CP, TH, depth)
    choicepo	B;
    envpo	E;
    codepo	CP;
    threadpo	TH;
    int		depth;
{
    envpo	B_E = (cellpo)B == TH->BLS ? (envpo)B : B->E;
    codepo	B_CP = B->CP;

    while ((E != (envpo)B || B_E != (envpo)B) && depth-- != 0) {
	/* stack not empty */
	if ((cellpo)B > (cellpo)E) {
	    if ((envpo)B > B_E) {
		prtchoice(B, TH);
		B_E = (cellpo)B == TH->BLS ? (envpo)B : B->E;
		B_CP = B->CP;
		B = B->B;
	    }
	    else {
		prtenv(B_E, B_CP, TH, TRUE);
		B_CP = B_E->CP;
		B_E = B_E->CE;
	    }
	}
	else {
	    if (E == B_E)
		B_E = (envpo)TH->BLS;
	    if (E > B_E) {
		prtenv(E, CP, TH , FALSE);
		CP = E->CP;
		E = E->CE;
	    }
	    else {
		prtenv(B_E, B_CP, TH, TRUE);
		B_CP = B_E->CP;
		B_E = B_E->CE;
	    }
	}
    }
}

prttrail(TR, TH, n)
    cellpo	TR;
    threadpo	TH;
    int		n;
{
    while (TR < TH->stacks+TH->TSZ && (n-- != 0)) {
	(void) printf("\t\t\t\t\t[%05lx]=0x%05lx: ", normal(TR), normal(vl(TR)));
	(void) prtcell((cellpo) vl(TR), TH);
	(void) printf("\n");
	TR++;
    }
}

prtdump(TH, from, to)
    threadpo	TH;
    fourBytes	from, to;
{
    fourBytes	i;
    cellpo	p;

    to = min(to, from+MAXPRTDUMP);
    for (i=from, p=TH->stacks+from; i<=to; i++, p++) {
	if (safeptr(p, TH)) {
	    (void) printf("[%05lx] = ", i);
	    (void) prtcell(p, TH);
	    (void) printf((i & 03) == 03 ? "\n" : "\t");
	}
	else (void) printf("\t[%05lx] not in thread\n", i);
    }
    if (i & 03)
	(void) printf("\n");
}

previous(count)
    int		count;
{
    int		where;
    fourBytes	 addr;

    where = tracePtr - count;
    if (where < 0)
	where += AUDITSIZE;

    while (count--) {
	addr = trace[where];
	if (addr < 0)
	    (void) printf("      : no instruction\n");
	else (void)diss(boot+addr, 1);
	if (++where == AUDITSIZE)
	    where = 0;
    }
}

help()
{
    (void) printf("  <CR>      - single step\n");
    (void) printf("  g         - Go until breakpoint reached\n");
    (void) printf("  b [addr]  - set Break point\n");
    (void) printf("  B <rel>   - set Break point on a relation\n");
    (void) printf("  c [addr]  - Clear break point\n");
    (void) printf("\n");
    (void) printf("  l [a [n]] - List/disassemble from address a, n instructions\n");
    (void) printf("  L <rel>   - List/disassemble a relation (pred/arity)\n");
    (void) printf("  r         - display Registers\n");
    (void) printf("  s [n]     - display top n records of local Stack\n");
    (void) printf("  t [n]     - display top n entries of Trail\n");
    (void) printf("  h [n]     - display top n cells of Heap (constructed term stack)\n");
    (void) printf("  m a1 a2   - dump Memory (heap cells) from address a1 to a2\n");
/*    (void) printf("  a x   - display x instructions from Audit\n");	*/
    (void) printf("  u         - list Undefined relations\n");
    (void) printf("  M         - output Memory map of predicates and addresses\n");
    (void) printf("\n");
    (void) printf("  d [level] - change Debug level\n");
    (void) printf("  D [depth] - change max display Depth of terms\n");
    (void) printf("  R [n]     - change #Regs and #locals displayed\n");
    (void) printf("  v         - toggle Verbose mode (show pointers)\n");
#ifdef THINK_C
    (void) printf("  C         - change console window to half-height\n");
#endif
    (void) printf("  ?         - display this help page\n");
    (void) printf("  q         - Quit\n");
}

/*------------------------------------------------------------*/



/* debugger provides interactive debugging facilities */
debugger(P)
    codepo	P;
{
    CHARTYPE	ch, input[80];
    codepo	list_addr = P;
    extern fourBytes collect_garbage();

    clearerr(stdin);		/* clear EOF, if any */
    (void) printf("\n");
    dmpreg(P);

    do {
	(void) printf("---------------------------------------------------------\n");
	(void) printf("[%d] %s>> ", debugLevel, (verbose) ? "verbose " : "");

	switch (ch = getchar()) {

	case EOF:
	case 'q':
	    icp_exit(0);

	case '?':
	    help();
	    break;

	case 'a':
	    {
		int count;
		if (sscanf(gets(input), "%d", &count) > 0)
		    previous(count);
		else previous(TRACELIST);
		continue;
	    }

	case 'b':
	    {
		fourBytes addr;
		short i = 0;
		if (sscanf(gets(input), "%ld", &addr) < 1)
		    addr = PC(P);
		while (i < numBreaks && brkpoint[i] != addr)
		    i++;
		if (i == BREAKPOINTS)
		    (void) printf("Too many breakpoints\n");
		else {
		    (void) printf("Setting breakpoint at ");
		    (void)diss(boot+addr, 1);
		    if (i == numBreaks) {
			numBreaks++;
			brkpoint[i] = addr;
			brkcode[i] = boot[addr];
			boot[addr] = brkinstr;
		    }
		}
		continue;
	    }

	case 'B':
	    {
		uchar		name[80];
		uchar		pred[84]; /* length must be 4 more than name */
		symbpo		sym = (symbpo)pred;
		strpo		ptr1, ptr2;
		int		arit;
		short		len = 0, hash = 0, i = 0;
		codepo		addr = NULL;
		fourBytes	address;
		dictionary	dict;

		switch (sscanf(gets(input), " %[^/]/%d", name, &arit)) {
		case 1:	arit = 0; break;
		case 2: break;
		default:
		    (void) printf("format is 'B <pred>/<arity>'\n");
		    continue;
		}

		ptr1 = name;
		ptr2 = symbname(sym);
		while (*ptr1) {
		    len++;
		    hash += *ptr1;
		    *ptr2++ = *ptr1++;
		}
		*ptr2++ = '\0';
		symblngth(sym) = len;
		symbhash(sym) = hash;
		if ((dict = find_entry(sym, arit)) && (addr = dict->addr) &&
			(dict->type == DYNAMIC)) {
		    codepo p = addr;
		    findFirst(p, C);
		    if (p==NULL)
			addr = NULL;
		}
		if (addr) {
		    address = PC((codepo)addr);
		    while (i < numBreaks && brkpoint[i] != address)
			i++;
		    if (i == BREAKPOINTS)
			(void) printf("Too many breakpoints\n");
		    else {
			(void) printf("Setting breakpoint at ");
			(void)diss(boot+address, 1);
			if (i == numBreaks) {
			    numBreaks++;
			    brkpoint[i] = address;
			    brkcode[i] = boot[address];
			    boot[address] = brkinstr;
			}
		    }
		}
		else (void) printf("%s/%d not defined\n", name, arit);
		continue;
	    }

	case 'c':
	    {
		fourBytes addr;
		short i = 0;
		if (sscanf(gets(input), "%ld", &addr) < 1)
		    addr = PC(P);
		while (i < numBreaks && brkpoint[i] != addr)
		    i++;
		if (i == numBreaks)
		    (void) printf("No breakpoint at %ld\n", addr);
		else {
		    boot[brkpoint[i]] = brkcode[i];
		    (void) printf("Clearing breakpoint at ");
		    (void)diss(boot+brkpoint[i], 1);
		    numBreaks--;
		    brkpoint[i] = brkpoint[numBreaks];
		    brkcode[i] = brkcode[numBreaks];
		}
		continue;
	    }

	case 'd':
	    {
		int newLevel;
		if (sscanf(gets(input), "%d", &newLevel) < 1)
		    (void) printf("Debug level is %d\n", debugLevel);
		else if  (newLevel <= MAXDUMP && newLevel >= 0) {
		    debugLevel = newLevel;
		    (void) printf("Debug level set to %d\n", debugLevel);
		    oldLevel = debugLevel;
		}
		else (void) printf("Debug level must be in range 0 to %d\n", MAXDUMP);
		continue;
	    }

	case 'g':
	    oldLevel = debugLevel;
	    debugLevel = 0;
	    break;

	case 'h':
	    {
		fourBytes n;
		if (sscanf(gets(input), "%ld", &n) < 1)
		    n = MAXPRTDUMP;
		prtheap(H, TH, n);
		continue;
	    }

	case 'l':
	    {
		fourBytes addr;
		int count;
		switch (sscanf(gets(input), "%ld %d", &addr, &count)) {
		case 1:
		    list_addr = diss(boot+addr, listingCount);
		    break;
		case 2:
		    list_addr = diss(boot+addr, count);
		    if (count > 0)
			listingCount = count;
		    break;
		default:
		    list_addr = diss(list_addr, listingCount);
		    break;
		}
		continue;
	    }

	case 'L':
	    {
		uchar		name[80];
		uchar		pred[84]; /* length must be 4 more than name */
		symbpo		sym = (symbpo)pred;
		strpo		ptr1, ptr2;
		int		arit;
		short		len = 0, hash = 0;
		codepo		addr = NULL;
		dictionary	dict;

		switch (sscanf(gets(input), " %[^/]/%d", name, &arit)) {
		case 1:	arit = 0; break;
		case 2: break;
		default:
		    (void) printf("format is 'L <pred>/<arity>'\n");
		    continue;
		}

		ptr1 = name;
		ptr2 = symbname(sym);
		while (*ptr1) {
		    len++;
		    hash += *ptr1;
		    *ptr2++ = *ptr1++;
		}
		*ptr2++ = '\0';
		symblngth(sym) = len;
		symbhash(sym) = hash;
		if ((dict = find_entry(sym, arit)) && (addr = dict->addr) &&
			(dict->type == DYNAMIC)) {
		    codepo p = addr;
		    findFirst(p, C);
		    if (p==NULL)
			addr = NULL;
		}
		if (addr)
		    list_addr = diss(addr, listingCount);
		else (void) printf("%s/%d not defined\n", name, arit);
		continue;
	    }

	case 'm':
	    {
		fourBytes from, to;
		if (sscanf(gets(input), "%lx %lx", &from, &to) > 1)
		    prtdump(TH, from, to);
		else (void) printf("format is 'm <from> <to>'\n");
		continue;
	    }

	case 'r':
	    dmpreg(P);
	    break;

	case 's':
	    {
		int depth;
		if (sscanf(gets(input), "%d", &depth) < 1)
		    depth = -1;
		prtstack(B, E, CP, TH, depth);
		continue;
	    }

	case 't':
	    {
		int depth;
		if (sscanf(gets(input), "%d", &depth) < 1)
		    depth = -1;
		prttrail(TR, TH, depth);
		continue;
	    }

	case 'u':
	    noref();
	    break;

	case 'v':
	    verbose = !verbose;
	    break;

	case 'D':
	    {
		int depth;
		if (sscanf(gets(input), "%d", &depth) > 0)
		    maxdepth = depth;
		else maxdepth = MAXDEPTH;
		dmpreg(P);
		continue;
	    }

	case 'R':
	    {
		int length;
		if (sscanf(gets(input), "%d", &length) > 0) {
		    numRegs = numLocals = length;
		    currentLocals = 0xFFFFFFFFL;
		}
		else {
		    numRegs = DISPLREG;
		    numLocals = 0;
		    currentLocals = 0L;
		}
		dmpreg(P);
		continue;
	    }

	case 'M':
	    refs();
	    break;

	case 'G':
	    (void)collect_garbage(0);
	    break;

#ifdef THINK_C
	case 'C':
	    console_options.top = 38;
	    console_options.nrows = 36 - console_options.nrows;
	    (void) freopenc(NULL, stdin);
	    (void) freopenc(stdin, stdout);
	    (void) freopenc(stdin, stderr);
	    continue;
#endif

	case '\n':
	    (void)ungetc('\n', stdin);
	    break;
	}

	while (getchar() != '\n')
	    ;

    } while (ch != '\n' && ch != 'g');
}
