/* Copyright (C) 1992 Imperial College */
/*
 * Term encoding/decoding functions
 */
#include "primitives.h"
extern symbpo	eof_sym;

/* maximum number of variables/shared terms */
#define MAXSHARE	0x80

/* tags for transmitting encoded terms */
#define S_VAR	0x00
#define VAR	0x10
#define S_INT	0x20
#define INT	0x30
#define FLT	0x40
#define S_SYMB	0x50
#define SYMB	0x60
#define NIL	0x70
#define LIST	0x80
#define S_TPL	0x90
#define TPL	0xA0
#define S_TAG	0xB0
#define TAG	0xC0
#define S_REF	0xD0
#define REF	0xE0

#define TAGMASK	0xF0
#define VALMASK	0x0F

#define emit(N)				(*charout)((char)(N))



/* table of variables/shared terms */
static cell shared[MAXSHARE];
static int current;

void
init_encode()
{
    current = 0;
}

static int
search_vars(addr)
cellpo	addr;
{
    register int i;
    for (i=1; i<=current; i++)
	if (IsVar(shared+i) && (cellpo) vl(shared+i) == addr)
	    return(i);

    /* not found */
    return(0);
}

static int
add_var(addr)
cellpo	addr;
{
    if (current++ == MAXSHARE)
	longjmp(icp_interrupt, 508);
    mklnk(shared+current, addr);
    return(current);
}

static
new_var(term, pos)
cellpo	term;
int	pos;
{
    mkunb(H);
    mklnk(term, H);
    mklnk(shared+pos, H);
    H++;
}

void
encoded_write_term(term)
cellpo	term;
{
    delnk(term);

    switch (tg(term)) {

	case int_ref: {
	    fourBytes num = intvl(term);
	    if (num < 0x10 && num >= 0)
		emit(S_INT | (char)num);
	    else {
		emit(INT);
		emit(num >> 030);
		emit(num >> 020);
		emit(num >> 010);
		emit(num);
	    }
	    break;
	}

	case nil_ref:
	    emit(NIL);
	    break;

	case var_ref: {
	    cellpo addr = (cellpo) vl(term);
	    int ref;
	    if (ref = search_vars(addr)) {
		if ((fourBytes)ref < 0x10L)
		      emit(S_REF | (char)ref);
		else {
		    emit(REF);
		    emit(ref);
		}
	    }
	    else {
		ref = add_var(addr);
		if ((fourBytes)ref < 0x10L)
		      emit(S_VAR | (char)ref);
		else {
		    emit(VAR);
		    emit(ref);
		}
	    }
	    break;
	}

	case symb_ref: {
	    symbpo sym = symbvl(term);
	    utwoBytes len = symblngth(sym);
	    register strpo name = symbname(sym);
	    if (len < 0x10)
		emit(S_SYMB | (char)len);
	    else {
		emit(SYMB);
		emit(len >> 010);
		emit(len);
	    }
	    /* N.B. Assumes one-byte characters ! */
	    while (len--)
		emit(*name++);
	    break;
	}

	case list_ref:
	    emit(LIST);
	    encoded_write_term(hd(term));
	    encoded_write_term(tl(term));
	    break;

	case tpl_ref: {
	    fourBytes i, ar = arity(term);
	    if ((fourBytes)ar < 0x10L)
	        emit(S_TPL | (char)ar);
	    else {
		emit(TPL);
		emit(ar >> 010);
		emit(ar);
	    }
	    for (i=0; i<ar; i++)
		encoded_write_term(arg(term,i));
	    break;
	}

	case float_ref: {
	    FLOAT f = floatvl(term);
	    strpo c = (strpo) &f;
	    emit(FLT);
	    emit(c[0]);
	    emit(c[1]);
	    emit(c[2]);
	    emit(c[3]);
	    emit(c[4]);
	    emit(c[5]);
	    emit(c[6]);
	    emit(c[7]);
	    break;
	}
    }
}

bool
pr_write_term()
{
    init_encode();
    encoded_write_term(&A[1]);
    return(SUCCEED);
}

bool encoded_read_term(term)
cellpo	term;
{
    CHARTYPE ch = (*charin)();

    if (ch == EOF) {
	if (errno)
		return(FAIL);
	mksymb(term, eof_sym);
	return(SUCCEED);
    }

    switch (ch & TAGMASK) {

	case S_INT:
	    mkint(term, ch & VALMASK);
	    break;

	case INT: {
	    fourBytes num = (*charin)() & 0xFF;
	    num = (num << 010) | ((*charin)() & 0xFF);
	    num = (num << 010) | ((*charin)() & 0xFF);
	    num = (num << 010) | ((*charin)() & 0xFF);
	    mkint1(term, num);
	    break;
	}

	case NIL:
	    mknil(term);
	    break;

	case S_VAR:
	    new_var(term, (int)(ch & VALMASK));
	    break;

	case VAR:
	    new_var(term, (*charin)());
	    break;

	case S_SYMB: {
	    utwoBytes len;
	    twoBytes hash = 0;
	    register strpo conpo;
	    symbpo symb;

	    len = (utwoBytes)(ch & VALMASK);
	    alloc_symb_struct(term, len);
	    symb = symbvl(term);
	    symblngth(symb) = len;
	    conpo = symbname(symb);
	    while (len--) {
		*conpo = (*charin)();
		hash += *conpo++;
	    }
	    *conpo = '\0';
	    symbhash(symb) = hash;
	    break;
	}

	case SYMB: {
	    utwoBytes len;
	    twoBytes hash = 0;
	    register strpo conpo;
	    symbpo symb;

	    len = (*charin)();
	    len = (len << 010) | ((*charin)() & 0xFF);
	    alloc_symb_struct(term, len);
	    symb = symbvl(term);
	    symblngth(symb) = len;
	    conpo = symbname(symb);
	    while (len--) {
		*conpo = (*charin)();
		hash += *conpo++;
	    }
	    *conpo = '\0';
	    symbhash(symb) = hash;
	    break;
	}

	case LIST:
	    alloc_list(term, term);
	    if (encoded_read_term(term++) == FAIL)
		return(FAIL);
	    if (encoded_read_term(term) == FAIL)
		return(FAIL);
	    break;

	case S_TPL: {
	    fourBytes ar = ch & VALMASK;
	    alloc_tpl(term, term, ar);
	    while (ar--)
		if (encoded_read_term(term++) == FAIL)
			return(FAIL);
	    break;
	}

	case TPL: {
	    fourBytes ar = (*charin)();
	    ar = (ar << 010) | ((*charin)() & 0xFF);
	    alloc_tpl(term, term, ar);
	    while (ar--)
		if (encoded_read_term(term++) == FAIL)
			return(FAIL);
	    break;
	}

	case S_TAG: {
	    int pos = ch & VALMASK;
	    if (encoded_read_term(term) == FAIL)
		return(FAIL);
	    *(shared+pos) = *term;
	    break;
	}

	case TAG: {
	    int pos = (*charin)();
	    if (encoded_read_term(term) == FAIL)
		return(FAIL);
	    *(shared+pos) = *term;
	    break;
	}

	case S_REF:
	    *term = *(shared+(ch & VALMASK));
	    break;

	case REF:
	    *term = *(shared+(*charin)());
	    break;

	case FLT: {
	    uchar c[8];
	    FLOAT *flt = (FLOAT *)c;
	    int i;
	    for (i=0; i<8; i++)
		c[i] = (uchar) (*charin)();
	    alloc_float(term, *flt);
	    break;
	}

	default:
	    (void) fprintf(stderr,"unknown tag in encoded_read_term\n");
    }
    return(SUCCEED);
}

bool
pr_read_term()
{
    register
    cellpo	t = &A[1];

    delnk(t);
    mkreset(t);
    return(encoded_read_term(t));
}

char *
encode_prop(term, ptr)
cellpo	term;
char	*ptr;
{
    delnk(term);

    switch (tg(term)) {

	case int_ref: {
	    fourBytes num = intvl(term);
	    if (num < 0x10 && num >= 0)
		*ptr++ = S_INT | (char)num;
	    else {
		*ptr++ = INT;
		*ptr++ = num >> 030;
		*ptr++ = num >> 020;
		*ptr++ = num >> 010;
		*ptr++ = num;
	    }
	    break;
	}

	case nil_ref:
	    *ptr++ = NIL;
	    break;

	case var_ref: {
	    cellpo addr = (cellpo) vl(term);
	    int ref;
	    if (ref = search_vars(addr)) {
		if ((fourBytes)ref < 0x10L)
		      *ptr++ = S_REF | (char)ref;
		else {
		    *ptr++ = REF;
		    *ptr++ = ref;
		}
	    }
	    else {
		ref = add_var(addr);
		if ((fourBytes)ref < 0x10L)
		      *ptr++ = S_VAR | (char)ref;
		else {
		    *ptr++ = VAR;
		    *ptr++ = ref;
		}
	    }
	    break;
	}

	case symb_ref: {
	    symbpo sym = symbvl(term);
	    utwoBytes len = symblngth(sym);
	    register strpo name = symbname(sym);
	    if (len < 0x10)
		*ptr++ = S_SYMB | (char)len;
	    else {
		*ptr++ = SYMB;
		*ptr++ = len >> 010;
		*ptr++ = len;
	    }
	    /* N.B. Assumes one-byte characters ! */
	    while (len--)
		*ptr++ = *name++;
	    break;
	}

	case list_ref:
	    *ptr++ = LIST;
	    ptr = encode_prop(hd(term), ptr);
	    ptr = encode_prop(tl(term), ptr);
	    break;

	case tpl_ref: {
	    fourBytes i, ar = arity(term);
	    if ((fourBytes)ar < 0x10L)
	        *ptr++ = S_TPL | (char)ar;
	    else {
		*ptr++ = TPL;
		*ptr++ = ar >> 010;
		*ptr++ = ar;
	    }
	    for (i=0; i<ar; i++)
		ptr = encode_prop(arg(term,i), ptr);
	    break;
	}

	case float_ref: {
	    FLOAT f = floatvl(term);
	    strpo c = (strpo) &f;
	    *ptr++ = FLT;
	    *ptr++ = c[0];
	    *ptr++ = c[1];
	    *ptr++ = c[2];
	    *ptr++ = c[3];
	    *ptr++ = c[4];
	    *ptr++ = c[5];
	    *ptr++ = c[6];
	    *ptr++ = c[7];
	    break;
	}
    }

    return(ptr);
}

strpo
encode_str_prop(values, ptr)
strpo *values, ptr;
{
	int len;
	strpo name, start;

	while (name = *values) {
		len = 0;
		*ptr++ = LIST;
		*ptr++ = SYMB;
		start = ptr;
		ptr += 2;
		while (*name) {
			*ptr++ = *name++;
			len++;
		}
		*start++ = len >> 010;
		*start++ = len;
		values++;
	}
	*ptr++ = NIL;
	return(ptr);
}

strpo
decode_prop(term, ptr)
cellpo	term;
strpo	ptr;
{
    uchar ch = *ptr++;

    switch (ch & TAGMASK) {

	case S_INT:
	    mkint(term, ch & VALMASK);
	    break;

	case INT: {
	    fourBytes num = *ptr++ & 0xFF;
	    num = (num << 010) | (*ptr++ & 0xFF);
	    num = (num << 010) | (*ptr++ & 0xFF);
	    num = (num << 010) | (*ptr++ & 0xFF);
	    mkint1(term, num);
	    break;
	}

	case NIL:
	    mknil(term);
	    break;

	case S_VAR:
	    new_var(term, (int)(ch & VALMASK));
	    break;

	case VAR:
	    new_var(term, *ptr++);
	    break;

	case S_SYMB: {
	    utwoBytes len;
	    twoBytes hash = 0;
	    register strpo conpo;
	    symbpo symb;

	    len = (utwoBytes)(ch & VALMASK);
	    alloc_symb_struct(term, len);
	    symb = symbvl(term);
	    symblngth(symb) = len;
	    conpo = symbname(symb);
	    while (len--) {
		*conpo = *ptr++;
		hash += *conpo++;
	    }
	    *conpo = '\0';
	    symbhash(symb) = hash;
	    break;
	}

	case SYMB: {
	    utwoBytes len;
	    twoBytes hash = 0;
	    register strpo conpo;
	    symbpo symb;

	    len = *ptr++;
	    len = (len << 010) | (*ptr++ & 0xFF);
	    alloc_symb_struct(term, len);
	    symb = symbvl(term);
	    symblngth(symb) = len;
	    conpo = symbname(symb);
	    while (len--) {
		*conpo = *ptr++;
		hash += *conpo++;
	    }
	    *conpo = '\0';
	    symbhash(symb) = hash;
	    break;
	}

	case LIST:
	    alloc_list(term, term);
	    ptr = decode_prop(term++, ptr);
	    ptr = decode_prop(term, ptr);
	    break;

	case S_TPL: {
	    fourBytes ar = ch & VALMASK;
	    alloc_tpl(term, term, ar);
	    while (ar--)
		ptr = decode_prop(term++, ptr);
	    break;
	}

	case TPL: {
	    fourBytes ar = *ptr++;
	    ar = (ar << 010) | (*ptr++ & 0xFF);
	    alloc_tpl(term, term, ar);
	    while (ar--)
		ptr = decode_prop(term++, ptr);
	    break;
	}

	case S_TAG: {
	    int pos = ch & VALMASK;
	    ptr = decode_prop(term, ptr);
	    *(shared+pos) = *term;
	    break;
	}

	case TAG: {
	    int pos = *ptr++;
	    ptr = decode_prop(term, ptr);
	    *(shared+pos) = *term;
	    break;
	}

	case S_REF:
	    *term = *(shared+(ch & VALMASK));
	    break;

	case REF:
	    *term = *(shared+*ptr++);
	    break;

	case FLT: {
	    uchar c[8];
	    FLOAT *flt = (FLOAT *)c;
	    int i;
	    for (i=0; i<8; i++)
		c[i] = *ptr++;
	    alloc_float(term, *flt);
	    break;
	}

	default:
	    (void) fprintf(stderr,"unknown tag in decode_prop\n");
    }
    return(ptr);
}

#define STR_PROP_LEN 255
strpo *
decode_str_prop(orig)
strpo	orig;
{
	strpo ptr, conpo, *values, *start;
	int len = 0, done;
	utwoBytes s_len;

	ptr = orig;
	if ((*ptr & TAGMASK) == SYMB) {
		ptr++;
		start = (strpo *) malloc(2 * sizeof(strpo));
		s_len = *ptr++;
		s_len = (s_len << 010) | (*ptr++ & 0xFF);
		conpo = (strpo)malloc(s_len+1);
		*start = conpo;
		while (s_len--)
			*conpo++ = *ptr++;
		*conpo = '\0';
		*(start+1) = NULL;
		return(start);
	}

	/* its a list so first calculate length */
	ptr = orig;
	for (len=0,done=0; !done; ) {
		switch (*ptr++ & TAGMASK) {
			case NIL:
				done = 1;
				break;
			case LIST:
				if ((*ptr++ & TAGMASK) == SYMB) {
					s_len = *ptr++;
					s_len = (s_len << 010)|(*ptr++ & 0xFF);
					ptr += s_len;
					len++;
					break;
				}
			default:
				(void) fprintf(stderr,"bad tag in decode_str_prop\n");
				return(NULL);
				break;
		}
	}

	/* now do it */
	start = (strpo *) malloc((len+1) * sizeof(strpo));
	values = start;
	ptr = orig;
	while (TRUE) {
		switch (*ptr++ & TAGMASK) {
			case NIL:
				*values = NULL;
				return(start);
				break;
			case LIST:
				if ((*ptr++ & TAGMASK) == SYMB) {
					s_len = *ptr++;
					s_len = (s_len << 010)|(*ptr++ & 0xFF);
					conpo = (strpo)malloc(s_len+1);
					*values++ = conpo;
					while (s_len--)
						*conpo++ = *ptr++;
					*conpo = '\0';
					break;
				}
		}
	}
	return(start);
}

bool
pr_encoded_write(reg, funct)
cellpo reg;
bool (*funct)();
{
	bool (*tmp_charout)();

	tmp_charout = charout;
	charout = funct;
	init_encode();
	encoded_write_term(reg);
	charout = tmp_charout;
	return(SUCCEED);
}

bool
pr_encoded_read(reg, funct)
cellpo reg;
CHARTYPE (*funct)();
{
	CHARTYPE (*tmp_charin)();
	int ret;

	tmp_charin = charin;
	charin = funct;
	ret = encoded_read_term(reg);
	charin = tmp_charin;
	return(ret);
}
