/* Copyright (C) 1992 Imperial College */
/*--------------------------------------------------------------*
 *								*
 *		Input and Output Primitives			*
 *								*
 *--------------------------------------------------------------*/

/* 
	27/6/90   dac
	Made pipes a new type.

	22/5/90   dac
	Major rewrite to integrate streams and ram files.

	19/3/90   pjs
	added error handling
*/

#include <fcntl.h>
#include <vfork.h>
#include <sys/types.h>
#ifdef GNUDOS
#include <std.h>
#include <sys/stat.h>
#else
#include <sys/ioctl.h>
#endif
#include "primitives.h"
#include "termio.h"
#include "select.h"

#define	MAXCHANNELS	128
#define	RAMSIZE		0x400

#ifdef GNUDOS
#define READ		"r"
#define READBIN		"rb"
#endif

#ifdef ANSI
#else
extern	char		*sprintf();
#endif

extern	void		(*thread_hook)();
extern	bool		add_to_runq();
extern	void		encoded_write_term();
extern	char		*getenv();
extern	FILE		*fdopen();
extern	void		init_encode();
extern	bool		encoded_read_term();

extern	symbpo		eof_sym;
extern	threadpo	prolog_th,
			parlog_th;
extern	bool		h_debug;
extern	int		signal_mask;
extern	fd_set		rfdset;
extern	int		h_deadlock;
#ifdef HERMES
extern	int		wait_pipe;
#endif

io_descriptor	io[MAXCHANNELS];

iopo	next_io = io;


CHARTYPE	(*charin)();	/* read one character  */
bool		(*charout)(),	/* write one character */
		(*charback)();	/* put back one char   */

extern FILE *open_ic_file();

iopo
getNextIO()
{
    iopo 	start;

    start = next_io;
    while (io_type(next_io) != CLOSED && ++next_io != start) {
	if (next_io == io + MAXCHANNELS)
	    next_io = io;
    }

    if (io_type(next_io) == CLOSED) {
	next_io->pid = 0;
	return(next_io);
    }
    else return(NULL);
}



/*------------------------------------------------------*
 *							*
 *			Stream I/O			*
 *							*
 *------------------------------------------------------*/

/* char linebuf[BUFSIZ]; Line Buffer for Syntax Error Handling */

/* A[2] is assumed to be 0, 1 or 2 (read, write, append)
   A[3] is assumed to be a variable */
bool
pr_open_stream()
{
    register
    cellpo	reg1 = &A[1],
		reg2 = &A[2],
		reg3 = &A[3];
    FILE	*temp;
    char	*mode, *fullstop;
    strpo	conpo;
    int		io_t;
    iopo	free_io;
    int 	file_exists=TRUE;

    delnk(reg1);
    delnk(reg2);
    delnk(reg3);

    if (NotSymb(reg1) || (string_len(reg1) >= MAXFILENAME))
	throw(203);

    conpo = string_val(reg1);

    fullstop = strrchr(conpo, '.');

    switch (intvl(reg2)) {
	case 0: {
	    io_t = IN_STREAM;
	    mode = READ;
	    if (fullstop && !strcmp(fullstop, ".icp"))
		    mode = READBIN;
	    break;
	}
	case 1: {
	    io_t = OUT_STREAM;
	    mode = WRITE;
	    file_exists=FALSE;
	    if (fullstop && !strcmp(fullstop, ".icp"))
		    mode = WRITEBIN;
	    break;
	}
	case 2: {
	    io_t = OUT_STREAM;
	    mode = APPEND;
	    file_exists=FALSE;
	    if (fullstop && !strcmp(fullstop, ".icp"))
		    mode = APPENDBIN;
	    break;
	}
    }

    if ((temp = open_ic_file(conpo, mode, file_exists)) == NULL)
	throw(300);

    if (!(free_io=getNextIO())) 
    	throw(601);

    fdes(free_io)    = temp;
    io_type(free_io) = io_t;
    ended(free_io)  = FALSE;
    free_io->last_nl_pos = 0;
    free_io->nl_count = 0;

    mkreset(reg3);
    mkint(reg3, free_io - io);
    return(SUCCEED);
}

/* A[1] is assumed to be an integer */
bool
pr_close_stream()
{
    fourBytes	i;
    iopo	io_ptr;
    cellpo reg1 = &A[1];

    delnk(reg1);
    i = intvl(reg1);
    io_ptr = io + i;

    if (i <= 2L)	/* cannot close standard streams */
	return(SUCCEED);

    if (is_stream(io_ptr)) {
	io_type(io_ptr) = CLOSED;
	if (fclose(fdes(io_ptr)))
	    throw(302);
    }
    else throw(302);
    if (io_ptr->pid) {
	(void) kill(io_ptr->pid, SIGTERM);
	io_ptr->pid = 0;
    }

    if (io_ptr < next_io)
	next_io = io_ptr;
    return(SUCCEED);
}

/* A[1] is assumed to be an integer */
bool
pr_flush()
{
    cellpo reg1 = &A[1];
    iopo	io_ptr;

    delnk(reg1);
    io_ptr = io + intvl(reg1);

    if (is_stream(io_ptr))
	(void)fflush(fdes(io_ptr));

    return(SUCCEED);
}

bool
pl_flush()
{
    if (is_stream(current_output))
	(void)fflush(fdes(current_output));
    return(SUCCEED);
}

CHARTYPE
pr_read_stream()
{
    CHARTYPE	ch;

    if ((ch = getc(fdes(current_input))) == EOF)
	ended(current_input) = TRUE;

    return(ch);
}

CHARTYPE
pr_read_stream0()
{
    CHARTYPE	ch;

    if ((ch = getchar()) == EOF)
	    clearerr(stdin);		/* stdin never really ends */
    else if (ch == '\n')		/* fixes the line position of stdout */
	io[1].last_nl_pos = ftell(fdes(io+1));

    return(ch);
}

bool
pr_write_stream(ch)
CHARTYPE ch;
{
    (void)putc(ch, fdes(current_output));
    if (ch == '\n') {
	current_output->nl_count++;
	current_output->last_nl_pos = ftell(fdes(current_output));
    }
    return(SUCCEED);
}

bool
pr_unread_stream(ch)
CHARTYPE ch;
{
    (void)ungetc(ch, fdes(current_input));
    ended(current_input) = FALSE;

    return(SUCCEED);
}



/*------------------------------------------------------*
 *							*
 *			Memory I/O			*
 *							*
 *------------------------------------------------------*/

/* A[2] is assumed to be 0, 1 or 2 (read, write, append)
   A[3] is assumed to be a variable */
bool
pr_open_ram()
{
    cellpo	reg1 = &A[1],
		reg2 = &A[2],
		reg3 = &A[3],
		reg4 = &A[4];
    iopo	io_ptr, io_old;
    rampo	ram_ptr;
    strpo	block, prefill, copy;
    utwoBytes	len, newlen;

    delnk(reg1);
    delnk(reg2);
    delnk(reg3);
    delnk(reg4);

    if (!(io_ptr=getNextIO())) 
    	throw(602);

    switch (intvl(reg2)) {
    case 0:	/* read mode */
	if (IsSymb(reg1)) {
	    len = string_len(reg1);
	    if (!(block = (strpo) malloc((size_t)(len+1)))) 
		throw(603);
    
	    copy = block;
	    prefill = string_val(reg1);
	    while (len--)
		*copy++ = *prefill++;
	    *copy = EOF;

	    if (!(ram_ptr = (rampo) malloc(sizeof(ram_des))))
		throw(603);

	    ram_ptr->current = block;
	    ram_ptr->write   = NULL;
	    ram_ptr->start   = block;
	    ram_ptr->end     = block + len + 1;
	    ramd(io_ptr)     = ram_ptr;
	}
	else if (IsInt(reg1) && is_ram(io+intvl(reg1))) {
	    io_old = io + intvl(reg1);
	    ram_ptr = ramd(io_old);
	    if (io_type(io_old) == OUT_RAM)
		*(ram_ptr->write) = EOF;
	    ram_ptr->current = ram_ptr->start;	/* rewind input */
	    ram_ptr->write   = NULL;
	    ramd(io_ptr)     = ramd(io_old);
	    io_type(io_old)  = CLOSED;
	}
	else throw(206);

	io_type(io_ptr) = IN_RAM;
	break;

    case 1:	/* write mode */
	len = IsInt(reg4) ? intvl(reg4)*2 : RAMSIZE;
	if (!(block = (strpo) malloc((size_t)len))) 
	    throw(603);

	if (!(ram_ptr = (rampo) malloc(sizeof(ram_des))))
	    throw(603);

	ram_ptr->current = block;
	ram_ptr->write   = block;
	ram_ptr->start   = block;
	ram_ptr->end     = block + len;
	ramd(io_ptr)     = ram_ptr;
	io_type(io_ptr)  = OUT_RAM;
	break;

    case 2:	/* append mode */
	newlen = IsInt(reg4) ? intvl(reg4)*2 : RAMSIZE;
	if (IsSymb(reg1)) {
	    len = string_len(reg1);
	    while (newlen <= len)
		newlen *= 2;
	    if (!(block = (strpo) malloc((size_t)newlen))) 
		throw(603);
    
	    copy = block;
	    prefill = string_val(reg1);
	    while (len--)
		*copy++ = *prefill++;

	    if (!(ram_ptr = (rampo) malloc(sizeof(ram_des))))
		throw(603);

	    ram_ptr->current = block;
	    ram_ptr->write   = copy;
	    ram_ptr->start   = block;
	    ram_ptr->end     = block + newlen;
	    ramd(io_ptr)     = ram_ptr;
	}
	else if (IsInt(reg1) && is_ram(io+intvl(reg1))) {
	    io_old = io + intvl(reg1);
	    ram_ptr = ramd(io_old);
	    if (io_type(io_old) == IN_RAM) {
		copy = ram_ptr->current;
		while ((char)*copy != EOF)
		    copy++;
		ram_ptr->write = copy;
	    }
	    ramd(io_ptr)     = ramd(io_old);
	    io_type(io_old)  = CLOSED;
	}
	else {
	    if (!(block = (strpo) malloc((size_t)newlen))) 
		throw(603);
	    if (!(ram_ptr = (rampo) malloc(sizeof(ram_des))))
		throw(603);

	    ram_ptr->current = block;
	    ram_ptr->write   = block;
	    ram_ptr->start   = block;
	    ram_ptr->end     = block + newlen;
	    ramd(io_ptr)     = ram_ptr;
	}
	io_type(io_ptr) = OUT_RAM;
	break;
    }

    ended(io_ptr) = FALSE;

    mkreset(reg3);
    mkint(reg3, io_ptr - io);
    return(SUCCEED);
}

/* A[1] is assumed to be an integer */
bool
pr_close_ram()
{
    fourBytes	i;
    iopo	io_ptr;
    cellpo reg1 = &A[1];

    delnk(reg1);
    i = intvl(reg1);
    io_ptr = io + i;

    if (is_ram(io_ptr)) {
	free((char *)(ramd(io_ptr)->start));
	free((char *)ramd(io_ptr));
    }
    else throw(306);

    io_type(io_ptr) = CLOSED;
    if (io_ptr < next_io)
	next_io = io_ptr;
    return(SUCCEED);
}

/* A[1] is assumed to be an integer */
bool
pr_ram_const()
{
    twoBytes leng;
    fourBytes numCells;
    cellpo reg1 = &A[1];
    cellpo reg2 = &A[2];
    iopo io_ptr;
    rampo ram_ptr;

    delnk(reg1);
    delnk(reg2);

    if (NotVar(reg2))
	throw(208);

    io_ptr = io + intvl(reg1);
    if (io_type(io_ptr) != OUT_RAM)
	throw(306);

    ram_ptr = ramd(io_ptr);
    leng = ram_ptr->write - ram_ptr->start;
    *(ram_ptr->write) = '\0';

    (void) bind_symbol(2, ram_ptr->start, 2);

    free((char *)(ram_ptr->start));
    free((char *)ram_ptr);
    io_type(io_ptr) = CLOSED;
    return(SUCCEED);
}

/*  This ram file contains compiled code.  Free the I/O
 *  descriptor and use the ram file as a loaded code segment.
 */
codepo grab_code(index)
int index;
{
    iopo	io_ptr;
    rampo	ram_ptr;
    codepo	code_start;

    io_ptr = io+index;
    ram_ptr = ramd(io_ptr);
    code_start = (codepo) ram_ptr->start;
    free((char *)ram_ptr);
    io_type(io_ptr) = CLOSED;
    return(code_start);
}

CHARTYPE
pr_read_ram()
{
    CHARTYPE	ch;
    rampo	ram_ptr;

    ram_ptr = ramd(current_input);
    if ((char)(ch = *(ram_ptr->current)++) == EOF) {
	ended(current_input) = TRUE;
	(ram_ptr->current)--;	/* don't read off the end */
	ch = EOF;
    }

    /* do not overtake write pointer */
    if (ram_ptr->write && ram_ptr->current > ram_ptr->write) {
	ram_ptr->current = ram_ptr->write;
	ch = EOF;
    }

    return(ch);
}

bool
pr_write_ram(ch)
CHARTYPE ch;
{
    int		len, r_offset;
    strpo	block;
    rampo	ram_ptr;

    ram_ptr = ramd(current_output);
    *(ram_ptr->write)++ = ch;
    if (ram_ptr->write == ram_ptr->end) {
	len = (ram_ptr->end - ram_ptr->start);
	r_offset = ram_ptr->current - ram_ptr->start;
	if (!(block = (strpo) realloc(ram_ptr->start, (size_t)(len*2)))) 
	    longjmp(icp_interrupt, 603);
	ram_ptr->start = block;
	ram_ptr->current = block + r_offset;
	ram_ptr->write = block + len;
	ram_ptr->end = block + len * 2;
    }
    return(SUCCEED);
}

bool
pr_unread_ram(ch)
CHARTYPE ch;
{
    if (ch != EOF) {
	*--(ramd(current_input)->current) = ch;
	ended(current_input) = FALSE;
    }
    return(SUCCEED);
}

/* A[1] and A[2] are assumed to be variables */
bool
pr_ram_pipe()
{
    cellpo reg1 = &A[1];
    cellpo reg2 = &A[2];
    iopo	io_ptr;
    rampo	ram_ptr;
    strpo	block;

    delnk(reg1);
    delnk(reg2);

    if (!(block = (strpo) malloc((size_t)RAMSIZE))) 
	throw(603);

    if (!(ram_ptr = (rampo) malloc(sizeof(ram_des))))
	throw(603);

    ram_ptr->current = block;
    ram_ptr->write   = block;
    ram_ptr->start   = block;
    ram_ptr->end     = block + RAMSIZE;

    if (!(io_ptr=getNextIO())) 
    	throw(602);

    ramd(io_ptr)     = ram_ptr;
    io_type(io_ptr)  = OUT_RAM;
    ended(io_ptr)    = FALSE;
    mkreset(reg1);
    mkint(reg1, io_ptr - io);

    if (!(io_ptr=getNextIO())) 
    	throw(602);

    ramd(io_ptr)     = ram_ptr;
    io_type(io_ptr)  = IN_RAM;
    ended(io_ptr)    = FALSE;
    mkreset(reg2);
    mkint(reg2, io_ptr - io);

    return(SUCCEED);
}



/*------------------------------------------------------*
 *							*
 *			Pipe I/O			*
 *							*
 *------------------------------------------------------*/

pipepo	curr_pipe;

bool
pr_pipe()
{
    cellpo reg1 = &A[1];
    cellpo reg2 = &A[2];
    iopo	io_in, io_out;
    pipepo	pipe_ptr;
    strpo	block;

    delnk(reg1);
    delnk(reg2);

    if (!(block = (strpo) malloc((size_t)PIPESIZE))) 
	throw(613);

    if (!(pipe_ptr = (pipepo) malloc(sizeof(pipe_des))))
	throw(613);

    if (!(io_out=getNextIO())) 
    	throw(602);

    piped(io_out)   = pipe_ptr;
    th_id(io_out)   = NULL;
    io_type(io_out) = OUT_PIPE;
    ended(io_out)   = FALSE;

    if (!(io_in=getNextIO())) 
    	throw(602);

    piped(io_in)    = pipe_ptr;
    th_id(io_in)    = NULL;
    io_type(io_in)  = IN_PIPE;
    ended(io_in)    = FALSE;

    pipe_ptr->look_ptr  = block;
    pipe_ptr->read_ptr  = block;
    pipe_ptr->write_ptr = block;
    pipe_ptr->iport     = io_in;
    pipe_ptr->oport     = io_out;
    pipe_ptr->start     = block;
    pipe_ptr->end       = block + PIPESIZE;
    pipe_ptr->status	= P_EMPTY;

    mkreset(reg1);
    mkint(reg1, io_out - io);
    mkreset(reg2);
    mkint(reg2, io_in - io);
    return(SUCCEED);
}

bool
pr_close_port()
{
    fourBytes	i;
    iopo	io_ptr;
    pipepo	pipe_ptr;
    cellpo reg1 = &A[1];

    delnk(reg1);
    if (NotInt(reg1))
	throw(307);
    i = intvl(reg1);
    io_ptr = io + i;

    if (!is_pipe(io_ptr))
	throw(307);

    pipe_ptr = piped(io_ptr);
    if (pipe_ptr->status & P_LOCK)
	return(SUSPEND);

    if (io_type(io_ptr) == IN_PIPE) {
	if (pipe_ptr->status & P_RSUSP)
	    throw(308);
	if (!pipe_ptr->oport)
	    free((char *)pipe_ptr);
	else pipe_ptr->iport = NULL;
    }
    else if (io_type(io_ptr) == OUT_PIPE) {
	if (pipe_ptr->status & P_WSUSP)
	    throw(309);
	if (pipe_ptr->status & P_RSUSP) {
	    h_deadlock = 0;
#ifdef HERMES
	    if (th_id(pipe_ptr->iport) == parlog_th) {
		wait_pipe--;
		parlog_assign(&suspnd(pipe_ptr->iport));
	    }
	    else
#endif
		(void) add_to_runq(th_id(pipe_ptr->iport), FALSE);
	    pipe_ptr->status &= ~P_RSUSP;
	}
	if (!pipe_ptr->iport) {
	    free((char *)pipe_ptr->start);
	    free((char *)pipe_ptr);
	}
	else pipe_ptr->oport = NULL;
    }

    io_type(io_ptr) = CLOSED;
    if (io_ptr < next_io)
	next_io = io_ptr;
    return(SUCCEED);
}

bool
pr_release_port()
{
    fourBytes	i;
    iopo	io_ptr;
    pipepo	pipe_ptr;
    cellpo reg1 = &A[1];

    delnk(reg1);
    if (NotInt(reg1))
	throw(307);
    i = intvl(reg1);
    io_ptr = io + i;

    if (!is_pipe(io_ptr))
	throw(307);

    if (th_id(io_ptr) && th_id(io_ptr) != TH)
	throw(312);

    pipe_ptr = piped(io_ptr);
    if (pipe_ptr->status & P_LOCK)
	return(SUSPEND);

    th_id(io_ptr) = NULL;
    return(SUCCEED);
}

bool
pr_is_iport()
{
    cellpo reg1 = &A[1];

    delnk(reg1);
    if (NotInt(reg1))
	throw(307);

    return(io_type(io + intvl(reg1)) == IN_PIPE);
}

bool
pr_is_oport()
{
    cellpo reg1 = &A[1];

    delnk(reg1);
    if (NotInt(reg1))
	throw(307);

    return(io_type(io + intvl(reg1)) == OUT_PIPE);
}

CHARTYPE
read_pipe_char()
{
    register	ch = *(curr_pipe->look_ptr)++;

    if (curr_pipe->look_ptr >= curr_pipe->end)
	curr_pipe->look_ptr = curr_pipe->start;
    return(ch);
}

bool
pr_commit_read()
{
    cellpo	reg1 = &A[1];
    iopo	io_ptr;
    pipepo	pipe_ptr;

    delnk(reg1);

    if (NotInt(reg1))
	throw(307);
    io_ptr = io + intvl(reg1);
    if (io_type(io_ptr) != IN_PIPE)
	throw(307);

    if (th_id(io_ptr) != TH)
	throw(310);

    pipe_ptr = piped(io_ptr);

    if (pipe_ptr->status & P_WSUSP) {
	if (h_debug)
	    (void) fprintf(stderr, "[ wake writer on port(%d) ]\n", pipe_ptr->oport - io);
	h_deadlock = 0;
#ifdef HERMES
	if (th_id(pipe_ptr->oport) == parlog_th) {
	    wait_pipe--;
	    parlog_assign(&suspnd(pipe_ptr->oport));
	}
	else
#endif
	    (void) add_to_runq(th_id(pipe_ptr->oport), FALSE);
	pipe_ptr->status &= ~P_WSUSP;
    }

    /* can always commit empty reads */
    if (pipe_ptr->status & P_EMPTY)
	return(SUCCEED);

    if (pipe_ptr->status & P_LOCK &&
	    (pipe_ptr->look_ptr > pipe_ptr->read_ptr ||
	    (pipe_ptr->read_ptr > pipe_ptr->write_ptr &&
	     pipe_ptr->write_ptr >= pipe_ptr->look_ptr))) {
	pipe_ptr->status &= ~P_LOCK;
	pipe_ptr->read_ptr = pipe_ptr->look_ptr;
	if (pipe_ptr->read_ptr == pipe_ptr->write_ptr)
	    pipe_ptr->status |= P_EMPTY;
	return(SUCCEED);
    }
    else return(FAIL);
}

bool
pr_unlock()
{
    cellpo	reg1 = &A[1];
    iopo	io_ptr;

    delnk(reg1);

    if (NotInt(reg1))
	throw(307);
    io_ptr = io + intvl(reg1);
    if (!is_pipe(io_ptr))
	throw(307);

    if (th_id(io_ptr) != TH)
	throw(312);

    piped(io_ptr)->status &= ~P_LOCK;
    return(SUCCEED);
}

bool
pr_empty_pipe()
{
    cellpo	reg1 = &A[1];
    iopo	io_ptr;
    pipepo	pipe_ptr;

    delnk(reg1);

    if (NotInt(reg1))
	throw(307);
    io_ptr = io + intvl(reg1);
    if (!is_pipe(io_ptr))
	throw(307);

    pipe_ptr = piped(io_ptr);

    return((pipe_ptr->status & P_EMPTY) == P_EMPTY && pipe_ptr->oport);
}

bool
pr_look_pipe()
{
    cellpo	reg1 = &A[1],
		reg2 = &A[2];
    cell	val;
    CHARTYPE	(*tmp_charin)();
    iopo	io_ptr;
    pipepo	pipe_ptr;
    int		ret;

    delnk(reg1);
    delnk(reg2);

    if (NotInt(reg1))
	throw(307);


    io_ptr = io + intvl(reg1);
    if (io_type(io_ptr) != IN_PIPE)
	throw(307);

    if (!th_id(io_ptr))
	th_id(io_ptr) = TH;
    else if (th_id(io_ptr) != TH)
	throw(310);

    pipe_ptr = piped(io_ptr);

    if (pipe_ptr->status & P_LOCK) {
	pipe_ptr->status |= P_RSUSP;
	if (h_debug)
	    (void) fprintf(stderr, "[ suspend read on port(%d) ]\n", io_ptr - io);
	return(SUSPEND);
    }

    if (pipe_ptr->status & P_EMPTY) {
	if (!pipe_ptr->oport) {
	    if (pipe_ptr->start) {
		free((char *)pipe_ptr->start);
		pipe_ptr->start = NULL;
	    }
	    mksymb(&val, eof_sym);
	    return(icp_unify(reg2, &val));
	}
	else {
	    if (h_debug)
		(void) fprintf(stderr, "[ suspend read on port(%d) ]\n", io_ptr - io);
	    pipe_ptr->status |= P_RSUSP;
	    return(SUSPEND);
	}
    }

    /* ... otherwise lock it and read */
    pipe_ptr->status |= P_LOCK;
    pipe_ptr->look_ptr = pipe_ptr->read_ptr;
    mkreset(reg2);
    curr_pipe = pipe_ptr;
    tmp_charin = charin;
    charin = read_pipe_char;
    ret = encoded_read_term(&val);
    charin = tmp_charin;

    return (ret == SUCCESS ? icp_unify(reg2, &val) : ret);
}

bool
pr_read_pipe()
{
    int		status;

    status = pr_look_pipe();
    if (status == SUCCEED)
	return(pr_commit_read());
    return(status);
}

bool
write_pipe_char(ch)
CHARTYPE ch;
{
    int		len, r_offset, l_offset;
    strpo	block;

    *(curr_pipe->write_ptr)++ = ch;

    if (curr_pipe->write_ptr == curr_pipe->end)
	curr_pipe->write_ptr = curr_pipe->start;

    if (curr_pipe->write_ptr == curr_pipe->read_ptr) {
	int new_len;

	len = curr_pipe->end - curr_pipe->start;
	new_len = 2 * len;
	if (!(block = (strpo) malloc((size_t)new_len)))
	    longjmp(icp_interrupt, 613);
	r_offset = curr_pipe->end - curr_pipe->read_ptr;
	l_offset = curr_pipe->look_ptr - curr_pipe->read_ptr;
	if (l_offset < 0)
	    l_offset += len;
	(void) memcpy(block, curr_pipe->read_ptr, r_offset);
	(void) memcpy(block+r_offset, curr_pipe->start, len - r_offset);
	(void) free(curr_pipe->start);
	curr_pipe->start = block;
	curr_pipe->end = block + new_len;
	curr_pipe->read_ptr = block;
	curr_pipe->write_ptr = block + len;
	curr_pipe->look_ptr = block + l_offset;
    }
    return(SUCCEED);
}

bool
pr_write_pipe()
{
    cellpo	reg1 = &A[1],
		reg2 = &A[2];
    bool	(*tmp_charout)();
    iopo	io_ptr;
    pipepo	pipe_ptr;

    delnk(reg1);
    delnk(reg2);

    if (NotInt(reg1))
	throw(307);
    io_ptr = io + intvl(reg1);
    if (io_type(io_ptr) != OUT_PIPE)
	throw(307);

    if (!th_id(io_ptr))
	th_id(io_ptr) = TH;
    else if (th_id(io_ptr) != TH)
	throw(311);

    pipe_ptr = piped(io_ptr);
    curr_pipe = pipe_ptr;
    tmp_charout = charout;
    charout = write_pipe_char;
    init_encode();
    encoded_write_term(reg2);
    charout = tmp_charout;

    pipe_ptr->status &= ~P_EMPTY;
    if (pipe_ptr->status & P_RSUSP) {
    if (h_debug)
	(void) fprintf(stderr, "[ wake reader on port(%d) ]\n", pipe_ptr->iport - io);
	h_deadlock = 0;
#ifdef HERMES
	if (th_id(pipe_ptr->iport) == parlog_th) {
	    wait_pipe--;
	    parlog_assign(&suspnd(pipe_ptr->iport));
	}
	else
#endif
	    (void) add_to_runq(th_id(pipe_ptr->iport), FALSE);
	pipe_ptr->status &= ~P_RSUSP;
    }

    return(SUCCEED);
}



/*------------------------------------------------------*
 *							*
 *		Switching Input / Output		*
 *							*
 *------------------------------------------------------*/

bool
pr_curr_input()
{
    register
    cellpo	reg1 = &A[1],
		reg2 = &A[2];

    delnk(reg1);
    delnk(reg2);

    mkreset(reg1);
    mkint(reg1, io_type(current_input));

    mkreset(reg2);
    mkint(reg2, current_input - io);
    return(SUCCEED);
}

bool
pr_curr_output()
{
    register
    cellpo	reg1 = &A[1],
		reg2 = &A[2];

    delnk(reg1);
    delnk(reg2);

    mkreset(reg1);
    mkint(reg1, -io_type(current_output));

    mkreset(reg2);
    mkint(reg2, current_output - io);
    return(SUCCEED);
}

/* A[1] is assumed to be 0 or 1.  A[2] is an integer */
bool
pr_set_input()
{
    register
    cellpo reg1 = &A[1],
	   reg2 = &A[2];
    iopo 	io_ptr;

    delnk(reg1);
    delnk(reg2);

    io_ptr = io + intvl(reg2);

    if (intvl(reg1) == 0 && io_type(io_ptr) == IN_STREAM) {
	if (fdes(io_ptr) == stdin)
		charin = pr_read_stream0;
	else
		charin = pr_read_stream;
	charback = pr_unread_stream;
	clearerr(fdes(io_ptr));
    }
    else if (intvl(reg1) == 1 && io_type(io_ptr) == IN_RAM) {
	charin = pr_read_ram;
	charback = pr_unread_ram;
    }
    else throw(301);

    current_input = io_ptr;
    return(SUCCEED);
}

/* A[1] is assumed to be 0 or 1.  A[2] is an integer */
bool
pr_set_output()
{
    register
    cellpo reg1 = &A[1],
	   reg2 = &A[2];
    iopo	io_ptr;

    delnk(reg1);
    delnk(reg2);

    io_ptr = io + intvl(reg2);

    if (intvl(reg1) == 0 && io_type(io_ptr) == OUT_STREAM)
	charout = pr_write_stream;
    else if (intvl(reg1) == 1 && io_type(io_ptr) == OUT_RAM)
	charout = pr_write_ram;
    else throw(301);

    current_output = io_ptr;
    return(SUCCEED);
}



init_io()
{
    /* note: stdin, stdout and stderr are streams 0, 1 and 2 respectively */

    /* standard input */
/* (void)setvbuf(stdin, linebuf, _IOLBF, BUFSIZ); */
    setlinebuf(stdin);
    fdes(next_io)    = stdin;
    io_type(next_io) = IN_STREAM;
    ended(next_io)   = FALSE;

    /* set current input to stdin */
    current_input   = next_io++;

    /* standard output */
    fdes(next_io)    = stdout;
    io_type(next_io) = OUT_STREAM;
    ended(next_io)   = FALSE;

    /* set current output to stdout */
    current_output  = next_io++;

    /* standard error */
    fdes(next_io)    = stderr;
    io_type(next_io) = OUT_STREAM;
    ended(next_io)   = FALSE;

    next_io++;
}



/*------------------------------------------------------*
 *							*
 *		    Output Primitives			*
 *							*
 *------------------------------------------------------*/



/* this primitive outputs spaces */
/* A[1] is assumed to be an integer */
bool
pr_tab()
{
    register
    fourBytes	spaces;
    cellpo	reg1 = &A[1];

    delnk(reg1);
    spaces = intvl(reg1);
    while (spaces-- > 0)
	(*charout)(' ');
    return(SUCCEED);
}

void
put_char(ch)
CHARTYPE	ch;
{
    switch (ch) {
	case '\b':	/* backspace */
	    (*charout)('\\');
	    (*charout)('b');
	    break;
	case '\t':	/* tab */
	    (*charout)('\\');
	    (*charout)('t');
	    break;
	case '\n':	/* linefeed */
	    (*charout)('\\');
	    (*charout)('n');
	    break;
	case '\v':	/* vertical tab */
	    (*charout)('\\');
	    (*charout)('v');
	    break;
	case '\f':	/* formfeed */
	    (*charout)('\\');
	    (*charout)('f');
	    break;
	case '\r':	/* carriage return */
	    (*charout)('\\');
	    (*charout)('r');
	    break;
	case '\007':	/* alarm (bell) */
	    (*charout)('\\');
	    (*charout)('a');
	    break;
	case '\033':	/* escape */
	    (*charout)('\\');
	    (*charout)('e');
	    break;
	case '\177':	/* delete */
	    (*charout)('\\');
	    (*charout)('d');
	    break;
	case '\\':	/* backslash */
	    (*charout)('\\');
	    (*charout)('\\');
	    break;
	default:
	    if (ch >= ' ')
		(*charout)(ch);
	    else {	/* control character */
		(*charout)('\\');
		(*charout)('^');
		(*charout)((CHARTYPE)('@'+ch));
	    }
	    break;
    }
}

/* this primitive writes a quoted atom to the current output */
/* A[1] is assumed to be a list of ASCII characters.  A[2] is an integer */
bool
pr_put_str()
{
    register
    cellpo	list = &A[1],
		qte  = &A[2],
		el;

    delnk(list);
    delnk(qte);

    /* print opening quote */
    (*charout)((CHARTYPE)intvl(qte));

    while (IsList(list)) {
	el = hd(list);
	delnk(el);

	/* the quote is doubled up */
	if (intvl(el) == intvl(qte))
	    (*charout)((CHARTYPE)intvl(el));

	put_char(intvl(el));
	list = tl(list);
	delnk(list);
    }

    /* print closing quote */
    (*charout)((CHARTYPE)intvl(qte));

    return(SUCCEED);
}

/* A[1] is assumed to be an atom */
bool
pr_put_q_atom()
{
    cellpo	atom = &A[1];
    utwoBytes	len;
    strpo	chars;
    CHARTYPE	ch;

    delnk(atom);

    len = string_len(atom);
    chars = string_val(atom);

    /* print opening quote */
    (*charout)('\'');

    while (len--) {
	if ((ch=*chars++) == '\'') {
	    (*charout)('\'');
	    (*charout)('\'');
	}
	else put_char(ch);
    }

    /* print closing quote */
    (*charout)('\'');

    return(SUCCEED);
}

/* A[1] is assumed to be an atom */
bool
pr_put_atom()
{
    cellpo	atom = &A[1];
    strpo	chars, start, pt;

    delnk(atom);

    start = chars = string_val(atom);

    if (io_type(current_output) == OUT_STREAM) {
	fputs(start, fdes(current_output));

	if (pt = (strpo) strchr(chars, '\n')) {
	    do {
		current_output->nl_count++;
		chars = pt + 1;
	    } while (pt = (strpo) strchr(chars, '\n'));

	    /* next line needed because ftell() resets _cnt to 0 ! */
	    fflush(fdes(current_output));

	    current_output->last_nl_pos = ftell(fdes(current_output))
			- string_len(atom) + (chars - start);
	}
    }
    else if (io_type(current_output) == OUT_RAM) {
	int len = string_len(atom);
	rampo ram_ptr = ramd(current_output);
	if (ram_ptr->write + len <  ram_ptr->end) {
	    (void)memcpy(ram_ptr->write, chars, (int)len);
	    ram_ptr->write += len;
	}
	else {
	    while (len--)
		(*charout)(*chars++);
	}
    }
    else return(FAIL);

    return(SUCCEED);
}

put_digits(number)
fourBytes number;
{
    if (number >= 10)
	put_digits(number/10);
    (*charout)((CHARTYPE)(number%10+'0'));
}

/* A[1] is assumed to be an integer */
bool
pr_put_number()
{
    cellpo	num = &A[1];
    fourBytes	number;

    delnk(num);

    if ((number = intvl(num)) < 0) {
	(*charout)('-');
	number = -number;
    }

    put_digits(number);
    return(SUCCEED);
}

bool
pr_put_float()
{
    cellpo	num = &A[1];
    char	ch[80], *ptr = ch;
    FLOAT	flt;

    delnk(num);
    flt = floatvl(num);

    if (flt < 0) {
	flt = 0 - flt;
	(*charout)('-');
    }

    (void)sprintf(ch, "%.16lg", flt);

#ifndef SUNOS41
    /* hack around SunOS 4.0 "%g" bug */
    if (ch[0] == '.')
	(*charout)('0');
#endif

    while (*ptr)
	(*charout)(*ptr++);

    return(SUCCEED);
}

#define alpha	1
#define graphic	2
#define special	3
/* A[1] is assumed to be an atom, A[2] is a variable */
bool
pr_atom_type()
{
    cellpo	atom = &A[1],
		type = &A[2];
    strpo	chars;
    utwoBytes	len;

    delnk(atom);
    delnk(type);

    if (!(len = string_len(atom)))
	return(FAIL);	/* the null constant '' needs quoting */
    chars = string_val(atom);

    switch(chtype(*chars)) {
	case lower:
		/* is this an alphanumeric atom ? */
		while (--len)
		    switch (chtype(*++chars)) {
			case lower:
			case upper:
			case number:
			    break;
			default:
			    return(FAIL);
		    }
		mkreset(type);
		mkint(type, alpha);
		return(SUCCEED);
		break;

	case solo:
		if (len == 1) {
		    mkreset(type);
		    mkint(type, special);
		    return(SUCCEED);
		}
		else return(FAIL);
		break;

	case punct:
		if (*chars == brace && *(chars+1) == endbrace && len == 2) {
		    mkreset(type);
		    mkint(type, special);
		    return(SUCCEED);
		}
		else if (*chars != dot)
		    return(FAIL);

		/* ... fall through */

	case graph:	/* is this a graphic atom ? */
		if (*chars == dot && len == 1)	/* single '.' needs quotes */
		    return(FAIL);
		while (len--) {
		    if (chtype(*chars) == graph || *chars == dot)
			chars++;
		    else return(FAIL);
		}
		mkreset(type);
		mkint(type, graphic);
		return(SUCCEED);

	default:
		return(FAIL);
    }
    return(FAIL);
}



/*------------------------------------------------------*
 *							*
 *		     TTY Primitives			*
 *							*
 *------------------------------------------------------*/

#define MASTER	"/dev/ptyXY"
#define SLAVE	"/dev/ttyXY"

bool
pr_tty()
{
#ifdef GNUDOS
	return(FAIL);
#else
    register
    cellpo	reg1 = &A[1],
		reg2 = &A[2],
		reg3 = &A[3];

    char	*master_path,
		*slave_path;
    strpo	name, ptr;
    uchar	buf[100];
    int		i, master, slave, err, old_mask;
    FILE	*in, *out;
    iopo	free_io;
    long	params;
    int		pid;

    delnk(reg1);
    delnk(reg2);
    delnk(reg3);

    if (NotSymb(reg1)) {
	(void) fprintf(stderr, "tty primitive : atom expected as first argument\n");
	return(FAIL);	/* temporary */
    }

    ptr = string_val(reg1);

    name = ptr;

    /* switch off interrupts */
    old_mask = sigblock(signal_mask);

    master_path = malloc((unsigned)(strlen(MASTER)+1));
    (void) strcpy(master_path, MASTER);

    slave_path = malloc((unsigned)(strlen(SLAVE)+1));
    (void) strcpy(slave_path, SLAVE);

    for(i=0; i <= ('s'-'p') * 16; ++i) {
	master_path[strlen(MASTER)-2] = i / 16 + 'p';
	master_path[strlen(MASTER)-1] = "0123456789abcdef"[i & 15];
	master = open(master_path, O_RDWR);
	if(master >= 0)
	    break;
    }

    if(master < 0) {
	(void) fprintf(stderr, "tty primitive : can't get a pty\n");
	return(FAIL);	/* temporary */
    }

    slave_path[strlen(SLAVE)-2] = master_path[strlen(MASTER)-2];
    slave_path[strlen(SLAVE)-1] = master_path[strlen(MASTER)-1];
    slave = open(slave_path, O_RDWR, 0);

    err = ioctl(master, TIOCLGET, &params);
    params |= LCRTERA;
    err |= ioctl(master, TIOCLSET, &params);
    if (err) {
	(void) fprintf(stderr, "ioctl error in tty primitive\n");
	return(FAIL);	/* temporary */
    }

    (void) sprintf(buf, "-S%s%d", &slave_path[strlen(slave_path)-2], master);

    free(master_path);
    free(slave_path);

    pid = vfork();
    if(pid == 0) {
	(void)close(slave);
        (void) execlp("xterm", "xterm", "-name", name, buf, 0);
        (void) fprintf(stderr, "xterm exec failed\n");
        _exit(1);	/* temporary */
    }

    /* restore interrupts */
    (void) sigsetmask(old_mask);

    (void)close(master);
    (void)read(slave, buf, sizeof(buf)); /* suck xterm junk before echo on */
    in = fdopen(slave, "r");
    out = fdopen(slave, "a");

    if (!(free_io=getNextIO()))
	throw(601);
    fdes(free_io)    = in;
    io_type(free_io) = IN_STREAM;
    ended(free_io)   = FALSE;
    free_io->pid     = pid;
    mkreset(reg2);
    mkint(reg2, free_io - io);

    if (!(free_io=getNextIO()))
	throw(601);
    fdes(free_io)    = out;
    io_type(free_io) = OUT_STREAM;
    ended(free_io)   = FALSE;
    free_io->pid     = pid;
    mkreset(reg3);
    mkint(reg3, free_io -io);

    return(SUCCEED);
#endif GNUDOS
}

bool
pr_tty_get0()
{
    CHARTYPE	ch;
    int		nbytes, fd;
    FILE	*infile;
    cellpo	reg1 = &A[1];

    if (thread_hook)
	(*thread_hook)();

    /* check if terminal input */
    infile = fdes(current_input);
    if (infile->_cnt == 0 && isatty((fd=fileno(infile)))) {
#ifdef GNUDOS
	nbytes=0L;
#else
	(void)ioctl(fd, FIONREAD, &nbytes);
#endif
	if (nbytes == 0) {
	    FD_SET(fd, &rfdset);
	    update_select_width(fd);
	    if (TH == prolog_th) {
		if (h_deadlock & D_PROLOG) {
		    wait_for_user();
		    return(REQUEUE);
		}
		else h_deadlock |= D_PROLOG;
	    }
	    return(WAIT);
	}
    }
    h_deadlock = 0;

    if ((ch = getc(infile)) == EOF) {
	ch = 26;		/* Ctrl-Z is Prolog EOF */
	clearerr(infile);	/* terminal input never really ends */
    }

    delnk(reg1);
    mkreset(reg1);
    mkint(reg1, ch);
    return(SUCCEED);
}

/***************** this has been ripped off from sicstus prolog **********/
#define NUMBUFSIZE	200
bool
pr_format_print_float()
{
	cellpo reg1, reg2, reg3;
	char flag, buf[10];
	int precision;
	double val;
	FILE *f = fdes(current_output);

	reg1 = &A[1]; delnk(reg1);
	reg2 = &A[2]; delnk(reg2);
	reg3 = &A[3]; delnk(reg3);

	if (!IsInt(reg1) || !IsInt(reg3))
		throw(213);

	if (IsInt(reg2))
		val = intvl(reg2);
	else if (IsFloat(reg2))
		val = floatvl(reg2);
	else
		throw(213);

	flag = (char) intvl(reg1);
	precision = intvl(reg3);

	if (precision >= 0)
		sprintf(buf,"%%.%d%c", precision, flag);
	else
		sprintf(buf,"%%%c", flag);
	fprintf(f,buf,val);
	return(SUCCEED);
}

bool
pr_format_print_integer()
{
	cellpo reg1, reg2, reg3;
	int val, precision;
	FILE *f = fdes(current_output);
	char numBuf[NUMBUFSIZE], formatChar;
	int fracStart;

	reg1 = &A[1]; delnk(reg1);
	reg2 = &A[2]; delnk(reg2);
	reg3 = &A[3]; delnk(reg3);

	if (!IsInt(reg1) || !IsInt(reg2) || !IsInt(reg3))
		throw(213);
	formatChar = (char) intvl(reg1);
	val = intvl(reg2);
	precision = intvl(reg3);
  
	switch(formatChar) {
		case 'd':
			if(precision <= 0)
			fprintf(f,"%d", val);
			else {
				sprintf(numBuf,"%.*d", precision + 1, val);
				fracStart = strlen(numBuf) - precision;
				fprintf(f,"%.*s.%s", fracStart, numBuf, numBuf + fracStart);
			}
			break;
		case 'D': {
			register int curPos;
			int charsBeforeComma;

			if (precision <= 0)
				precision = 0;
			sprintf(numBuf,"%.*d", precision + 1, val);
			fracStart = strlen(numBuf) - precision;
			curPos = 0;
			if (numBuf[0] == '-') {
				putc('-', f);
				curPos++;
			}
			if ((charsBeforeComma = (fracStart - curPos)%3) || (charsBeforeComma = (fracStart - curPos) ? 3 : 0)) {
				fprintf(f,"%.*s", charsBeforeComma, numBuf + curPos);
				curPos += charsBeforeComma;
			}
			for (; curPos < fracStart; curPos += 3)
				fprintf(f,",%.3s", numBuf + curPos);
			if (precision)
				putc('.', f);
			fputs(numBuf + fracStart, f);
			}
			break;
		case 'r':
		case 'R': {
			char alphaBase;
			register unsigned long u;
			register int curPos;
			register char code;

			if(precision < 2 || precision > 36)
				precision = 8;


			if (val < 0) {
				putc('-', f);
				u = -(val + 1);
				u++;
			} else if (!val) {
				putc('0', f);
				break;
			} else
				u = val;
			alphaBase = 'A' - 10;
			if (formatChar == 'r')
				alphaBase += 'a' - 'A';

			for (curPos = NUMBUFSIZE - 1, numBuf[curPos] = 0; u > 0; u = u/precision) {
				code = u%precision;
				numBuf[--curPos] = (code < 10) ? ('0' + code) : (alphaBase + code);
			}
			fputs(numBuf + curPos, f);
			}
			break;
	}
	return(SUCCEED);
}

/* assuming arg1 is int, arg2 is var */
bool pr_line_position()
{
	cellpo arg1 = &A[1], arg2;
	iopo io_ptr;

	delnk(arg1);
	io_ptr = io + intvl(arg1);

	if (is_stream(io_ptr)) {
		gc_test(3L, 2);
		arg2 = &A[2]; delnk(arg2);
		mkreset(arg2);

		/* next line needed because ftell() resets _cnt to 0 ! */
		fflush(fdes(io_ptr));

		mkint1(arg2, ftell(fdes(io_ptr)) - io_ptr->last_nl_pos);

	} else
		throw(302);
	return(SUCCEED);
}


/* assuming arg1 is int, arg2 is var */
bool pr_line_count()
{
	cellpo arg1 = &A[1], arg2;
	iopo io_ptr;

	delnk(arg1);
	io_ptr = io + intvl(arg1);

	if (is_stream(io_ptr)) {
		gc_test(3L, 2);
		arg2 = &A[2]; delnk(arg2);
		mkreset(arg2);
		mkint1(arg2, io_ptr->nl_count);
	} else
		throw(302);
	return(SUCCEED);
}
