/* Copyright (C) 1992 Imperial College */
#include <ctype.h>
#include "primitives.h"

#define LOAD_FACTOR	0.8
#define NUMBERS		20
#define VARIABLES	30
#define MAXNAME 	5 /*maximum length of variable names(_0.._9999!)*/
#define MAXNO		9999
#define FOUR		4
#define	FIVE		5

#ifdef ANSI
extern twoBytes length(cellpo list);
extern bool samesymb(symbpo,symbpo);
#else
extern char     *memset();
extern twoBytes length();
extern bool	samesymb();
extern char	*sprintf();
#endif

/* forward reference */
static void rehash_variables();

typedef struct var_rec {
    cellpo	var;  /* the address of an unbound variable */
    symbpo	name; /* the name of a variable */
    short	link;
    short	order;	/* used to keep the order of occurrence */
} *varpo;

/* when searching sequentially for dangerous numbers, 9999 is to be used
   to mark the end of the array which contains them, while -1, which is
   used to mark the beginning of the array during insertion, must be skipped
*/

static	varpo 	vars = NULL; 	/* table of variables */
static	varpo  	free_var;	/* index used to locate next free entry */
static	short	tabSize;	/* size of table of variables */
static	short	addrSize;	/* divisor used for hashing */
static	short 	*usedNos; 	/* the table of numbers which cannot be used */
static	short	*endNos;	/* the end of the table */
static	short	*nextNo;	/* pointer into table */
static	short	nsize;		/* size of table of numbers */
static	short	heap_cells;	/* the heap cells needed to ground the term */
static	short	v_count;	/* counter used to create new variable names */
static	cellpo 	next_name; 	/* the tail of the list of variable names */
static	cellpo	next_var;	/* the tail of the list of variables */
static	jmp_buf	errjump;
static	short 	argno;		/* the number of arguments passed to pr_toground */
static	short	order;		/* used to maintain ordering of occurrence */
static	bool	ground_term;	/* flag to indicate if term is ground */


/* sets up table of variables */
void vars_init(varno)
short varno;
{
    if (vars == NULL) {
	addrSize = VARIABLES;
	tabSize = addrSize / LOAD_FACTOR;
	if (!(vars = (varpo)alloc(tabSize, sizeof(struct var_rec))))
	    longjmp(icp_interrupt,613);
    }

    if (varno > addrSize) {
	do {
	rehash_variables(FALSE);
	} while (varno > addrSize);
    }
    else {
	(void) memset((char *)vars, 0, (int)(tabSize*sizeof(struct var_rec)));
	free_var = vars + tabSize - 1;
    }

    order = 0;
}

/* sets up array of dangerous numbers */
void nums_init()
{
    if (usedNos == NULL) {
	nsize = NUMBERS;
	if (!(usedNos = (short*) alloc(nsize, sizeof(short))))
	    longjmp(icp_interrupt,613);
    }

    *usedNos = -1;		/* needed as a guard during insertion */
    nextNo = usedNos;
    endNos = usedNos + nsize - 1;
    v_count = 0;
}

/* returns the symbol '_n' */
symbpo mksym(n)
short n;
{
    cell	c;
    uchar 	varname[MAXNAME];
    strpo	str = varname;
    fourBytes	len;		/* length of '_n' */

    (void)sprintf(str,"_%d",n);
    len = (fourBytes)strlen(str);
    alloc_symb(&c, len, str);
    return(symbvl(&c));
}

/* returns n >= 0 if sym is of the form '_n' and -1 otherwise */
short dangerous(sym)
symbpo sym;
{
    short 	len = symblngth(sym);
    strpo 	str = symbname(sym), p;

    if (*str == '_' && len < 6) {
    	p = str+1;
    	while (--len && isdigit((int)*p))
	    p++;
    }
    return( len ? -1 : (short)atoi(str+1));
}

/* add a name to the list of variable names */
add_name(n)
symbpo n;
{
    alloc_list(next_name, next_name);
    mksymb(next_name,n);
    next_name++;
}

/* add a variable to the list of variables */
void add_var(v)
cellpo v;
{
    alloc_list(next_var, next_var);
    mklnk(next_var, v);
    next_var++;
}

/* returns next free entry in table of variables */
varpo nextv()
{
    while (free_var >= vars && free_var->var)
	free_var--;
    return((free_var >= vars) ? free_var : NULL);
}

/* if v is already in the variable table then it returns
   false otherwise it enters v in the table and returns true
*/
bool insertv(v, where)
cellpo	v;
varpo	*where;
{
    varpo entry;
    varpo probe = vars + ((fourBytes) v % addrSize);

    if (probe < vars)
	probe += addrSize - 1;

    for(;;) {
    	if (probe->var && v == probe->var) { /* found it */
	    *where = probe;
	    return(FAIL);
	}
	if (!probe->link) { /* end of chain */
	    if (entry = (probe->var ? nextv() : probe)) {
	        probe->link = entry - probe;
	        entry->var = v;
		entry->order = order++;
		*where = entry;
	        return(TRUE);
	    } else {
	    	rehash_variables(TRUE);
	    	return(insertv(v, where));
	    }
	}
	probe+= probe->link; /* follow chain */
    }
}

/* if v is in the table then it returns its entry, otherwise it returns NULL */
varpo find(v)
cellpo v;
{
    varpo probe = vars + ((fourBytes) v % addrSize);
    if (probe < vars)
    	probe += addrSize - 1;
    for(;;) {
    	if (probe->var && v == probe->var)  /* found it */
	    return(probe);
	if (!probe->link)  /* end of chain */
	    return(NULL);
	probe+= probe->link; /* follow chain */
    }
}

/* inserts n into the array of usedNos whose elements are sorted in
   increasing order. If necessary, the array is made twice as large
   Note that the array may contain duplicates.
*/
insertn(n)
short n;
{
    short *j, nos;

    if (nextNo == endNos) {	/* is the table of numbers full? */
	nos = endNos - usedNos;
    	nsize = nsize * 2;	/* allow for twice as many numbers */
    	usedNos = (short*) realloc((char*)usedNos,(size_t)(nsize * sizeof(short)));
    	if (!usedNos)
    	    longjmp(icp_interrupt,613);
    	endNos = usedNos + nsize - 1;
    	nextNo = usedNos + nos;
    }

    /* inserts n into the sorted array usedNos */
    j = nextNo++;
    while (n < *j) {
    	*(j+1) = *j;
	j--;
    }
    *(j+1) = n;
}

/* returns the next name of the form '_n' where n is
   such that the name does not clash with any constant
*/
symbpo newName()
{
    /* exhaust all numbers less than the constant before skipping over it */
    while (v_count == *nextNo) {
	while (v_count == *++nextNo)	/* skip duplicates */
	    ;
	v_count++;
    }
    return(mksym(v_count++));
}

/* if variable v has been assigned a name, then that name is returned,
   otherwise a new name is created and returned after the list of
   variables and that of variable names have been updated.
*/
symbpo name(v)
cellpo v;
{
    varpo probe = vars + ((fourBytes) v % addrSize);

    if (probe < vars)
	probe += addrSize - 1;

    for(;;) {
    	if (probe->var && v == probe->var) { /* found it */
	    if (!probe->name) {
	    	add_name(probe->name = newName());
	    	if (argno == FOUR)
	    	    add_var(v);
	    }
	    return(probe->name);
	}
	probe+= probe->link; /* follow chain */
    }
}

void process(Vars,Varnames)
cellpo 	Vars,
	Varnames;
{
    cellpo var, name;
    varpo  entry;
    symbpo s;
    short  n;

    while (NotNil(Vars)) {
    	var = hd(Vars);
    	name = hd(Varnames);
    	Vars = tl(Vars);
    	Varnames = tl(Varnames);
    	delnk(var);
    	delnk(name);
    	if (NotSymb(name)) {
    	    longjmp(errjump,0);
    	}
    	delnk(Vars);
    	delnk(Varnames);
    	if (IsVar(var) && (entry = find(var))) {
    	    if ((n = dangerous(s = symbvl(name))) >= 0)
    		insertn(n);
    	    entry->name = s;
    	    add_name(s);
    	}
    }
}

/* enters the term's variables in the variable table and constructs a
   sorted array of those integers n such that the constant '_n' appears
   in the term. In addition, a count of the heap cells needed to construct
   a ground copy of the term is computed.
*/
void traverse_h(term)
cellpo term;
{
    delnk(term);

    switch (tg(term)) {
	case symb_ref: 	{
    	    short n;
    	    if ((n = dangerous(symbvl(term))) >= 0)
    	    	insertn(n);
	    break;
	}
	case var_ref:	{
	    varpo where;
    	    if (insertv(term, &where))
    		heap_cells += 3;
	    break;
	}
	case list_ref:	{
	    heap_cells += 2;
	    traverse_h(hd(term));
	    traverse_h(tl(term));
	    break;
	}
	case tpl_ref: 	{
	    cellpo tuple = functor(term);
	    short  n 	 = arity(term);
	    heap_cells += n + 1;
	    while (n--)
	        traverse_h(tuple++);
	}
    }
}

/* g becomes a ground version of h. g is assumed to point to
   a cell allocated from the heap
*/
void to_ground(h, g)
cellpo h, g;
{
    delnk(h);

    switch (tg(h)) {

	case var_ref:	{
	    mksymb(g,name(h));
	    break;
	}
	case list_ref:	{
	    cellpo  start;
	    alloc_list(g,start);
	    to_ground(hd(h),start++);
	    to_ground(tl(h),start);
	    break;
	}
	case tpl_ref: 	{
	    short  n = arity(h);
	    cellpo start,
	    	   tuple = functor(h);
	    alloc_tpl(g,start,n);
	    while (n--)
	    	to_ground(tuple++,start++);
	    break;
	}
	default:
	    *g = *h;
	    break;
    }
}

/* There are two modes of usage:
Mode (1):
    ASSUMPTIONS: Hollow is bound to some term, while Ground, Vars, Varnames
	and Usednames are unbound.
    EFFECTS: [1] Ground is instantiated with a ground copy of Hollow in
    	which variables have been replaced with constants. The constants
    	used  are of the form '_n', where n is any integer 0, 1, 2... such
    	that it is not among Hollow's constants. [2] Vars is instantiated
    	to a list of the variables of Hollow. [3] Usednames is instantiated
    	to a list of the names which replace these variables in Ground (the
    	nth element of Vars being replaced with the nth element of Usednames).
Mode (2):
    ASSUMPTIONS: Hollow is bound to some term, Ground and Usednames are
    	unbound, while Vars is bound to a list of terms (multiple occurr-
    	ences of atoms in this list are not recognised) and Varnames to
    	a list (of the same length) of atoms.
    EFFECTS: [1] Ground is instantiated with a ground copy of Hollow in
    	which variables have been replaced with constants. If the nth
    	term in Vars is a variable which appears in Hollow, then the
    	constant used to replace it in Ground is the nth atom of Varnames.
    	Those variables (if any) which appear in Hollow but not in Vars,
    	are replaced with constants of the form '_n', where n is any
    	integer  0,1,2... such that it is not among Hollow's constants.
    	[2] Usednames is instantiated to a list of the constants used to
    	replace Hollow's variables.
*/
bool pr_to_ground()
{
    cellpo 	Hollow 	    = &A[1],
    		Ground 	    = &A[2],
    		Vars 	    = &A[3],
    		Varnames    = &A[4],
    		Usednames   = &A[5];
    short 	len;

    delnk(Hollow);
    delnk(Ground);
    delnk(Vars);
    delnk(Varnames);
    delnk(Usednames);

    argno = IsVar(Varnames) ? FOUR  : FIVE;

    len = length(Varnames);

    if (NotVar(Ground)
      ||NotVar(Usednames)
      ||(argno == FOUR && (NotVar(Vars) || NotVar(Varnames)))
      ||(argno == FIVE && (len!=length(Vars)
     			 ||(NotList(Vars) && NotNil(Vars))
     			 ||(NotList(Varnames)&&NotNil(Varnames)))))
      	throw(210);

    heap_cells = 0;

    vars_init(VARIABLES);
    nums_init();

    traverse_h(Hollow);

    if (gc_test((fourBytes)(heap_cells+(argno==FIVE?len*2:0)),argno)) {
    	Hollow 	    = &A[1];
    	Ground 	    = &A[2];
    	Vars 	    = &A[3];
    	Varnames    = &A[4];
    	Usednames   = &A[5];
    	delnk(Hollow);
    	delnk(Ground);
    	delnk(Vars);
    	delnk(Varnames);
    	delnk(Usednames);
    }

    mkreset(Ground);
    mkreset(Usednames);
    if (argno == FOUR) mkreset(Vars);

    next_name = Usednames;
    if (argno == FOUR) next_var = Vars;

    if (argno == FIVE)
    	if (setjmp(errjump)) throw(210)
    	else process(Vars,Varnames);

    *(nextNo+1) = MAXNO;	/* insert sentinel value at end */
    nextNo = usedNos+1;		/* reset, skip initial sentinel value */

    to_ground(Hollow,Ground);

    mknil(next_name);
    if (argno == FOUR) mknil(next_var);

    return(SUCCEED);
}

/* enters each variable name in the table associating it with some
   unbound variable. In doing this it constructs the list of unbound
   variables that is to be returned by pr_to_hollow
*/
int enter_names(Varnames)
cellpo Varnames;
{
    varpo   entry,
	    probe;
    cellpo  hed;
    symbpo  name;

    while (NotNil(Varnames)) {
	hed = hd(Varnames);
	Varnames = tl(Varnames);
	delnk(hed);
	if (NotSymb(hed))
		return(FALSE);
	name = symbvl(hed);
	delnk(Varnames);
    	probe = vars + symbhashval(name) % addrSize;
	if (probe < vars)
	    probe += addrSize - 1;
	for(;;) {
	    if (probe->name && samesymb(name,probe->name)) /* found it */
	    	break;
	    if (!probe->link) { /* end of chain */
	    	cellpo var;
		entry = (probe->name ? nextv() : probe);
		probe->link = entry - probe;
		entry->name = name;
		alloc_cell(var);
		mkunb(var);
		add_var(var);
		entry->var = var;
		break;
	    }
	    probe += probe->link; /* follow chain */
	}
    }
    return(TRUE);
}

/* if the name is in the table then it returns the name's
   associated unbound variable, otherwise it returns NULL
*/
cellpo isa_var(name)
symbpo name;
{
    varpo probe = vars + symbhashval(name) % addrSize;

    if (probe < vars)
	probe += addrSize - 1;

    for(;;) {
	if (probe->name && samesymb(name,probe->name)) /* found it */
	    return(probe->var);
	if (!probe->link) /* end of chain */
	    return(NULL);
	probe += probe->link; /* follow chain */
    }
}


/* traverse ground term g counting the number of
   heap cells needed to construct a copy of it
*/
void traverse_g(term)
cellpo term;
{
    delnk(term);

    switch(tg(term)) {
    case var_ref: {
	if (ground_term)
	    longjmp(errjump,0);
	else {
	    varpo where;
	    (void) insertv(term, &where) ;
	}
	break;
    }
    case list_ref: {
	heap_cells += 2;
	traverse_g(hd(term));
	traverse_g(tl(term));
	break;
    }
    case tpl_ref:{
	cellpo tuple = functor(term);
	short  n = arity(term);
	heap_cells += n + 1;
	while (n--)
	    traverse_g(tuple++);
    }}
}

/* h becomes a hollow version of g. h is assumed to
   point to a cell allocated from the heap
*/
void to_hollow(g, h)
cellpo g, h;
{
    delnk(g);

    switch (tg(g)) {
	case symb_ref: {
	    symbpo s = symbvl(g);
	    cellpo var;
	    if (var = isa_var(s)) {
		mklnk(h,var);
	    } else mksymb(h,s);
	    break;
	}
	case list_ref: {
	    cellpo  start;
	    alloc_list(h,start);
	    to_hollow(hd(g),start++);
	    to_hollow(tl(g),start);
	    break;
	}
	case tpl_ref: {
	    short  n = arity(g);
	    cellpo start,
	       tuple = functor(g);
	    alloc_tpl(h,start,n);
	    while (n--)
		to_hollow(tuple++,start++);
	    break;
	}
	default:
	    *h = *g;
	    break;
    }
}

/* Assumptions: Ground is bound to some ground term (if the term contains
   any variables the primitive throws an error), Varnames is bound to a
   list of atoms (multiple occurrences of atoms in this list are not
   recognised), and Hollow and Vars are unbound.

   Effects:
   [1] Hollow is instantiated with a hollow copy of Ground in which those
   atoms which appear in Varnames have been replaced with unique variables.
   [2] Vars is instantiated to a list of these variables (the nth element
   of Varnames being replaced with the nth element of Vars).
*/
bool pr_to_hollow()
{
    cellpo  Ground   	= &A[1],
	    Hollow   	= &A[2],
    	    Varnames 	= &A[3],
    	    Vars  	= &A[4];

    delnk(Ground);
    delnk(Hollow);
    delnk(Varnames);
    delnk(Vars);

    heap_cells = 0;
    ground_term = TRUE;

    if (setjmp(errjump)
    	|| (NotList(Varnames) && NotNil(Varnames))
    	|| NotVar(Vars)
    	|| NotVar(Hollow))
    	throw(210)
    else traverse_g(Ground);

    v_count = (short)length(Varnames);

    if (gc_test((fourBytes) heap_cells + v_count*2, 4)) {
    	Ground = &A[1];
    	Hollow = &A[2];
    	Varnames = &A[3];
    	Vars = &A[4];
    	delnk(Ground);
    	delnk(Hollow);
    	delnk(Varnames);
    	delnk(Vars);
    }

    next_var = Vars;

    vars_init(v_count ? v_count : 1);
    if (!enter_names(Varnames))
	throw(210);

    mkreset(Hollow);
    mkreset(Vars);

    to_hollow(Ground,Hollow);

    mknil(next_var);
    return(SUCCEED);
}

/* makes the variable table twice as large */
static void
rehash_variables(redo)
bool	redo;
{
    short n = tabSize;
    varpo p = vars + tabSize - 1;

    addrSize *= 2;
    tabSize = addrSize / LOAD_FACTOR;
    if (!(vars = (varpo) alloc(tabSize,sizeof(struct var_rec))))
        longjmp(icp_interrupt,613);
    (void) memset((char *)vars, 0, (int)(tabSize*sizeof(struct var_rec)));
    free_var = vars + tabSize - 1;

    if (redo) {
	while (n--) {
	    reinsertv(p->var, p->order);
	    p--;
	}
	p++;
    }
    else p -= (n - 1);

    free((char*)p);
}

reinsertv(v, order)
cellpo	v;
short	order;
{
    varpo entry;
    varpo probe = vars + ((fourBytes) v % addrSize);

    if (probe < vars)
	probe += addrSize - 1;

    while (probe->link)
	probe+= probe->link;	/* follow chain */

    entry = (probe->var ? nextv() : probe);
    probe->link = entry - probe;
    entry->var = v;
    entry->order = order;
}

/* enters the term's variables in the variable table */
void
find_vars(term)
cellpo	term;
{
    delnk(term);

    switch (tg(term)) {
	case var_ref:	{
	    varpo where;
    	    if (insertv(term, &where))
    		heap_cells += 2;
	    break;
	}
	case list_ref:	{
	    find_vars(hd(term));
	    find_vars(tl(term));
	    break;
	}
	case tpl_ref: 	{
	    cellpo tuple = functor(term);
	    short  n 	 = arity(term);
	    while (n--)
	        find_vars(tuple++);
	}
    }
}

void make_varlist(Vars)
cellpo Vars;
{
    varpo p;
    short n = 0;

    for (n=0; n<order; n++) {
	p = vars;
    	while (!p->var || p->order != n)
	    p++;
    	alloc_list(Vars,Vars);
	mklnk(Vars,p->var);
	Vars++;
    }
    mknil(Vars);
}

/*
    ASSUMPTIONS: Term is any term, while Vars must be un unbound.
    EFFECTS: 	 Vars is bound to a list of the variables occurring in Term.
*/
bool pr_varsin()
{

    cellpo 	Term = &A[1],
    		Vars = &A[2];
    cell	temp;

    delnk(Term);

    vars_init(VARIABLES);
    heap_cells = 0;
    find_vars(Term);

    (void) gc_test((fourBytes)heap_cells,2);

    make_varlist(&temp);
    return(icp_unify(Vars, &temp));
}

to_copy(Term, Copy)
cellpo Term, Copy;
{
    cellpo	ptr, tuple;
    short	n;

    delnk(Term);

    switch (tg(Term)) {

	case var_ref:	{
	    varpo where;
	    if (insertv(Term, &where)) {
		alloc_cell(ptr);
		mkunb(ptr);
		mklnk(Copy, ptr);
		where->name = (symbpo) ptr;
	    }
	    else {
		mklnk(Copy, where->name);
	    }
	    break;
	}
	case list_ref:	{
	    alloc_list(Copy, ptr);
	    to_copy(hd(Term), ptr++);
	    to_copy(tl(Term), ptr);
	    break;
	}
	case tpl_ref: 	{
	    n = arity(Term);
	    tuple = functor(Term);
	    alloc_tpl(Copy, ptr, n);
	    while (n--)
	    	to_copy(tuple++, ptr++);
	    break;
	}
	default:
	    *Copy = *Term;
	    break;
    }
}

bool pr_copy()
{
    cellpo 	Term = &A[1],
		Copy = &A[2];
    cell	temp;

    delnk(Term);

    /* calculate heap space requirement */
    heap_cells = 0;
    ground_term = FALSE;
    vars_init(VARIABLES);
    traverse_g(Term);

    if (gc_test((fourBytes) heap_cells, 2)) {
    	Term = &A[1];
    	delnk(Term);
    }

    /* construct the copy */
    vars_init(VARIABLES);
    to_copy(Term, &temp);

    return(icp_unify(Copy, &temp));
}
