/* Copyright (C) 1992 Imperial College */
/*
   operators.c - a suite of functions/primitives to handle
   operators in IC-Prolog ][

   Written by Frank McCabe, Philip Schwarz and Damian Chu
   Imperial College, Winter 1989

   Modifications :
   21/3/90	dac
		fixed bug in next_free to catch table full errors.

   19/3/90	pjs
		pr_op_prefix, pr_op_postfix, and pr_op_infix now throw error 505,
		when the operator table fills up.
		Currently this error is not being caught anywhere.
*/

#include "primitives.h"
#include "termio.h"

/* size of operator hash table */
#define OP_ADDR_SIZE	89
#define OP_TABLE_SIZE	100

/* The operator table is pre-loaded with 48 operators.
   The table can hold a maximum of 100 operators.
   See termio.h for definition of fields in table. */
operator optable[OP_TABLE_SIZE] = {
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { ":",              0x1003a, 2000, 2000,  599,  600,  600, 2000, 2000,   0 },
  { ";",              0x1003b, 2000, 2000, 1099, 1100, 1100, 2000, 2000,   0 },
  { "<",              0x1003c, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "=",              0x1003d, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { ">",              0x1003e, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "<<",             0x20078, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "=<",             0x20079, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "==",             0x2007a, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "=:=",            0x300b4, 2000, 2000,  699,  700,  699, 2000, 2000,  97 },
  { "@<",             0x2007c, 2000, 2000,  699,  700,  699, 2000, 2000,  93 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "@>",             0x2007e, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "multifile",      0x903cb, 1150, 1149, 2000, 2000, 2000, 2000, 2000,   0 },
  { "@=<",            0x300b9, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "mode",           0x401a5, 1150, 1149, 2000, 2000, 2000, 2000, 2000,   0 },
  { "@>=",            0x300bb, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "nospy",          0x50239,  900,  900, 2000, 2000, 2000, 2000, 2000,  99 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "\\+",            0x20087,  900,  900, 2000, 2000, 2000, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "\\/",            0x2008b, 2000, 2000,  500,  500,  499, 2000, 2000,  94 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "public",         0x6027f, 1150, 1149, 2000, 2000, 2000, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "\\",             0x1005c,  500,  499, 2000, 2000, 2000, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "^",              0x1005e, 2000, 2000,  199,  200,  200, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "\\=",            0x20099, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "meta_predicate", 0xe05b7, 1150, 1149, 2000, 2000, 2000, 2000, 2000,   0 },
  { "\\==",           0x300d6, 2000, 2000,  699,  700,  699, 2000, 2000,  98 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { ". ",             0x2004e, 2000, 2000, 1200, 1200, 1200, 1200, 1200,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "**",             0x20054, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "mod",            0x30140, 2000, 2000,  299,  300,  299, 2000, 2000,   0 },
  { "++",             0x20056, 2000, 2000,  500,  500,  499, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "--",             0x2005a, 2000, 2000,  500,  500,  499, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "//",             0x2005e, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "-->",            0x30098, 2000, 2000, 1199, 1200, 1199, 2000, 2000,   0 },
  { "=..",            0x30099, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "*",              0x1002a, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "dynamic",        0x702e5, 1150, 1149, 2000, 2000, 2000, 2000, 2000,  96 },
  { ",",              0x1002c, 2000, 2000,  999, 1000, 1000, 2000, 2000,   0 },
  { "not",            0x30151,  900,  900, 2000, 2000, 2000, 2000, 2000,  95 },
  { ":-",             0x20067, 1200, 1199, 1199, 1200, 1199, 2000, 2000,   0 },
  { "/",              0x1002f, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "->",             0x2006b, 2000, 2000, 1049, 1050, 1050, 2000, 2000,   0 },
  { "?-",             0x2006c, 1200, 1199, 2000, 2000, 2000, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "spy",            0x3015c,  900,  900, 2000, 2000, 2000, 2000, 2000,   0 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { "",               0x00000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,  -1 },
  { ">>",             0x2007c, 2000, 2000,  400,  400,  399, 2000, 2000,   0 },
  { "/\\",            0x2008b, 2000, 2000,  500,  500,  499, 2000, 2000,   0 },
  { "-",              0x1002d,  500,  499,  500,  500,  499, 2000, 2000,   0 },
  { "+",              0x1002b,  500,  499,  500,  500,  499, 2000, 2000,   0 },
  { ">=",             0x2007b, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "=\\=",           0x300d6, 2000, 2000,  699,  700,  699, 2000, 2000,   0 },
  { "is",             0x200dc, 2000, 2000,  699,  700,  699, 2000, 2000,   0 }
};

/* global table pointers */
static
operatorpo	top_op = optable + OP_TABLE_SIZE - 1,
		next_op = optable + OP_TABLE_SIZE - 1;

static
jmp_buf		errjmp;



/*------------------------------------------------------------*
 *	    U T I L I T Y     F U N C T I O N S               *
 *------------------------------------------------------------*/

/* retrieve operator info from table */
void
retrieve_op(op,xr,ox,xor,lxr,lox,lx,xo)
operatorpo	op;
cellpo		xr,ox,xor,lxr,lox,lx,xo;
{
    delnk(xr);
    delnk(ox);
    delnk(xor);
    delnk(lxr);
    delnk(lox);
    delnk(lx);
    delnk(xo);
    mkreset(xr);
    mkreset(ox);
    mkreset(xor);
    mkreset(lxr);
    mkreset(lox);
    mkreset(lx);
    mkreset(xo);
    mkint(xr,  op->prefixp);
    mkint(ox,  op->preform);
    mkint(xor, op->infleft);
    mkint(lxr, op->infixp);
    mkint(lox, op->infright);
    mkint(lx,  op->postfixp);
    mkint(xo,  op->postleft);
}

/* find the table entry for a given operator */
operatorpo
oplook(name,hshcde,len,opr)
char		*name;
fourBytes	hshcde;
utwoBytes	len;
operatorpo	*opr;
{
    register
    operatorpo op = optable + (hshcde % OP_ADDR_SIZE) + 1;

    /* follow chain until found or end of chain */
    for (;;) {
	if (op->hsh==hshcde && !strncmp(op->op,name,(int)len))	/* found */
	    return(op);
	if (op->lnk <= 0) {	/* end of chain */
	    *opr = op;
	    return(NULL);
	}
	op = optable + op->lnk;
    }
}

/* locate next free slot in table */
operatorpo
next_free(o)
operatorpo	o;
{
    while (next_op->lnk != FREE && next_op > optable)
	next_op--;
    if (next_op->lnk != FREE)
	longjmp(errjmp, 505);
    o->lnk = next_op - optable;
    return(next_op);
}

/* enter new operator into table */
operatorpo
op_enter(o)
symbpo	o;
{
    operatorpo	op, op2;

    if (!(op=oplook(symbname(o),symbhashval(o),symblngth(o),&op2))) {
	op = op2;
	if (op->lnk != FREE)
	    op = next_free(op);
	(void)strcpy(op->op,symbname(o));
	op->hsh = symbhashval(o);
	op->lnk = 0;
    }
    return(op);
}

/*------------------------------------------------------------*/

/* (re)define prefix operator */
bool
pr_op_prefix()
{
    cellpo	xr = &A[1],
		ox = &A[2],
		o  = &A[3];

    operatorpo	op;
    int		errcode;

    delnk(o);

    if (errcode = setjmp(errjmp)) {
	throw(errcode);
    }
    else {
	op = op_enter(symbvl(o));
	delnk(xr);

	if (op->prefixp=intvl(xr)) {
	    delnk(ox);
	    op->preform = intvl(ox);
	}
	else {	/* undefine operator */
	    op->prefixp = undefprior;
	    op->preform = undefprior;
	    if (op->infixp == undefprior && op->postfixp == undefprior)
		op->op[0] = '\0';
	}
	return(SUCCEED);
    }
}

/*------------------------------------------------------------*/

/* (re)define postfix operator */
bool
pr_op_postfix()
{
    cellpo	xo = &A[1],
		lx = &A[2],
		o  = &A[3];

    operatorpo	op;
    int		errcode;

    delnk(o);

    if (errcode = setjmp(errjmp)) {
	throw(errcode);
    }
    else {
	op = op_enter(symbvl(o));
	delnk(lx);

	if (op->postfixp=intvl(lx)) {
	    delnk(xo);
	    op->postleft = intvl(xo);
	}
	else {	/* undefine operator */
	    op->postfixp = undefprior;
	    op->postleft = undefprior;
	    if (op->prefixp == undefprior && op->infixp == undefprior)
		op->op[0] = '\0';
	}
	return(SUCCEED);
    }
}

/*------------------------------------------------------------*/

/* (re)define infix operator */
bool
pr_op_infix()
{
    cellpo	xor = &A[1],
		lxr = &A[2],
		lox = &A[3],
		o   = &A[4];

    operatorpo	op;
    int		errcode;

    delnk(o);

    if (errcode = setjmp(errjmp)) {
	throw(errcode);
    }
    else {
	op = op_enter(symbvl(o));
	delnk(lxr);

	if (op->infixp=intvl(lxr)) {
	    delnk(xor);
	    delnk(lox);
	    op->infleft  = intvl(xor);
	    op->infright = intvl(lox);
	}
	else {	/* undefine operator */
	    op->infleft  = undefprior;
	    op->infixp   = undefprior;
	    op->infright = undefprior;
	    if (op->prefixp == undefprior && op->postfixp == undefprior)
		op->op[0] = '\0';
	}
	return(SUCCEED);
    }

}

/*------------------------------------------------------------*/

/* retrieve info for a specific operator */
bool
pr_op_look()
{
    cellpo	o   = &A[1],
		xr  = &A[2],
		ox  = &A[3],
		xor = &A[4],
		lxr = &A[5],
		lox = &A[6],
		lx  = &A[7],
		xo  = &A[8];

    operatorpo	op, dummy;
    symbpo	sym;

    delnk(o);
    sym = symbvl(o);

    if (op=oplook(symbname(sym),symbhashval(sym),symblngth(sym),&dummy)) {
        retrieve_op(op,xr,ox,xor,lxr,lox,lx,xo);
	return(SUCCEED);
    }
    return(FAIL);
}

/*------------------------------------------------------------*/

/* find next operator in table, starting from table index n */
bool
pr_op_get()
{
    register
    operatorpo	op;

    cellpo   /* o   = &A[1], not used in this function */
		n   = &A[2],
		xr  = &A[3],
		ox  = &A[4],
		xor = &A[5],
		lxr = &A[6],
		lox = &A[7],
		lx  = &A[8],
		xo  = &A[9],
		m   = &A[10];

    delnk(n);
    op  = &optable[intvl(n)+1];

    /* skip undefined slots in table */
    while (op <= top_op && op->lnk < 0)
	op++;

    if (op <= top_op) {
	retrieve_op(op,xr,ox,xor,lxr,lox,lx,xo);
	delnk(m);
	mkreset(m);
	mkint(m, op - optable);

	(void) bind_symbol(1, op->op, 10);
	return(SUCCEED);
    }
    return(FAIL);
}

/*------------------------------------------------------------*/

/*
    Tests to see if a string is an operator.
    Returns the corresponding entry in the table.
*/

operatorpo
is_op(tok)
token *tok;
{
    switch(tok->tt) {
	case solo:
	case graph:
	case lower:
	case quoted:
	case comma:
	case semicolon: {
	    register
	    strpo	s	= tok->buff;
	    fourBytes	hshcde	= 0,
			len;
	    operatorpo	dummy;

	    while(*s)
		hshcde += *s++;
	    len = strlen(tok->buff);
	    hshcde = (hshcde & 0x0FFFF) | len << 16;

	    return(oplook(tok->buff,hshcde,(utwoBytes)len,&dummy));
	}

	default:		/* by default a token is not an operator */
	    return(NULL);
    }
}
