/* Copyright (C) 1992 Imperial College */
/*
 * Support for Property Management in IC Prolog ][
 */

#include "primitives.h"

extern void	init_encode();
extern strpo	encode_prop(), decode_prop(), encode_str_prop();
extern bool	samesymb();
extern bool	icp_unify();

/* property database */
/* WARNING : do not change NUMPROPS without looking at the code !! */
#define NUMPROPS	0x1000
#define PROPSIZE	10000

typedef struct proprec {
    cell	object;
    cell	property;
    strpo	value;
    struct
    proprec	*next;
} prop_rec, *propptr;

static	propptr	buf;

static	propptr props[NUMPROPS];

/*----------------------------------------------------------------------*/

bool
init_props()
{
    buf = (propptr)alloc((size_t)PROPSIZE, 1);
    return(buf ? SUCCEED : FAIL);
}

static utwoBytes
hash(obj, prop)
cellpo	obj, prop;
{
    register utwoBytes h1, h2;
    h1 = (IsInt(obj))  ? intvl(obj) & 0x003F
			       : string_hash(obj) & 0x003F;
    h2 = (IsInt(prop)) ? intvl(prop) & 0x003F
			       : string_hash(prop) & 0x003F;
    return((h1 << 6) | h2);
}

bool
same(first, second)
cellpo	first, second;
{
    if (sameTag(first, second)) {
	if (IsInt(first))
	    return(intvl(first) == intvl(second));
	else return(samesymb(symbvl(first), symbvl(second)));
    }
    else return(FAIL);
}

propptr
find_prop(initial, obj, prop)
propptr	initial;
cellpo	obj, prop;
{
    if (initial == NULL)
	return(NULL);

    while (!(same(&initial->object,obj) && same(&initial->property,prop)))
	if (!(initial=initial->next))
	    break;
    return(initial);
}

propptr
str_find_prop(initial, obj, prop)
propptr	initial;
strpo	obj, prop;
{
	if (initial == NULL)
		return(NULL);

	while (initial) {
		if (IsSymb(&initial->object) && IsSymb(&initial->property) &&
			!strcmp(string_val(&initial->object), obj) &&
			!strcmp(string_val(&initial->property), prop))
			break;
		initial=initial->next;
	}
	return(initial);
}


find_props(initial, obj, list)
propptr	initial;
cellpo	*obj, *list;
{
    cell tempcell;
    cellpo temp;

    while (initial) {
	if (same(*obj, &initial->object)) {

	    /* need to re-access arguments if g/c was called */
	    if (gc_test(2L, 2)) {
		*obj  = &A[1];
		*list = &A[2];
		delnk(*obj);
		delnk(*list);
	    }

	    tempcell = **list;
	    alloc_list(*list, temp);
	    *temp++ = initial->property;
	    *temp = tempcell;
	}
	initial = initial->next;
    }
}

find_cons(initial, prop, list)
propptr	initial;
cellpo	*prop, *list;
{
    cell tempcell;
    cellpo temp;

    while (initial) {
	if (same(*prop, &initial->property)) {

	    /* need to re-access arguments if g/c was called */
	    if (gc_test(2L, 2)) {
		*prop = &A[1];
		*list = &A[2];
		delnk(*prop);
		delnk(*list);
	    }

	    tempcell = **list;
	    alloc_list(*list, temp);
	    *temp++ = initial->object;
	    *temp = tempcell;
	}
	initial = initial->next;
    }
}

del_props(index, obj)
utwoBytes index;
cellpo	obj;
{
    propptr back = props[index],
	    temp = back,
	    found;

    while (temp) {
	if (same(obj, &temp->object)) {
	    if (temp == props[index]) {
		props[index] = temp->next;
		back = props[index];
	    }
	    else back->next = temp->next;
	    found = temp;
	    temp = temp->next;
	    free((char *)found);
	}
	else {
	    back = temp;
	    temp = temp->next;
	}
    }
}

del_cons(index, prop)
utwoBytes index;
cellpo	prop;
{
    propptr back = props[index],
	    temp = back,
	    found;

    while (temp) {
	if (same(prop, &temp->property)) {
	    if (temp == props[index]) {
		props[index] = temp->next;
		back = props[index];
	    }
	    else back->next = temp->next;
	    found = temp;
	    temp = temp->next;
	    free((char *)found);
	}
	else {
	    back = temp;
	    temp = temp->next;
	}
    }
}

/*----------------------------------------------------------------------*/

bool
pr_set_prop()
{
    register
    cellpo	obj   = &A[1],
		prop  = &A[2],
		value = &A[3];
    utwoBytes	pos, s_len;
    propptr	initial, new_prop, found;
    twoBytes	len;
    strpo	new, out;
    symbpo	symb;


    delnk(obj);
    delnk(prop);
    delnk(value);

    if (  (NotInt(obj) && NotSymb(obj)) ||
	  (NotInt(prop) && NotSymb(prop)) )
	throw(210);

    /* allocate space for property record AND the strings */
    new = (strpo)buf;
    new += sizeof(prop_rec);	/* skip over the record */

    if (IsInt(obj)) {
	mkint(&buf->object, intvl(obj));
    }
    else {
	symb = symbvl(obj);
	s_len = (symblngth(symb) + 2) & ~01;	/* ensure even number */

	/* align to 4 byte boundary */
	new = (strpo)(((ufourBytes)new + 3) & ~03);

	mksymb(&buf->object, new-(strpo)buf);	/* an offset */
#ifdef GNUDOS
	symbhashvalin((symbpo)new) = symbhashval(symb);
#else
	symbhashval((symbpo)new) = symbhashval(symb);
#endif
	new = symbname((symbpo)new);
	out = symbname(symb);
	while (s_len--)
	    *new++ = *out++;
    }

    if (IsInt(prop)) {
	mkint(&buf->property, intvl(prop));
    }
    else {
	symb = symbvl(prop);
	s_len = (symblngth(symb) + 2) & ~01;	/* ensure even number */

	/* align to 4 byte boundary */
	new = (strpo)(((ufourBytes)new + 3) & ~03);

	mksymb(&buf->property, new-(strpo)buf); /* an offset */
#ifdef GNUDOS
	symbhashvalin((symbpo)new) = symbhashval(symb);
#else
	symbhashval((symbpo)new) = symbhashval(symb);
#endif
	new = symbname((symbpo)new);
	out = symbname(symb);
	while (s_len--)
	    *new++ = *out++;
    }

    buf->value = (strpo)(new-(strpo)buf);	/* lastly, the value in encoded form */
    out = new;
    init_encode();
    out = encode_prop(value, out);
    len = out - (strpo)buf;

    if (len > PROPSIZE)		/* property too big */
	throw(509);

    new_prop = (propptr)malloc((size_t)len);
    if (!new_prop)	/* no system heap space */
	throw(611);

    (void)memcpy((char *)new_prop, (char *)buf, (SIZE_TYPE)len);

    /* re-adjust offsets */
    if (IsSymb(&new_prop->object))
	setval(&new_prop->object, ((strpo)new_prop + (int)vl(&new_prop->object)));
    if (IsSymb(&new_prop->property))
	setval(&new_prop->property, ((strpo)new_prop + (int)vl(&new_prop->property)));
    new_prop->value = (strpo)((strpo)new_prop + (int)new_prop->value);

    /* now link the new property into the database */
    pos = hash(obj, prop);
    initial = props[pos];

    if (initial == NULL) {
	new_prop->next = NULL;
	props[pos] = new_prop;
    }
    else {
	if (found=find_prop(initial, obj, prop)) {
	    /* property already exists - remove old value */
	    if (found == initial) {
		new_prop->next = found->next;
		props[pos] = new_prop;
	    }
	    else {
		while (initial->next != found)
		    initial = initial->next;
		new_prop->next = found->next;
		initial->next = new_prop;
	    }
	    free((char *)found);
	}
	else {
	    new_prop->next = initial;
	    props[pos] = new_prop;
	}
    }

    return(SUCCEED);
}

bool
set_str_prop(obj, prop, values)
strpo obj, prop, *values;
{
	utwoBytes	pos, s_len;
	propptr	initial, new_prop, found;
	twoBytes	len;
	strpo	new, out;

	/* allocate space for property record AND the strings */
	new = (strpo)buf;
	new += sizeof(prop_rec);	/* skip over the record */

	s_len = strlen(obj);
	/* align to 4 byte boundary */
	new = (strpo)(((ufourBytes)new + 3) & ~03);
	mksymb(&buf->object, new-(strpo)buf);	/* an offset */
	strsymb(new, s_len, obj);
	new = symbname((symbpo)new);
	new += (s_len + 2) & ~01;	/* ensure even number */

	s_len = strlen(prop);
	/* align to 4 byte boundary */
	new = (strpo)(((ufourBytes)new + 3) & ~03);
	mksymb(&buf->property, new-(strpo)buf);	/* an offset */
	strsymb(new, s_len, prop);
	new = symbname((symbpo)new);
	new += (s_len + 2) & ~01;	/* ensure even number */

	buf->value = (strpo)(new-(strpo)buf);	/* lastly, the value in encoded form */
	out = new;
	init_encode();
	out = encode_str_prop(values, out);
	len = out - (strpo)buf;

	if (len > PROPSIZE)		/* property too big */
		throw(509);

	new_prop = (propptr)malloc((size_t)len);
	if (!new_prop)	/* no system heap space */
		throw(611);

	(void)memcpy((char *)new_prop, (char *)buf, (SIZE_TYPE)len);

	/* re-adjust offsets */
	if (IsSymb(&new_prop->object))
		setval(&new_prop->object, ((strpo)new_prop + (int)vl(&new_prop->object)));
	if (IsSymb(&new_prop->property))
		setval(&new_prop->property, ((strpo)new_prop + (int)vl(&new_prop->property)));
	new_prop->value = (strpo)((strpo)new_prop + (int)new_prop->value);

	/* now link the new property into the database */
	pos = hash(&new_prop->object, &new_prop->property);
	initial = props[pos];

	if (initial == NULL) {
		new_prop->next = NULL;
		props[pos] = new_prop;
	} else {
		if (found=str_find_prop(initial, obj, prop)) {
			/* property already exists - remove old value */
			if (found == initial) {
				new_prop->next = found->next;
				props[pos] = new_prop;
			} else {
				while (initial->next != found)
					initial = initial->next;
				new_prop->next = found->next;
				initial->next = new_prop;
			}
			free((char *)found);
		} else {
			new_prop->next = initial;
			props[pos] = new_prop;
		}
	}

    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

bool
pr_get_prop()
{
    register
    cellpo	obj   = &A[1],
		prop  = &A[2],
		value = &A[3];
    propptr	initial;
    cellpo	v;

    delnk(obj);
    delnk(prop);
    delnk(value);

    if (   (NotInt(obj) && NotSymb(obj)) ||
	   (NotInt(prop) && NotSymb(prop)) )
	throw(210);

    initial = props[hash(obj, prop)];

    if (initial=find_prop(initial, obj, prop)) {
	if (IsVar(value)) {
	    mkreset(value);
	    (void)decode_prop(value, initial->value);
	    return(SUCCEED);
	}
	else {
	    alloc_cell(v);
	    (void)decode_prop(v, initial->value);
	    return(icp_unify(value, v));
	}
    }
    else return(FAIL);
}

strpo *str_get_prop(name, object)
strpo name, object;
{
	extern strpo *decode_str_prop();
	cell obj, prop;
	propptr	initial;

	alloc_symb(&obj, strlen(name), name);
	alloc_symb(&prop, strlen(object), object);

	initial = find_prop(props[hash(&obj, &prop)], &obj, &prop);
	if (!initial)
		return((strpo *) NULL);
	
	return(decode_str_prop(initial->value));
}

/*----------------------------------------------------------------------*/

bool
pr_del_prop()
{
    register
    cellpo	obj  = &A[1],
		prop = &A[2];
    utwoBytes	pos;
    propptr	initial, found;

    delnk(obj);
    delnk(prop);

    if (   (NotInt(obj) && NotSymb(obj)) ||
	   (NotInt(prop) && NotSymb(prop)) )
	throw(210);

    pos = hash(obj, prop);
    initial = props[pos];
    if (found=find_prop(initial, obj, prop)) {
	if (found == initial)
	    props[pos] = found->next;
	else {
	    while (initial->next != found)
		initial = initial->next;
	    initial->next = found->next;
	}
	free((char *)found);
    }
    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

bool
pr_get_props()
{
    cellpo	obj	= &A[1],
		list	= &A[2];
    propptr	initial;
    utwoBytes	hsh;
    int		i;

    delnk(obj);
    delnk(list);

    if ((NotInt(obj) && NotSymb(obj)) || (NotVar(list)))
	throw(210);

    /* initialise the list */
    mkreset(list);
    mknil(list);

    hsh = (IsInt(obj))  ? intvl(obj) & 0x003F
				: string_hash(obj) & 0x003F;
    hsh <<= 6;

    for (i=0; i<0x40; i++) {
	initial = props[hsh+i];
	find_props(initial, &obj, &list);
    }

    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

bool
pr_get_cons()
{
    cellpo	prop	= &A[1],
		list	= &A[2];
    propptr	initial;
    utwoBytes	hsh;
    int		i;

    delnk(prop);
    delnk(list);

    if ((NotInt(prop) && NotSymb(prop)) || (NotVar(list)))
	throw(210);

    /* initialise the list */
    mkreset(list);
    mknil(list);

    hsh = (IsInt(prop))  ? intvl(prop) & 0x003F
				 : string_hash(prop) & 0x003F;

    for (i=0; i<0x1000; i+=0x40) {
	initial = props[hsh+i];
	find_cons(initial, &prop, &list);
    }

    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

bool
pr_del_props()
{
    cellpo	obj	= &A[1];
    utwoBytes	hsh;
    int		i;

    delnk(obj);

    if (NotInt(obj) && NotSymb(obj))
	throw(210);

    hsh = (IsInt(obj))  ? intvl(obj) & 0x003F
				: string_hash(obj) & 0x003F;
    hsh <<= 6;

    for (i=0; i<0x40; i++) {
	del_props(hsh+i, obj);
    }

    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

bool
pr_del_cons()
{
    cellpo	prop	= &A[1];
    utwoBytes	hsh;
    int		i;

    delnk(prop);

    if (NotInt(prop) && NotSymb(prop))
	throw(210);

    hsh = (IsInt(prop))  ? intvl(prop) & 0x003F
				 : string_hash(prop) & 0x003F;

    for (i=0; i<0x1000; i+=0x40) {
	del_cons(hsh+i, prop);
    }

    return(SUCCEED);
}

/*----------------------------------------------------------------------*/

/* temporary hack to allow Parlog to read port numbers set by Prolog */
#define MAXPORT	2
int	ports[MAXPORT];

bool pr_set_port()
{
    cellpo	index	= &A[1],
		port	= &A[2];

    delnk(index);
    delnk(port);
    if (NotInt(index) || NotInt(port) || intvl(index) >= MAXPORT)
	return(FAIL);

    ports[intvl(index)] = intvl(port);
    return(SUCCEED);
}

bool pr_get_port()
{
    cellpo	index	= &A[1],
		port	= &A[2];

    delnk(index);
    delnk(port);
    if (NotInt(index) || NotVar(port) || intvl(index) >= MAXPORT)
	return(FAIL);

    mkreset(port);
    mkint(port, ports[intvl(index)]);
    return(SUCCEED);
}
