/* Copyright (C) 1992 Imperial College */
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/times.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <netdb.h>
#include <setjmp.h>

#define TRUE		1
#define FALSE		0
#define TBLOCK		-1	
#define TPOLL		-2
#define FDSIZE		64
#define TCPBUFSIZE	4096
#define BRBUFSIZE	1024

#define	SINIT		0x00
#define SOPEN		0x01
#define SBIND		0x02
#define SLISTEN		0x04
#define SCONNECT	0x08
#define SCONNECTION	0x10
#define SCONNLESS	0x20
#define SCLOSED		0x40

#define MODE		0xF0
#define ST		0x0F

#define NORMAL_DATA	1
#define RAW_DATA	2
#define PEEK_DATA	3
#define PEEK_RAW_DATA	4

#define MAX_STRING	2048

/* descriptor for a socket */
typedef struct {
	int		(*funct)();
	char		*buf;
	int		buf_len;
	struct sockaddr_in add;
	char		*start, *end, *pos;
	int		peek;
	char		lock;
	u_short		status;
	int		fp;
} socket_desc, *socketpo;

char write_buffer[TCPBUFSIZE+1];

struct {
	char		*buf;
	int		length;
	u_short		port;
	u_long		machine;
	int		family;
	int		fp;
} write_socket;

#define socket_ident(Socket)    (Socket - sockets)

#define init_socket_list	{					\
	int i;								\
	for (i=0; i < FDSIZE; i++)					\
		sockets[i].status = SINIT;				\
}

static socketpo read_socket = NULL;

socket_desc sockets[FDSIZE];
static  socketpo next_socket = sockets;

#define socket_status(socket)   socket->status
#define socket_fp(socket)       socket->fp
#define need_read(Sockid) (Sockid->end == Sockid->pos)
#define free_socket(socket)	\
	((socket->status == SCLOSED) || (socket->status == SINIT))
#define just_closed(socket)		(socket->status & SCLOSED)
#define open_allowed(socket)		(socket->status == SINIT)
#define bind_allowed(socket)		(socket->status & SOPEN)
#define listen_allowed(socket)	\
	((socket->status & SCONNECTION) && (socket->status & SBIND))

#define accept_allowed(socket)	\
	((socket->status & SCONNECTION) && (socket->status & SLISTEN))
#define connect_allowed(socket)	\
	((socket->status & (SCONNECTION | SCONNLESS)) &&	\
	(socket->status & (SOPEN | SBIND)))
#define send_allowed(socket)	\
	((socket->status & (SCONNECTION | SCONNLESS)) &&	\
	(socket->status & (SCONNECT)))
#define recv_allowed(socket)	\
	(((socket->status & (SCONNLESS)) &&			\
	(socket->status & (SOPEN | SBIND | SCONNECT))) ||	\
	((socket->status & (SCONNECTION)) &&			\
	(socket->status & (SCONNECT))))
#define sendto_allowed(socket)	\
	((socket->status & (SCONNECTION | SCONNLESS)) &&	\
	(socket->status & (SOPEN | SBIND | SCONNECT)))
#define recvfrom_allowed(socket)	\
	((socket->status & (SBIND)) ||				\
	((socket->status & (SCONNECTION | SCONNLESS)) &&	\
	(socket->status & (SOPEN | SCONNECT))))
#define sendbr_allowed(socket)	\
	(is_broadcast(socket) && (socket->status & (SBIND)))
#define close_allowed(socket)	(socket->status != SINIT)

#define set_open_c(socket)		(socket->status = (SCONNECTION | SOPEN))
#define set_open_cl(socket)		(socket->status = (SCONNLESS | SOPEN))
#define set_bind(socket)	\
	(socket->status = ((socket->status & MODE) | SBIND))
#define set_listen(socket)	\
	(socket->status = ((socket->status & MODE) | SLISTEN))
#define set_accept(sock1, sock2)	\
	(sock2->status = ((sock1->status & MODE) | SCONNECT))
#define set_connect(socket)	\
	(socket->status = ((socket->status & MODE) | SCONNECT))
#define set_close(socket) {				\
	(socket->status = (SCLOSED));			\
	(void) free(socket->buf);			\
	if (socket < next_socket) next_socket=socket;}

#define is_connection(socket)		(socket->status & SCONNECTION)
#define is_connless(socket)		(socket->status & SCONNLESS)

#define reinit_read_buf(sockid)		\
{					\
	sockid->start = sockid->buf;	\
	sockid->end = sockid->buf;	\
	sockid->pos = sockid->buf;	\
}

#define fix_read_buf_after(sockid)			\
{							\
	if (sockid->peek)				\
		sockid->pos = sockid->start;		\
	else						\
		sockid->start = sockid->pos;		\
}

/****************************************************************/
/*			Global Variables			*/
/****************************************************************/
extern	fd_set	rfdset, wfdset, efdset;
extern	int	h_deadlock;

static jmp_buf write_error;

extern time_t time();
extern void endhostent();
extern void endservent();
extern void sethostent();
extern void setservent();

static int fix_read_buffer()
{
	int len;
	register char *pt1, *pt2;

	if (read_socket->peek) {
		len = read_socket->start - read_socket->buf;
		if (len > 0) {		/* shuffle data to start */
			pt1 = read_socket->buf;
			pt2 = read_socket->start;
			while (pt2 < read_socket->end)
				*pt1++ = *pt2++;
			read_socket->start = read_socket->buf;
			read_socket->end   = pt1;
		}
		len = read_socket->end - read_socket->buf;
		if (len > (read_socket->buf_len / 2)) {
			char *old;

			old=read_socket->buf;
			read_socket->buf_len *= 2;
			read_socket->buf = malloc((unsigned) read_socket->buf_len + 2);
			pt1 = read_socket->buf;
			pt2 = old;
			while (pt2 < read_socket->end)
				*pt1++ = *pt2++;
			read_socket->start = read_socket->buf;
			read_socket->end = pt1;
			(void) free(old);
		}
		read_socket->pos = read_socket->end;
		len = read_socket->buf_len - len;
	} else {
		reinit_read_buf(read_socket);
		len = read_socket->buf_len;
	}
	return(len);
}

static int buffered_recv()
{
	int len;

	len = fix_read_buffer();
	len = recv(read_socket->fp, read_socket->pos, len, 0);
	if (len == -1)
		return(-1);
	read_socket->end += len;
	return(len);
}

static int buffered_recvfrom()
{
	int len, length = sizeof(read_socket->add);

	len = fix_read_buffer();
	len = recvfrom(read_socket->fp, read_socket->pos, len, 0,
			(struct sockaddr *) &(read_socket->add),
			&length);
	if (len == -1)
		return(-1);
	read_socket->end += len;
	return(len);
}

static int buffered_read()
{
	if (need_read(read_socket)) {
		if ((read_socket->funct)() <= 0)	/* 0=EOF, -1=error */
			return(EOF);
	}
	return(*(read_socket->pos)++);
}

static int buffered_send()
{
	int len;

	len = send(write_socket.fp, write_socket.buf, write_socket.length, 0);
	if (len < 0)
		return(717);
	write_socket.length = 0;
	return(0);
}

static int buffered_sendto()
{
	struct sockaddr_in add;

	add.sin_family = write_socket.family;
	add.sin_addr.s_addr = htonl(write_socket.machine);
	add.sin_port = htons(write_socket.port);

	if (sendto(write_socket.fp, write_socket.buf, write_socket.length, 0,
			(struct sockaddr *)&add, sizeof(add)) < 0)
		return(719);
	write_socket.length = 0;
	return(0);
}

static int buffered_sendbr()
{
	struct ifconf ifc;
	char buf[BRBUFSIZE];
	struct ifreq *ifr;
	struct sockaddr_in dst;
	int n;

	bzero((char *)&dst, sizeof(struct sockaddr_in));
	dst.sin_family = AF_INET;
	dst.sin_port = htons(write_socket.port);

	ifc.ifc_len = sizeof buf;
	ifc.ifc_buf = buf;
	if (ioctl(write_socket.fp, SIOCGIFCONF, (char *)&ifc) < 0) {
		return(716);
	}
	ifr = ifc.ifc_req;
	for (n = (ifc.ifc_len/sizeof(struct ifreq)); --n >= 0; ifr++) {
		if (ifr->ifr_addr.sa_family != AF_INET)
			continue;
		if (ioctl(write_socket.fp, SIOCGIFFLAGS, (char *)ifr) < 0) {
			return(716);
		}
		if (!((ifr->ifr_flags & IFF_UP) &&
					(ifr->ifr_flags & IFF_BROADCAST)))
			continue;
		if (ioctl(write_socket.fp, SIOCGIFBRDADDR, (char *)ifr) < 0) {
			return(716);
		}
		dst.sin_addr.s_addr = ((struct sockaddr_in *)&ifr->ifr_broadaddr)->sin_addr.s_addr;
		if (sendto(write_socket.fp,write_socket.buf,
				write_socket.length, 0,
				(struct sockaddr *)&dst, sizeof(dst)) < 0)
			return(719);
	}
	write_socket.length = 0;
	return(0);
}

static socketpo getNextSocket()
{
	socketpo start;

	start = next_socket;
	while (!free_socket(next_socket) && (++next_socket != start)) {
		if (next_socket == sockets + FDSIZE)
			next_socket = sockets;
	}

	if (free_socket(next_socket)) {
		next_socket->buf = malloc(TCPBUFSIZE+2);
		next_socket->buf_len = TCPBUFSIZE;
		reinit_read_buf(next_socket);
		return(next_socket);
	} else
		return(NULL);
}

static int is_broadcast(sockid)
socketpo sockid;
{
	int val;
	int len = sizeof(val);

	if (getsockopt(socket_fp(sockid),SOL_SOCKET,SO_BROADCAST,(char *)&val,
			&len))
		return(FALSE);
	return(val != 0);
}

static int writing_allowed(sockid)
socketpo sockid;
{
	fd_set	write_mask;
	struct timeval timeout;
	int fp;

	fp = socket_fp(sockid);
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	FD_ZERO(&write_mask);
	FD_SET(fp, &write_mask);
	FD_SET(fp, &wfdset);
	update_select_width(fp);
	if (select(fp+1,(fd_set *)NULL,&write_mask,(fd_set *)NULL,&timeout)<0)
		return(FALSE);
	if (FD_ISSET(fp, &write_mask))
		return(TRUE);
	return(FALSE);
}

static int reading_allowed(sockid)
socketpo sockid;
{
	fd_set	read_mask;
	struct timeval timeout;
	int fp;

	if (!need_read(sockid))
		return(TRUE);
	fp = socket_fp(sockid);
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	FD_ZERO(&read_mask);
	FD_SET(fp, &read_mask);
	FD_SET(fp, &rfdset);
	update_select_width(fp);
	if (select(fp+1,&read_mask,(fd_set *)NULL,(fd_set *)NULL,&timeout) < 0)
		return(FALSE);
	if (FD_ISSET(fp, &read_mask))
		return(TRUE);
	return(FALSE);
}

static void tcp_exit1()
{
	socketpo start = sockets;

	for (start=sockets; start <= next_socket; start++) {
		if (start->status & SOPEN)
			(void) close(start->fp);
	}
}
