/* Copyright (C) 1992 Imperial College */
/****************************************************************/
/* 	Prolog Primitives to support TCP comunication		*/
/****************************************************************/

	/************************************************/
	/* 		Implementation 			*/
	/************************************************/

#include "primitives.h"

#include "tcp.h"
#include "select.h"

CHARTYPE	(*charin)();	/* read one character  */
bool		(*charout)();	/* write one character */

#define get_socket(Arg, Socket)	{	\
	int Index = intvl(Arg);		\
	if (Index >= FDSIZE)		\
		throw(703);		\
	Socket = sockets + Index;}

static int prolog_throw;

/****************************************************************/
/*		Functions Declarations				*/
/****************************************************************/

bool pr_tcp_init();
bool pr_socket();
bool pr_setsockopt();
bool pr_bind();
bool pr_listen();
bool pr_accept();
bool pr_connect1();
bool pr_connect2();
bool pr_send();
bool pr_recv();
bool pr_sendto();
bool pr_recvfrom();
bool pr_sendbr();
bool pr_checkrecv();
bool pr_checkconn();

bool pr_tcp_close();
bool pr_lock_sock();

bool pr_getsockaddr();
bool pr_getpeeraddr();
bool pr_gethost();
bool pr_getport();

static bool buffered_write();
static bool tcp_read();
static int tcp_write();

/****************************************************************/
/*		Functions Definitions		(tcp.c)		*/
/****************************************************************/

/****************************************************************/
/*			Global Variables			*/
/****************************************************************/
extern	symbpo	eof_sym;

/****************************************************************/
/*		functions to support tcp primitives		*/
/****************************************************************/

bool pr_tcp_init()
{
	extern void (*tcp_exit) ();

	tcp_exit = tcp_exit1;
	init_socket_list;
	return(SUCCESS);
}

/****************************************************************/
/*	check whether the operation requested has timed out	*/
/****************************************************************/
static bool
check_time(reg)
cellpo reg;
{
	time_t ti;

	if (IsSymb(reg) && !strcmp(string_val(reg), "block")) {
		h_deadlock |= D_TCP;
		return(WAIT);
	}
	else if (IsSymb(reg) && !strcmp(string_val(reg), "poll")) 
		{ throw(702); }
	else if (IsFloat(reg))
		ti = (time_t) floatvl(reg);
	else
		throw(210);
	if (time((time_t *) NULL) < ti) {
		set_select_time(ti);
		h_deadlock |= D_TCP;
		return(WAIT);
	}
	throw(702);
}

/****************************************************************/
/* tcp_socket(+Type, +Protocol, -Socket)			*/
/*	Type: sock_stream, sock_dgram, sock_raw			*/
/*	Protocol: ipproto_ip, ipproto_udp, ipproto_tcp		*/
/*		  ipproto_icmp, ipproto_raw			*/
/*	socket(family, type, protocol)				*/
/* We have commited ourselves to AF_INET family which means the	*/
/* type = sock_dgram | sock_stream | sock_raw			*/
/* as far as protocols are concerned the following are allowed	*/
/*	any		-> ipproto_ip				*/
/*	sock_dgram	-> ipproto_udp				*/
/*	sock_stream	-> ipproto_tcp				*/
/*	sock_raw	-> ipproto_icmp | ipproto_raw		*/
/****************************************************************/
bool pr_socket()
{
	int type, protocol;
	strpo type_s, protocol_s;
	cellpo argT, argP, argS;
	socketpo sockid;

	argT = &A[1]; delnk(argT);
	argP = &A[2]; delnk(argP);
	argS = &A[3]; delnk(argS);

	if (!IsSymb(argT) || !IsSymb(argP) || !IsVar(argS))
		throw(210);

	type_s = string_val(argT);
	protocol_s = string_val(argP);

	if (!strcmp(type_s, "sock_stream"))
		type = SOCK_STREAM;
	else if (!strcmp(type_s, "sock_dgram"))
		type = SOCK_DGRAM;
	else if (!strcmp(type_s, "sock_raw"))
		type = SOCK_RAW;
	else
		throw(210);

	if (!strcmp(protocol_s, "ipproto_ip"))
		protocol = IPPROTO_IP;
	else if (!strcmp(protocol_s, "ipproto_udp"))
		protocol = IPPROTO_UDP;
	else if (!strcmp(protocol_s, "ipproto_tcp"))
		protocol = IPPROTO_TCP;
	else if (!strcmp(protocol_s, "ipproto_icmp"))
		protocol = IPPROTO_ICMP;
	else if (!strcmp(protocol_s, "ipproto_raw"))
		protocol = IPPROTO_RAW;
	else
		throw(210);

	if (!(sockid = getNextSocket()))
		throw(730);

	if ((socket_fp(sockid) = socket(AF_INET, type, protocol)) < 0)
		throw(710);
	
	if (type = SOCK_STREAM)
		set_open_c(sockid);
	else
		set_open_cl(sockid);

	mkreset(argS);
	mkint(argS, (int) socket_ident(sockid));
	return(SUCCESS);
}

/************************************************************************/
/* tcp_setsockopt(+Socket, +Optname, +Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_setsockopt(Socket Opt, Val) maps onto the C calls		*/
/*	int val = Val, opt = Opt, sock = Sock;				*/
/*	setsockopt(sock, SOL_SOCKET, opt, (char *)&val, sizeof(val));	*/
/************************************************************************/
bool pr_setsockopt()
{
	cellpo argS, argO, argV;
	int opt, val;
	strpo optname;
	socketpo sockid;

	argS = &A[1]; delnk(argS);
	argO = &A[2]; delnk(argO);
	argV = &A[3]; delnk(argV);

	if (!IsInt(argS) || !IsSymb(argO) || !IsInt(argV))
		throw(210);

	get_socket(argS, sockid);
	optname = string_val(argO);
	val = intvl(argV);

	if (!strcmp(optname, "so_debug"))
		opt = SO_DEBUG;
	else if (!strcmp(optname, "so_reuseaddr"))
		opt = SO_REUSEADDR;
	else if (!strcmp(optname, "so_keepalive"))
		opt = SO_KEEPALIVE;
	else if (!strcmp(optname, "so_dontroute"))
		opt = SO_DONTROUTE;
	else if (!strcmp(optname, "so_broadcast"))
		opt = SO_BROADCAST;
	else if (!strcmp(optname, "so_oobinline"))
		opt = SO_OOBINLINE;
	else if (!strcmp(optname, "so_sndbuf"))
		opt = SO_SNDBUF;
	else if (!strcmp(optname, "so_rcvbuf"))
		opt = SO_RCVBUF;
	else if (!strcmp(optname, "so_sndtimeo"))
		opt = SO_SNDTIMEO;
	else if (!strcmp(optname, "so_rcvtimeo"))
		opt = SO_RCVTIMEO;
	else
		throw(210);

	if (setsockopt(socket_fp(sockid), SOL_SOCKET, opt, (char *)&val, sizeof(val)))
		throw(711);

	return(SUCCESS);
}

/************************************************************************/
/* tcp_getsockopt(+Socket, +Optname, -Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*		so_error	= get error status and clear		*/
/*		so_type		= get socket type			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_getsockopt(Socket Opt, Val) maps onto the C calls		*/
/*	int val = Val, opt = Opt, sock = Sock;				*/
/*	setsockopt(sock, SOL_SOCKET, opt, (char *)&val, sizeof(val));	*/
/************************************************************************/
bool pr_getsockopt()
{
	cellpo argS, argO, argV;
	int opt, val;
	strpo optname;
	int len = sizeof(val);
	socketpo sockid;

	argS = &A[1]; delnk(argS);
	argO = &A[2]; delnk(argO);
	argV = &A[3]; delnk(argV);

	if (!IsInt(argS) || !IsSymb(argO) || !IsVar(argV))
		throw(210);

	get_socket(argS, sockid);
	optname = string_val(argO);

	if (!strcmp(optname, "so_debug"))
		opt = SO_DEBUG;
	else if (!strcmp(optname, "so_reuseaddr"))
		opt = SO_REUSEADDR;
	else if (!strcmp(optname, "so_keepalive"))
		opt = SO_KEEPALIVE;
	else if (!strcmp(optname, "so_dontroute"))
		opt = SO_DONTROUTE;
	else if (!strcmp(optname, "so_broadcast"))
		opt = SO_BROADCAST;
	else if (!strcmp(optname, "so_oobinline"))
		opt = SO_OOBINLINE;
	else if (!strcmp(optname, "so_sndbuf"))
		opt = SO_SNDBUF;
	else if (!strcmp(optname, "so_rcvbuf"))
		opt = SO_RCVBUF;
	else if (!strcmp(optname, "so_sndtimeo"))
		opt = SO_SNDTIMEO;
	else if (!strcmp(optname, "so_rcvtimeo"))
		opt = SO_RCVTIMEO;
	else if (!strcmp(optname, "so_error"))
		opt = SO_ERROR;
	else if (!strcmp(optname, "so_type"))
		opt = SO_TYPE;
	else
		throw(210);

	if (getsockopt(socket_fp(sockid), SOL_SOCKET, opt, (char *)&val, &len))
		throw(729);

	mkreset(argV);
	mkint(argV, val);
	return(SUCCESS);
}

static u_long machine_address(addr)
cellpo addr;
{
	if (IsFloat(addr))
		return(floatvl(addr));
	else if (IsInt(addr))
		return(intvl(addr));
	else if (IsSymb(addr)) {
		struct hostent *hp;
		if (!(hp = gethostbyname(string_val(addr)))) {
			struct in_addr in;
			in.s_addr = inet_addr(string_val(addr));
			hp = gethostbyaddr((char *) &in, sizeof(in), AF_INET);
		}
		endhostent();
		if (hp)
			return(ntohl(*(int *)hp->h_addr_list[0]));
	}
	return(0);
}

/****************************************************************/
/*	tcp_bind(+Socket, +Port, +Address)			*/
/*		Socket: Integer					*/
/*		Port: Integer					*/
/*		Address: Float or inaddr_any			*/
/****************************************************************/
bool pr_bind()
{
	struct sockaddr_in server;
	cellpo argS, argP, argA;
	u_long machine;
	u_short port;
	socketpo sockid;

	argS = &A[1]; delnk(argS);
	argP = &A[2]; delnk(argP);
	argA = &A[3]; delnk(argA);

	if (!IsInt(argS) || !IsInt(argP))
		throw(210);

	if (!(machine = machine_address(argA))) {
		if (IsSymb(argA)&&!strcmp(string_val(argA),"inaddr_any"))
			machine = INADDR_ANY;
		else
			throw(210);
	}
	get_socket(argS, sockid);
	port = intvl(argP);

	if (!bind_allowed(sockid))
		throw(700);

        server.sin_family = AF_INET;
        server.sin_addr.s_addr = htonl(machine);
        server.sin_port = htons(port);

        if (bind(socket_fp(sockid), (struct sockaddr *) &server, sizeof(server)) < 0)
		throw(712);

	set_bind(sockid);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_listen(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
bool pr_listen()
{
	cellpo argS;
	socketpo sockid;

	argS = &A[1]; delnk(argS);

	if (!IsInt(argS))
		throw(210);
	get_socket(argS, sockid);

	if (!listen_allowed(sockid))
		throw(700);
	if (listen(socket_fp(sockid), 5) < 0)
		throw(713);

	set_listen(sockid);
	return(SUCCESS);
}

/************************************************************************/
/*	tcp_accept(+Socket, -NewSocket, -Port, -Address, +TimeOut)	*/
/*		Socket: Integer						*/
/*		NewSocket: Integer					*/
/*		Port: Number						*/
/*		Address: float						*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
bool pr_accept()
{
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in add;
	cellpo	argS, argNS, argTO, argA, argP;
	socketpo sockid, newsockid;

	(void) gc_test(8L, 5);
	argS  = &A[1]; delnk(argS);
	argNS = &A[2]; delnk(argNS);
	argP  = &A[3]; delnk(argP);
	argA  = &A[4]; delnk(argA);
	argTO = &A[5]; delnk(argTO);

	if (!IsInt(argS) || !IsVar(argNS) || !IsVar(argA) || !IsVar(argP))
		throw(210);
	get_socket(argS, sockid);

	if (just_closed(sockid))
		throw(701);
	if (!accept_allowed(sockid))
		throw(700);
	if (reading_allowed(sockid)) {
		if (!(newsockid = getNextSocket()))
			throw(730);
		if ((socket_fp(newsockid) = accept(socket_fp(sockid),(struct sockaddr *) &add,&length)) == -1)
			throw(714);
		set_accept(sockid, newsockid);
		h_deadlock = 0;
		mkreset(argA);
		alloc_float(argA, ntohl(add.sin_addr.s_addr));
		mkreset(argP);
		mkint(argP, ntohs(add.sin_port));
		mkreset(argNS);
		mkint(argNS, (int) socket_ident(newsockid));
		return(SUCCESS);
	}
	return(check_time(argTO));
}

/****************************************************************/
/*	tcp_connect(+Socket, +Port, +Address, +TimeOut)		*/
/*		Socket: Integer					*/
/*		TimeOut: block, poll, Number			*/
/*		Address: Number					*/
/*		Port: Number					*/
/****************************************************************/
/* '$ tcp_connect'(Socket, Port, Address, RealTime) :-		*/
/*	'tcp_connect1%f'(Socket, Port, Address),		*/
/*	'tcp_connect2%f'(Socket, RealTime).			*/
/****************************************************************/
/*	'tcp_connect1%f'(+Socket, +Port, +Address)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
bool pr_connect1()
{
	struct sockaddr_in add;
	cellpo	argS, argA, argP;
	u_long machine;
	u_short port;
	socketpo sockid;

	argS = &A[1]; delnk(argS);
	argP = &A[2]; delnk(argP);
	argA = &A[3]; delnk(argA);

	if (!IsInt(argS) || !IsInt(argP))
		throw(210);
	if (!(machine = machine_address(argA)))
		throw(210);
	get_socket(argS, sockid);
	port = intvl(argP);

	if (!connect_allowed(sockid))
		throw(700);

	add.sin_family = AF_INET;
	add.sin_addr.s_addr = htonl(machine);
	add.sin_port = htons(port);
	if (fcntl(socket_fp(sockid), F_SETFL, FASYNC) < 0)
		throw(716);
	if (connect(socket_fp(sockid), (struct sockaddr *) &add, sizeof(add)) < 0)
		if (errno != EINPROGRESS)
			throw(715);

	if (fcntl(socket_fp(sockid), F_SETFL, 0) < 0)
		throw(716);
	return(SUCCESS);
}

/****************************************************************/
/*	'tcp_connect2%f'(+Socket, +TimeOut)			*/
/*		Socket: Integer					*/
/*		TimeOut: block, poll, Number			*/
/****************************************************************/
bool pr_connect2()
{
	cellpo	argS, argTO;
	socketpo sockid;	

	argS = &A[1]; delnk(argS);
	argTO = &A[2]; delnk(argTO);

	if (!IsInt(argS))
		throw(210);
	get_socket(argS, sockid);

	if (writing_allowed(sockid)) {
		set_connect(sockid);
		return(SUCCESS);
	}
	return(check_time(argTO));
}

/****************************************************************/
/*	tcp_send(+Socket, +Term, +EFlag)			*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
bool pr_send()
{
	int flag;
	socketpo sockid;
	strpo eflag;
	cellpo  argS, argT, argE;

	argS = &A[1]; delnk(argS);
	argT = &A[2]; delnk(argT);
	argE = &A[3]; delnk(argE);

	if (!IsInt(argS) || !IsSymb(argE))
		throw(210);
	get_socket(argS, sockid);
	eflag = string_val(argE);

	if (!strcmp(eflag, "normal"))
		flag = NORMAL_DATA;
	else if (!strcmp(eflag, "raw") && IsSymb(argE))
		flag = RAW_DATA;
	else
		throw(210);
	if (!send_allowed(sockid))
		throw(700);
	flag = tcp_write(argT, buffered_send, sockid, flag, 0,0, 0L);
	if (flag == FAIL)
		throw(prolog_throw);
	return(flag);
}

/************************************************************************/
/*	tcp_recv(+Socket, -Term, +EFlag, +TimeOut)			*/
/*		Socket: Integer						*/
/*		Term: term						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
bool pr_recv()
{
	int flag;
	socketpo sockid;
	strpo eflag;
	cellpo  argS, argT, argE, argTO;

	argS = &A[1]; delnk(argS);
	argT = &A[2]; delnk(argT);
	argE = &A[3]; delnk(argE);
	argTO = &A[4]; delnk(argTO);

	if (!IsInt(argS) || !IsSymb(argE) || !IsVar(argT))
		throw(210);
	get_socket(argS, sockid);
	eflag = string_val(argE);

	if (!strcmp(eflag, "normal"))
		flag = NORMAL_DATA;
	else if (!strcmp(eflag, "raw"))
		flag = RAW_DATA;
	else if (!strcmp(eflag, "peek"))
		flag = PEEK_DATA;
	else if (!strcmp(eflag, "peek_raw"))
		flag = PEEK_RAW_DATA;
	else
		throw(210);
	if (!recv_allowed(sockid))
		throw(700);
	if (reading_allowed(sockid)) {
		int ret;

		if (gc_test((fourBytes)(symbSize(sockid->buf_len,sizeof(cell))+2),4)) { /* HACK */
			argT = &A[2]; delnk(argT);
		}
		if ((ret=tcp_read(argT,buffered_recv,sockid,flag)) == FAIL)
			throw(718);
		return(ret);
	}
	return(check_time(argTO));
}

/****************************************************************/
/*	tcp_sendto(+Socket, +Term, +Port, +Address, +EFlag)	*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		Port: Number					*/
/*		Address: float					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
bool pr_sendto()
{
	int flag;
	socketpo sockid;
	strpo eflag;
	u_long machine;
	u_short port;
	cellpo  argS, argT, argA, argP, argE;

	argS = &A[1]; delnk(argS);
	argT = &A[2]; delnk(argT);
	argP = &A[3]; delnk(argP);
	argA = &A[4]; delnk(argA);
	argE = &A[5]; delnk(argE);

	if (!IsInt(argS) || !IsSymb(argE) || !IsInt(argP))
		throw(210);
	if (!(machine = machine_address(argA)))
		throw(210);
	get_socket(argS, sockid);
	eflag = string_val(argE);
	port = intvl(argP);

	if (!strcmp(eflag, "normal"))
		flag = NORMAL_DATA;
	else if (!strcmp(eflag, "raw") && IsSymb(argT))
		flag = RAW_DATA;
	else
		throw(210);
	if (!sendto_allowed(sockid))
		throw(700);
	flag=tcp_write(argT,buffered_sendto,sockid,flag,port,AF_INET,machine);
	if (flag == FAIL) {
		throw(prolog_throw);
	} else if ((flag == SUCCESS) && (bind_allowed(sockid))) {
		set_bind(sockid);
	}
	return(flag);
}

/************************************************************************/
/*   tcp_recvfrom(+Socket, -Term, -Port, -Address, +EFlag, +TimeOut)	*/
/*		Socket: Integer						*/
/*		Term: term						*/
/*		Port: Number						*/
/*		Address: float						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
bool pr_recvfrom()
{
	int flag;
	socketpo sockid;
	strpo eflag;
	cellpo  argS, argE, argA, argP, argT, argTO;

	argS  = &A[1]; delnk(argS);
	argT  = &A[2]; delnk(argT);
	argP  = &A[3]; delnk(argP);
	argA  = &A[4]; delnk(argA);
	argE  = &A[5]; delnk(argE);
	argTO = &A[6]; delnk(argTO);

	if (!IsInt(argS) || !IsSymb(argE) || !IsVar(argA) || !IsVar(argP) || !IsVar(argT))
		throw(210);
	get_socket(argS, sockid);
	eflag = string_val(argE);

	if (!strcmp(eflag, "normal"))
		flag = NORMAL_DATA;
	else if (!strcmp(eflag, "raw"))
		flag = RAW_DATA;
	else if (!strcmp(eflag, "peek"))
		flag = PEEK_DATA;
	else if (!strcmp(eflag, "peek_raw"))
		flag = PEEK_RAW_DATA;
	else
		throw(210);

	if (!recvfrom_allowed(sockid))
		throw(700);
	if (reading_allowed(sockid)) {
		int ret;

		if (gc_test((fourBytes)(symbSize(sockid->buf_len, sizeof(cell))+10),6)) { /* HACK */
			argT = &A[2]; delnk(argT);
			argP = &A[3]; delnk(argP);
			argA = &A[4]; delnk(argA);
		}
		if ((ret=tcp_read(argT,buffered_recvfrom,sockid,flag)) == FAIL)
			throw(720);
		mkreset(argA);
		alloc_float(argA, ntohl(sockid->add.sin_addr.s_addr));
		mkreset(argP);
		mkint(argP, ntohs(sockid->add.sin_port));
		return(ret);
	}
	return(check_time(argTO));
}

/****************************************************************/
/*	tcp_sendbr(+Socket, +Term, +Port, +EFlag)		*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		Port: Integer					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
bool pr_sendbr()
{
	int flag;
	cellpo  argS, argE, argP, argT;
	strpo  eflag;
	socketpo sockid;
	u_short port;

	argS = &A[1]; delnk(argS);
	argT = &A[2]; delnk(argT);
	argP = &A[3]; delnk(argP);
	argE = &A[4]; delnk(argE);

	if (!IsInt(argS) || !IsSymb(argE) || !IsInt(argP))
		throw(210);

	get_socket(argS, sockid);
	eflag = string_val(argE);
	port = intvl(argP);

	if (!strcmp(eflag, "normal"))
		flag = NORMAL_DATA;
	else if (!strcmp(eflag, "raw") && IsSymb(argT))
		flag = RAW_DATA;
	else
		throw(210);
	if (!sendbr_allowed(sockid))
		throw(700);
	flag = tcp_write(argT,buffered_sendbr,sockid,flag,port,0,0L);
	if (flag == FAIL)
		throw(prolog_throw);
	return(flag);
}

/****************************************************************/
/*	tcp_checkrecv(+Socket, +EFlag)				*/
/*		Socket: Integer					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
bool pr_checkrecv()
{
	cellpo  argS, argE;
	socketpo sockid;
	strpo eflag;

	argS = &A[1]; delnk(argS);
	argE = &A[2]; delnk(argE);

	if (!IsInt(argS) || !IsSymb(argE))
		throw(210);

	get_socket(argS, sockid);
	eflag = string_val(argE);

	if (!strcmp(eflag, "normal") || !strcmp(eflag, "raw")){
		if (reading_allowed(sockid))
			return(SUCCESS);
		throw(702);
	} else
		throw(210);
}

/****************************************************************/
/*	tcp_checkconn(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
bool pr_checkconn()
{
	cellpo  argS;
	socketpo sockid;

	argS = &A[1]; delnk(argS);

	if (!IsInt(argS))
		throw(210);
	get_socket(argS, sockid);

	if (writing_allowed(sockid)) {
		set_connect(sockid);
		return(SUCCESS);
	}
	throw(702);
}

/****************************************************************/
/*	'tcp_real_socket%f'(+Socket, -RealSocket)		*/
/*		SocketIndex: Integer				*/
/*		Socket: Integer					*/
/****************************************************************/
bool pr_real_socket()
{
	cellpo  argS, argRS;
	socketpo sockid;

	argS = &A[1]; delnk(argS);
	argRS = &A[2]; delnk(argRS);

	if (!IsInt(argS))
		throw(210);
	if (!IsVar(argRS))
		throw(210);
	get_socket(argS, sockid);

	mkreset(argRS);
	mkint(argRS, (int) socket_fp(sockid));
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_close(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
bool pr_tcp_close()
{
	cellpo  argS;
	socketpo sockid;

	argS = &A[1]; delnk(argS);

	if (!IsInt(argS))
		throw(210);
	get_socket(argS, sockid);
	if (free_socket(sockid))
		throw(701);

	if (close(socket_fp(sockid)) < 0)
		throw(721);
	set_close(sockid);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_getsockaddr(+Socket, -Port, -Address)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
bool pr_getsockaddr()
{
	cellpo  argS, argA, argP;
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in addr;
	struct hostent *hp;
	u_long machine;
	u_short port;
	char tcp_buf[MAX_STRING];
	socketpo sockid;

	(void) gc_test(8L, 3);

	argS = &A[1]; delnk(argS);
	argP = &A[2]; delnk(argP);
	argA = &A[3]; delnk(argA);

	if (!IsInt(argS) || !IsVar(argA) || !IsVar(argP))
		throw(210);
	get_socket(argS, sockid);

	if (getsockname(socket_fp(sockid),(struct sockaddr *)&addr,&length) < 0)
                throw(722);
	if (gethostname(tcp_buf, MAX_STRING) < 0)
		throw(723);
	if ((hp = gethostbyname(tcp_buf)) == NULL)
		throw(724);
	machine = ntohl(*(int *)hp->h_addr_list[0]);
	port = ntohs(addr.sin_port);

	mkreset(argA);
	alloc_float(argA, machine);
	mkreset(argP);
	mkint(argP, port);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_getpeeraddr(+Socket, -Port, -Address)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
bool pr_getpeeraddr()
{
	cellpo  argS, argA, argP;
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in addr;
	socketpo sockid;

	(void) gc_test(8L, 3);
	argS = &A[1]; delnk(argS);
	argP = &A[2]; delnk(argP);
	argA = &A[3]; delnk(argA);

	if (!IsInt(argS) || !IsVar(argA) || !IsVar(argP))
		throw(210);

	get_socket(argS, sockid);
	if (getpeername(socket_fp(sockid),(struct sockaddr *)&addr,&length) < 0)
		throw(725);

	mkreset(argA);
	alloc_float(argA, ntohl(addr.sin_addr.s_addr));
	mkreset(argP);
	mkint(argP, ntohs(addr.sin_port));
	return(SUCCESS);
}

/****************************************************************/
/* 	Prolog Primitives to support local network		*/
/****************************************************************/
/****************************************************************/
/*	tcp_gethost(?Name, ?Address)				*/
/*		Name: Atom or Var				*/
/*		Address: Float or Var				*/
/****************************************************************/
bool pr_gethost()
{
	cellpo  argN, argA;
	struct hostent *hp;
	u_long machine;

	argN = &A[1]; delnk(argN);
	argA = &A[2]; delnk(argA);

	if (IsVar(argN) && IsVar(argA)) {
		char tcp_buf[MAX_STRING];

		if (gethostname(tcp_buf, MAX_STRING) == -1)
			throw(723);
		hp = gethostbyname(tcp_buf);
		endhostent();
		if (!hp)
			throw(724);
		machine = ntohl(*(int *)hp->h_addr_list[0]);
		if (gc_test(4L, 2)) {
			argA = &A[2]; delnk(argA);
		}
		mkreset(argA);
		alloc_float(argA, (FLOAT) machine);
		(void) bind_symbol(1, tcp_buf, 2);
	} else if (IsSymb(argN)) {
		if (!(hp = gethostbyname(string_val(argN)))) {
			struct in_addr in;
			in.s_addr = inet_addr(string_val(argN));
			hp = gethostbyaddr((char *) &in, sizeof(in), AF_INET);
		}
		endhostent();
		if (!hp)
			throw(724);
		machine = ntohl(*(int *)hp->h_addr_list[0]);
		if (IsVar(argA)) {
			if (gc_test(4L, 2)) {
				argA = &A[2]; delnk(argA);
			}
			mkreset(argA);
			alloc_float(argA, machine);
		} else if (IsFloat(argA)) {
			if (htonl((u_long) floatvl(argA)) != machine)
				return(FAIL);
		} else if (IsInt(argA)) {
			if (htonl((u_long) intvl(argA)) != machine)
				return(FAIL);
		} else
			throw(210);
	} else if (!IsVar(argA)) {
		if (IsFloat(argA))
			machine = htonl((u_long) floatvl(argA));
		else if (IsInt(argA))
			machine = htonl((u_long) intvl(argA));
		else
			throw(210);
		hp = gethostbyaddr((char *)&machine, sizeof(machine), AF_INET);
		endhostent();
		if (!hp)
			throw(726);
		if (IsVar(argN))
			(void) bind_symbol(1, hp->h_name, 2);
		else
			throw(210);
	} else
		throw(210);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_getport(?Name, ?Proto, ?Port)			*/
/*		Name: Atom					*/
/*		Proto: Atom					*/
/*		Port: Number					*/
/****************************************************************/
bool pr_getport()
{
	cellpo  argN, argPr, argPo;
	struct servent *sp;
	strpo proto;
	u_short port;

	argN = &A[1]; delnk(argN);
	argPr = &A[2]; delnk(argPr);
	argPo = &A[3]; delnk(argPo);

	if (IsVar(argPr))
		proto = NULL;
	else if (IsSymb(argPr))
		proto = string_val(argPr);
	else
		throw(210);

	if (IsSymb(argN) && (IsInt(argPo) || IsVar(argPo))) {
		sp = getservbyname(string_val(argN), proto);
		(void) endservent();
		if (!sp)
			throw(727);

		if (IsVar(argPo)) {
			mkreset(argPo);
			mkint(argPo, ntohl(sp->s_port));
		} else if (intvl(argPo) != ntohl(sp->s_port))
			return(FAIL);
	} else if (IsInt(argPo) && IsVar(argN)) {
		port = htonl(intvl(argPo));
		sp = getservbyport((int) port, proto);
		(void) endservent();
		if (!sp)
			throw(728);
		if (bind_symbol(1, sp->s_name, 3)) {
			argPr = &A[2]; delnk(argPr);
		}
	} else
		throw(210);
	if (IsVar(argPr))
		(void) bind_symbol(2, sp->s_proto, 3);
	return(SUCCESS);
}

/************************************************************************/
/*			Internal functions				*/
/************************************************************************/
static bool buffered_write(c)
char c;
{
	if (write_socket.length >= TCPBUFSIZE) {
		longjmp(write_error, 731);
	}
	write_socket.buf[write_socket.length++] = c;
	return(SUCCESS);
}

static bool tcp_read(reg, function, sockid, flag)
cellpo reg;
int (*function)();
socketpo sockid;
int flag;
{
	int ret;

	errno = 0;
	mkreset(reg);

	if (bind_allowed(sockid))
		set_bind(sockid);
	h_deadlock = 0;
	sockid->funct = function;
	sockid->peek  = 0;
	read_socket = sockid;

	switch(flag) {
		case PEEK_DATA:
			sockid->peek  = 1;
			/* fall through */
		case NORMAL_DATA:
			{
				CHARTYPE (*tmp_charin)();

				tmp_charin = charin;
				charin = buffered_read;
				ret = encoded_read_term(reg);
				charin = tmp_charin;
				if (ret == FAIL)
					reinit_read_buf(sockid);
			}
			break;
		case PEEK_RAW_DATA:
			sockid->peek  = 1;
			/* fall through */
		case RAW_DATA:
			{
				fourBytes len;

				if (need_read(sockid)) {
					ret = function();
					if (ret == -1) {
						reinit_read_buf(sockid);
						return(FAIL);
					} else if (ret == 0) {
						reinit_read_buf(sockid);
						mksymb(reg, eof_sym);
						return(SUCCESS);
					}
				}
				sockid->pos = sockid->end;
				*(sockid->end) = '\0';
				len = sockid->end - sockid->start;
				alloc_symb(reg, len, (strpo)sockid->start);
				ret = SUCCESS;
			}
			break;
	}
	fix_read_buf_after(sockid);
	return(ret);
}

static int tcp_write(reg, function, sockid, flag, port, family, machine)
cellpo reg;
int (*function)();
socketpo sockid;
u_long machine;
u_short port;
int flag, family;
{
	bool (*tmp_charout)();
	int ret;
	int block;

	if (!writing_allowed(sockid)) {
/*		fprintf(stderr, "ERROR: reque\n");	*/
		return(WAIT);
	}
	write_socket.fp = socket_fp(sockid);
	write_socket.port = port;
	write_socket.family = family;
	write_socket.machine = machine;

	switch(flag) {
		case NORMAL_DATA:
			write_socket.buf = write_buffer;
			write_socket.length = 0;
			tmp_charout = charout;
			charout = buffered_write;
			init_encode();
			if (prolog_throw = setjmp(write_error)) {
				charout = tmp_charout;
				return(FAIL);
			} else {
				encoded_write_term(reg);
				charout = tmp_charout;
			}
			break;
		case RAW_DATA:
			write_socket.buf = (char *)string_val(reg);
			write_socket.length = string_len(reg);
			break;
	}
	block = 1;
	(void) ioctl(write_socket.fp, FIONBIO, (char *) &block);
	prolog_throw = function();
	block = 0;
	(void) ioctl(write_socket.fp, FIONBIO, (char *) &block);
	switch (prolog_throw) {
		case 0:		/* success */
			ret = SUCCESS;
			break;
		case 717:	/* send() error */
		case 719:	/* sendto() error */
			if (errno == EWOULDBLOCK)
				ret = WAIT;
			else
				ret = FAIL;
			break;
		default:	/* other error (eg ioctl) */
			ret = FAIL;
			break;
	}
	return(ret);
}
