/* Copyright (C) 1992 Imperial College */
/*
    termin.c - Edinburgh term input procedure for IC-Prolog ][
    Written by Frank McCabe, Philip Schwarz and Damian Chu
    Imperial College, Winter 1989

    Modifications:
    5/3/90    dac
	added garbage collection

*/

#include <errno.h>
#include <sys/types.h>
#ifdef GNUDOS
#include <std.h>
#include <sys/stat.h>
#else
#include <sys/ioctl.h>
#endif
#include "primitives.h"
#include "termio.h"
#include "select.h"

#ifdef ANSI
#ifndef GNUDOS
extern int	atoi(char *nptr);
#endif
extern operator	*is_op(token *);	/* test for the presence of an operator */
extern bool	samesymb(symbpo, symbpo);
extern toktype	hedtoken(token**,bool);
extern toktype	nextoken(token**,bool);
extern void	dump_token(token*);
extern void	mark_variable(cellpo);
extern void	compact_segment(cellpo, cellpo);
extern fourBytes collect_garbage(short);
extern long int	tok_len(token*);
#else
extern int	atoi();
extern operator	*is_op();	/* test for the presence of an operator */
extern bool	samesymb();
extern toktype	hedtoken();
extern toktype	nextoken();
extern void	dump_token();
extern void	mark_variable();
extern void	compact_segment();
extern fourBytes collect_garbage();
extern long	int tok_len();
extern char	*sprintf();
extern char	*memset();
#endif

extern void	(*thread_hook)();
extern CHARTYPE hedchar();
extern CHARTYPE	(*charin)();
extern void	reset_brace();
extern void	reset_total_marked();
extern int	debugLevel;
extern symbpo	equals_sym, eof_sym;
extern fd_set	rfdset;
extern threadpo	prolog_th;
extern int	h_deadlock;



/* forward references */
#ifdef ANSI
symbpo	make_symbol(strpo pname, fourBytes len);
cellpo	hash_variable(symbpo var, cellpo val);
symbpo	hash_constant(symbpo con);
#else
symbpo	make_symbol();
cellpo	hash_variable();
symbpo	hash_constant();
#endif



static jmp_buf	termjump;
static token	braces = {(toktype)graph, 3, "{}"};

static twoBytes varcount;	/* Used to create names for anonymous vars */
static cellpo	next_eqn;	/* A pointer to the next equation 	   */
static cell	term_cell,
		eqns_cell;

char	*prompt = "|: ";	/* continuation prompt */



/* global variables usd in handling syntax errors */
/* extern char linebuf[BUFSIZ];	/* Line Buffer for input on stdin */
static long SP1, SP2, SP3;	/* Stream Pointers for marking positions */
static token *err_token;	/* token which caused syntax error */

/* globals used for garbage collection during term read */
static bool	GC;		/* have we called the garbage collector?   */
static cellpo	oldH, oldHMAX;  /* shadow registers for H and HMAX	   */



/*------------------------------------------------------------*
 *      V A R I A B L E / C O N S T A N T      T A B L E      *
 *------------------------------------------------------------*/

/* maximum number of distinct constant/variable names 
   in a term before the table is expanded */
#define READER_TABLE_SIZE    50

/* structure of reader table */
typedef struct bucket {
    symbpo	     nme;    /* The (constant/variable)'s name.	     */
    cellpo	     val;    /* An unbound variable (var) or NULL(const). */
    struct bucket    *lnk;
} *bucketpo;

/* access functions for reader table */
#define name(bucket)		((bucket) -> nme)
#define value(bucket)		((bucket) -> val)
#define variable(bucket)	(value(bucket))
#define constant(bucket)	(! value(bucket))
#define link(bucket)		((bucket) -> lnk)
#define nonempty(b)		(link(b))
#define NO_LINK			((bucketpo) 1)
#define haslink(b)		(link(b) > NO_LINK)
#define table_full		(next_bucket < first_bucket)

/* This macro assumes that t points to a cellpo in the heap */
#define make_constant(t,tok)	mksymb((t),hash_constant(make_symbol(	\
				(tok)->buff, (fourBytes)((tok)->bufflen-1))))

static bucketpo	first_bucket = NULL,	/* reader table            */
		next_bucket,		/* The next free slot      */
		tabletop;		/* The end of the table    */

static twoBytes entryno = READER_TABLE_SIZE;	/* size of above   */



void
init_reader()
{
    varcount = 0;
    GC = FALSE;		/* garbage collector has not yet been called */
    reset_brace();	/* initialise the tokenizer */

    /* Initialize Hash Table */
    if (!first_bucket) {
	if (!(first_bucket=(bucketpo)alloc(entryno, sizeof(struct bucket))))
	    longjmp(icp_interrupt, 609);
	tabletop = first_bucket + entryno;
    }
    (void)memset((char*)first_bucket, 0, (SIZE_TYPE)(entryno*sizeof(struct bucket)));

    next_bucket = tabletop - 1;
}

/* find next free slot in table */
static bucketpo
nextFree(b)
bucketpo b;
{
    while (nonempty(next_bucket)  && !table_full)
	next_bucket--;
    link(b) = next_bucket;
    return(next_bucket);
}

/* reader table is doubled in size */
static void
rehash_table()
{
    register
    bucketpo	copy1, copy2;
    twoBytes	old_entryno;

    copy1 = copy2 = first_bucket;
    old_entryno   = entryno;
    entryno	  = 2 * old_entryno;
    if (!(first_bucket=(bucketpo)alloc(entryno, sizeof(struct bucket))))
	longjmp(icp_interrupt, 609);
    (void)memset((char*)first_bucket, 0, (SIZE_TYPE)(entryno*sizeof(struct bucket)));
    tabletop    = first_bucket + entryno;
    next_bucket = tabletop - 1;

    while (old_entryno--) {
	if (nonempty(copy1)) {
	    if (variable(copy1))
		(void) hash_variable(name(copy1),value(copy1));
	    else (void) hash_constant(name(copy1));
	}
	copy1++;
    }

   free((char *) copy2);
}



/*------------------------------------------------------------*
 *      S Y N T A X     E R R O R     H A N D L I N G         *
 *------------------------------------------------------------*/

void
syntax_error(errcode, tok)
twoBytes	errcode;
token		**tok;
{
    FILE *infile;

    (void)fflush(stdout);
    reset_brace();

    if (io_type(current_input) == IN_STREAM) {
	infile = fdes(current_input);
	if (isatty(fileno(infile)))
	    err_token = *tok;
	else {
	    SP2 = ftell(infile) - tok_len(*tok);
	    while (nextoken(tok,TRUE) != dot && (*tok)->tt != (toktype)eof_token);
	    SP3 = ftell(infile);
	}
    }
    else if (io_type(current_input) == IN_RAM) {
	SP2 = (long) (ramd(current_input)->current) - tok_len(*tok);
	while (nextoken(tok,TRUE) != dot && (*tok)->tt != (toktype)eof_token);
	SP3 = (long) (ramd(current_input)->current);
    }

    longjmp(termjump, errcode);
}

/* A[1] is assumed to be an integer */
bool
pr_syntax_error()
{
    cellpo	reg1 = &A[1];
    char	*msg;
    int		before, after;
    FILE	*infile;

    delnk(reg1);

    switch (intvl(reg1)) {
	case  1: msg = "'}' expected"; break;
	case  2: msg = "']' expected"; break;
	case  3: msg = "',', '|' or ']' expected"; break;
	case  4: msg = "invalid term0"; break;
	case  5: msg = "',' or ')' expected"; break;
	case  6: msg = "prefix operator precedence too high"; break;
	case  7: msg = "invalid term"; break;
	case  8: msg = "'.' expected"; break;
	case  9: msg = "unexpected end of line in string"; break;
	case 10: msg = "unexpected end of file in string"; break;
	case 11: msg = "unexpected end of line in quoted atom"; break;
	case 12: msg = "unexpected end of file in quoted atom"; break;
	case 13: msg = "unexpected end of file in number"; break;
	case 14: msg = "badly formed floating-point number"; break;
	case 15: msg = "')' expected"; break;
    }
    (void) fprintf(stderr,"\nSYNTAX ERROR : ** %s **\n", msg);

    if (io_type(current_input) == IN_STREAM) {
	infile = fdes(current_input);
	if (isatty(fileno(infile))) {
	    CHARTYPE ch;
	    (void) fprintf(stderr, "discarding input : ");
	    dump_token(err_token);
	    if (infile->_cnt > 0) {
		while ((ch = getc(infile)) != '\n')
		    (void)fputc(ch, stderr);
	    }
	    (void)fputc('\n', stderr);
	    (void)fflush(infile);
	}
	else {
	    before = SP2 - SP1,
	    after  = SP3 - SP2;
	    (void)fseek(infile, SP1, 0);
	    while (before-- && fputc((*charin)(),stderr));
	    (void) fprintf(stderr, " *HERE* ");
	    while (after-- && fputc((*charin)(),stderr));
	    (void) fputc('\n', stderr);
	}
    }
    else if (io_type(current_input) == IN_RAM) {
	before = SP2 - SP1;
	after  = SP3 - SP2;
	ramd(current_input)->current = (strpo) SP1;
	while (before-- && fputc((*charin)(),stderr));
	(void) fprintf(stderr, " *HERE* ");
	while (after-- && fputc((*charin)(),stderr));
	(void) fputc('\n', stderr);
    }
    return(FAIL);
}



/*------------------------------------------------------------*
 *         H E A P    S P A C E    M A N A G E M E N T        *
 *------------------------------------------------------------*/

/* make sure we have enough heap space.
   Call the garbage collector if necessary. */
void
interm_gc_test(Space)
int Space;
{
    if (H + Space >= HMAX) {
	if (GC)		/* second time around, we are really out of space ! */
	    longjmp(icp_interrupt, 505);
	GC = TRUE;
	oldHMAX = H;
	H = oldH;
	HMAX = H + 1;
	if (collect_garbage(4) < Space)		/* mark registers A0 to A3 */
	    longjmp(icp_interrupt, 505);
	oldH = H;
    }
}

/* This procedure is only called if garbage collection was required
   during a read.  This recompacts the heap so that the 'hole' created
   by G/C is removed. */
recompact_heap(t, eqns)
cellpo    t, eqns;
{
    /* Here is where we use those reserved cells in high memory.
       This ensures that the term and eqn cells are on the top
       of heap after compacting the heap. */
    *oldHMAX = *eqns;	/* copy root of eqn list to heap */
    mklnk(eqns,oldHMAX);
    *++oldHMAX = *t;	/* copy root of term to heap */
    mklnk(t,oldHMAX);

    /* the quickest way to unmark a block */
    while (H < HMAX)
	mknil(H++);

    /* mark and compact the segment used by interm */
    H = HMAX = ++oldHMAX;
    reset_total_marked();
    mark_variable(t);
    mark_variable(eqns);
    compact_segment(oldH, H);

    *t = *--H;		/* top of heap is root of term */
    *eqns = *--H;	/* next top of heap is root of equation list */
}



/*------------------------------------------------------------*
 *          U T I L I T Y     F U N C T I O N S               *
 *------------------------------------------------------------*/

/*
    Variable names are changed if there is a constant with that
    name in the term.  The name X is changed to X$n where n is
    some integer.  Uniqueness of this name within the term must be
    guaranteed.  Anonymous variables are given a unique name _$n
    in a similar manner.
*/
void
new_variable_name(s)
strpo s;
{
    register strpo p;

    if ((s[0] == '_') && (s[1] == '$')) /* An anonymous variable */
	(void) sprintf(s+2,"%hd",varcount++);
    else
	if (p = (strpo) strrchr(s,'$')) {
	    p++;
	    (void) sprintf(p,"%d",atoi((char*)p) + 1);
	}
	else (void)strcpy(s + strlen(s),"$0");
}

symbpo
make_symbol(pname, len)
strpo	pname;
fourBytes     len;
{
    cell	c;
    int		numcells;    /* no. of cells needed to represent the symbol */

    numcells = symbSize(len, sizeof(cell));
    interm_gc_test(numcells+2);
    alloc_symb(&c, len, pname);
    return(symbvl(&c));
}

symbpo
new_symb(in)
symbpo in;
{
    int		len = symblngth(in);
    strpo	ch;	/* space for constructing variable names */
    symbpo	result;

    if (!(ch=(strpo)malloc((size_t)(len+5))))
	longjmp(icp_interrupt, 612);

    (void) strcpy(ch,symbname(in));
    new_variable_name(ch);
    result = make_symbol(ch,(fourBytes)strlen(ch));

    free(ch);
    return(result);
}

symbpo
install_constant(c, b)
symbpo		c;
bucketpo	b;
{
    name(b)  = c;
    value(b) = NULL;
    link(b)  = NO_LINK;
    return(c);
}

cellpo
install_variable(var,val,b)
symbpo		var;
cellpo		val;
bucketpo	b;
{
    register cellpo ptr;

    if (val == NULL) { /* first time this variable was entered */
	/*
	    Create a new equation :
	    set LHS of equation and name(b) to var
	    set RHS of equation and value(b) to new_unbound
	*/
	interm_gc_test(6);

	/* Make space for a new element of output_list */
	alloc_list(next_eqn, next_eqn);
	alloc_tpl(next_eqn, ptr, 3);
	next_eqn++;
	mknil(next_eqn);	/* do not leave garbage */

	mksymb(ptr,equals_sym);	/* Functor of tuple is '=' */
	ptr++;

	mksymb(ptr,var);	/* 1st arg is variable name */
	ptr++;

	mkunb(ptr);		/* 2nd arg is an unbound variable */
	name(b)  = var;
	value(b) = ptr;
	link(b)  = NO_LINK;

	return(ptr);
    }
    else {
	/* set LHS of equation (found from val) and name(b) to var.
	   set value(b) to val.  */

	cellpo LHS = val - 1;	/* val points to the unbound variable */
	setval(LHS, (cellpo)var);

	name(b)  = var;
	value(b) = val;
	link(b)  = NO_LINK;

	return(val);
    }
}

/* returns a pointer to an unbound variable */
cellpo
hash_variable(var,val)
symbpo    var;
cellpo    val; /* Either val is NULL or it points to the RHS of an equation */
{
    register
    bucketpo b = first_bucket + symbhashval(var) % entryno;

    if (b < first_bucket)
	b += entryno;

    if (nonempty(b)) {
	bool found = FALSE;
	while (!(found=samesymb(var,name(b))) && haslink(b))
	    b = link(b);
	if (found) {
	    if (constant(b))
		return(hash_variable(new_symb(var),val));
	    return(value(b));
	}
	b = nextFree(b);
	if (table_full) {
	    rehash_table();
	    return(hash_variable(var,val));
	}
    }
    return(install_variable(var,val,b));
}


void
make_variable(tt,tok)
cellpo	tt;	/* must be pointing to a heap cell */
token	*tok;
{
    register
    fourBytes	len;
    cellpo	newvar;
    uchar	ch[6];	/* space for constructing variable names */

    if ((len=tok->bufflen-1) == 1 && *tok->buff == '_') {
	(void) sprintf(ch,"_$%hd",varcount++);
	newvar = hash_variable(make_symbol(ch,(fourBytes)strlen(ch)),(cellpo)NULL);
	free(ch);
    }
    else newvar = hash_variable(make_symbol(tok->buff,len),(cellpo)NULL);

    mklnk(tt,newvar);
}

symbpo
hash_constant(con)
symbpo con;
{
    register
    bucketpo b = first_bucket + symbhashval(con) % entryno;

    if (b < first_bucket)
	b += entryno;

    if (nonempty(b)) {
	bool found = FALSE;
	while (!(found=samesymb(con,name(b))) && haslink(b))
	    b = link(b);
	if (found) {
	    if (variable(b)) {
		(void) hash_variable(new_symb(name(b)),value(b));
		value(b) = NULL;
	    }
	    return(name(b));
	}
	b = nextFree(b);
	if (table_full) {
	    rehash_table();
	    return(hash_constant(con));
	}
    }
    return(install_constant(con, b));
}

/* look ahead one token to see if we can commit to a postfix operator */
bool usepostfix(prior)
twoBytes prior;
{
    register
    operator *nextop;
    token *tok;

    /* look ahead at the next token for the beginning of a term */
    switch (hedtoken(&tok,TRUE)) {
	case quoted:
	case lower:
	case graph:
	case solo:
	case comma:
	case semicolon:
	    return((nextop=is_op(tok)) != NULL
		    && nextop->prefixp == undefprior
		    && nextop->postleft > prior
		    && nextop->infleft > prior);

	case bra:	/* In these cases, the next token */
	case brace:	/* is definitely the beginning of */
	case sqbra:	/* a new term0; therefore the     */
	case string:	/* current operator must be infix.*/
	case number:
	case floating:
	case upper:
	    return FALSE;

	default:	    /* this is when the next token signals end  */
	    return TRUE;    /* of term. i.e.. ')', '}', ']', '.' or EOF */
    }
}

/* look ahead one character to see if we can commit to a postfix operator */
bool postfixchar()
{
    /* look ahead at the next character for the beginning of a term */
    switch (hedchar()) {
	case ket:
	case endbrace:
	case sqket:
	case dot:
	case comma:
	case bar:
	case semicolon:
	    return(TRUE);
	default:
	    return(FALSE);
    }
}

/*------------------------------------------------------------*/

/* term0 reads in a primitive term */
void term0(t)
cellpo t;
{
    token *tok;

    switch (nextoken(&tok,TRUE)) {
	case quoted:
	case lower:
	case graph:
	case solo:
	    make_constant(t, tok);
	    break;

	case bra:			    /* parenthetical term */
	    term(t,minprior);		    /* read in a bracketted term */
	    if (nextoken(&tok,FALSE)!=ket)  /* check for closing bracket */
		    syntax_error(15, &tok);
	    break;

	case brace:
	    if (hedtoken(&tok,FALSE)==endbrace) {
		make_constant(t, &braces);  /* the constant '{}' */
		(void)nextoken(&tok,FALSE);
		break;
	    }
	    else {			    /* a brace {term} */
		register cellpo s;
		interm_gc_test(3);
		alloc_tpl(t,s,2);	    /* create a new tuple and point it */
		make_constant(s,&braces);   /* fill in with braces symbol */
		term(++s,minprior);    /* read in a curly bracketted term */
		if (nextoken(&tok,FALSE)!=endbrace)
			syntax_error(1, &tok);
		break;
	    }

	case sqbra:			    /* we have a list to read */
	    if (hedtoken(&tok,FALSE)==sqket) {
		mknil(t);
		(void)nextoken(&tok,FALSE); /* consume the closing bracket */
		break;
	    }
	    else {
		register cellpo s=t;	/* point to the list built */

		while (TRUE) {
		    interm_gc_test(2);
		    alloc_list(s,s);
		    term(s++,argprior);	/* read a term of argument priority */

		     /* construct a pair */
		    if (nextoken(&tok,FALSE)==comma) {
			continue;
		    }

		    else if (tok->tt==semicolon) {
			term(s,argprior);

			 /* check for closing bracket */
			if (nextoken(&tok,FALSE)!=sqket)
			    syntax_error(2, &tok);
			break;
		    }
		    else if (tok->tt == sqket) {
			mknil(s);
			break;
		    }
		    else
			syntax_error(3, &tok);
		}
		break;
	    }

	case string: {
	    register strpo s = tok->buff;
	    register int i = (int)tok->bufflen - 1;
	    cellpo p = t;

	    interm_gc_test(2*i);
	    for (;i--;p++) {
		alloc_list(p,p);
		mkint(p,*s++);
	    }
	    mknil(p);
	    break;
	}

	case number:
	    /* bufflen field of token is overloaded for numbers */
	    mkint(t,tok->bufflen);
	    break;

	case floating: {
	    FLOAT fl_num;
	    double dummy;
	    fourBytes num;
	    (void)sscanf(tok->buff, "%lf", &dummy);
	    fl_num = dummy;

	    /* convert to an integer if possible */
	    interm_gc_test(3);

	    num = fl_num;
	    if (num == fl_num) {
		mkint1(t, num);
	    } else {
		alloc_float(t,fl_num);
	    }
	    break;
	}

	case upper:
	    make_variable(t, tok);
	    break;

	default:
	    syntax_error(4, &tok);
    }

    while (hedtoken(&tok,FALSE) == bra) {
	cell args[MAXARG];
	register cellpo a,s;
	register int maxarg = 0;

	/* copy the functor into the temporary array */
	args[maxarg++]= *t;

	(void)nextoken(&tok,FALSE);
	if (hedtoken(&tok,FALSE)!=ket)
	    while (tok->tt!=ket) {
		term(&args[maxarg++],argprior);
		if (nextoken(&tok,FALSE)==ket)
		    break;
		else if (tok->tt == comma)
		    continue;
		else
		    syntax_error(5, &tok);
	    }
	else
	    (void)nextoken(&tok,FALSE);

	/* determine if we should convert this to a list or a tuple */
	if (   maxarg == 3 &&
	       IsSymb(args) &&
	       string_len(args) == 1 &&
	       *(string_val(args)) == '.'  ) {

	    /* convert .(Head,Tail) to [Head|Tail] */
	    interm_gc_test(2);
	    alloc_list(t,s);
	    *s++ = args[1];
	    *s = args[2];
	}

	else {
	    /* copy out the tuple into actual space */
	    interm_gc_test(maxarg+1);
	    alloc_tpl(t,s,maxarg);
	    for (a=args; maxarg--;)
		*s++=*a++;
	}
    }
}

/*------------------------------------------------------------*/

/* main term input function */
term(t,prior)
cellpo t;
int prior;
{
    register
    operator *curr_op;		/* current operator */
    token *tok;
    cell lefterm;	/* buffer to hold the left term */
    int lprior;			/* left hand priority */
    int rprior;			/* right hand priority */


    /* BEWARE the use of case fall through in the following switch */

    switch (hedtoken(&tok,TRUE)) {
	case quoted:
	case lower:
	case graph:
	case solo:
	    if (hedchar()!=bra && (curr_op=is_op(tok))!=NULL
		    && curr_op->prefixp!=undefprior) {    /* we have a prefix operator */
		lprior=curr_op->prefixp;	/* priority of left */
		(void)nextoken(&tok,TRUE);	/* commit to the operator */
		make_constant(&lefterm, tok);   /* construct a constant into t */

		/* check for negative number here? */
		if (*curr_op->op == '-' && *(curr_op->op+1) == '\0') {
		    toktype t = hedtoken(&tok,FALSE);
		    if (t == (toktype)number) {
			(void)nextoken(&tok,TRUE); /* commit to the number */
			/* bufflen field of token is overloaded for numbers */
			mkint(&lefterm,-(tok->bufflen));
			lprior=0;		/* this is a term0 after all */
			break;
		    }
		    else if (t == (toktype)floating) {
			FLOAT fl_num;
			double dummy;
			fourBytes num;
			(void)nextoken(&tok,TRUE); /* commit to the number */
			(void)sscanf(tok->buff, "%lf", &dummy);
			fl_num = -dummy;
			interm_gc_test(3);

			/* convert to an integer if possible */
			num = fl_num;
			if (num == fl_num)
			    mkint1(&lefterm, num)
			else 
			    alloc_float(&lefterm, fl_num);
			lprior=0;		/* this is a term0 after all */
			break;
		    }
		}
		if (lprior>prior)		/* a syntax error */
		    syntax_error(6, &tok);

		/* pick up the right precedence of the prefix op */
		rprior=curr_op->preform;

		/* look ahead again for another operator */
		/* Note the use of fall through in the cases ! */
		switch (hedtoken(&tok,TRUE)) {
		    case quoted:
		    case lower:
		    case graph:
		    case solo:
			if (hedchar() != bra
			    && (curr_op=is_op(tok)) != NULL
			    && curr_op->prefixp > rprior
			    && curr_op->postleft >= lprior
			    && curr_op->infleft >= lprior)
			    break;
			else			/* use case fall through */

		    case bra:			/* parenthetical term */
		    case brace:			/* a brace {term} */
		    case sqbra:			/* we have a list to read */
		    case string:
		    case number:
		    case floating:
		    case upper: {
			cell s,*p;
			/* construct a prefix operator term */
			s=lefterm;		/* copy the old cell */
			interm_gc_test(3);
			alloc_tpl(&lefterm,p,2);/* create a new tuple */
			*p=s;			/* copy old cell in */
			term(++p,rprior);	/* read in another term */
			break;
		    }

		    default:
			lprior=0;
			break;
		}
		break;  /* top level switch */
	    }
	    else	 /* use case fall through */

	case bra:				/* parenthetical term */
	case brace:				/* a brace {term} */
	case sqbra:				/* we have a list to read */
	case string:
	case number:
	case floating:
	case upper: {    /* do not delete these braces */
	    term0(&lefterm);			/* read in a term0 */
	    lprior = 0;				/* left hand priority is 0 */
	    break;
	}
	default:
/**/	(void)nextoken(&tok, FALSE);		/* read past the non-term */
	    syntax_error(7, &tok);
    }

    /* rator branch deals with infix and postfix operators */
    while (TRUE)
	switch (hedtoken(&tok,TRUE)) {
	    case quoted:
	    case lower:
	    case graph:
	    case solo:
	    case comma:
	    case semicolon:
		if ((curr_op=is_op(tok))!=NULL) {
		    cell op, s,*p;
		    if (curr_op->infixp<=prior && curr_op->infleft>=lprior) {

			 /* commit to and make the operator */
			 (void)nextoken(&tok,TRUE);
			 make_constant(&op,tok);

			if (curr_op->postfixp<=prior &&
				curr_op->postleft>=lprior &&
				usepostfix(curr_op->infright)) {
			    /* make a postfix term here */
			    interm_gc_test(3);
			    alloc_tpl(&s,p,2);
			    *p++=op;
			    *p=lefterm;
			    lefterm=s;
			    lprior=curr_op->postfixp;
			    continue;
			}
			else {
			    /* make an infix term here */
			    interm_gc_test(4);
			    alloc_tpl(&s,p,3);
			    *p++=op;
			    *p++=lefterm;	/* left hand side */
			    lefterm=s;
			    term(p,curr_op->infright);
			    lprior=curr_op->infixp;
			    continue;
			}
		    }
		    else if (curr_op->postfixp<=prior &&
			    curr_op->postleft>=lprior &&
			    (curr_op->infixp == undefprior ||
				postfixchar())) {

			 /* commit to and make the operator */
			 (void)nextoken(&tok,TRUE);
			 make_constant(&op,tok);

			/* make a postfix term here */
			interm_gc_test(3);
			alloc_tpl(&s,p,2);
			*p++=op;
			*p++=lefterm;
			lefterm=s;
			lprior=curr_op->postfixp;
			continue;
		    }
		}

	    default:
		*t = lefterm;
		return;
	}
}

/*------------------------------------------------------------*/

/* main entry point for term input */
/* A[1] and A[3] are assumed to be variables.  A[2] is an integer */
bool
pr_interm()
{
    cellpo	reg1 = &A[1],
		reg2 = &A[2],
		reg3 = &A[3];

/* NOTE: must not declare term_cell or eqns_cell here.  
   They should be global variables to ensure a low address.
 */
    cellpo	t = &term_cell,
		eqns = &eqns_cell;

    token	*tok;
    int		prior, fd;
    FILE	*infile;
    long	nbytes, block;
    twoBytes    errcode;
    CHARTYPE	c;

#ifndef GNUDOS /* should be fixed in future */
    /* check if terminal input */
    infile = fdes(current_input);
    if (infile->_cnt == 0 && isatty((fd=fileno(infile)))) {	/* a tty */

	if (thread_hook)
	    (*thread_hook)();

	(void)ioctl(fd, FIONREAD, &nbytes);
	if (nbytes == 0) {
	    /* This code detects end of file (CTRL-D) but costs performance */
	    /* using non-blocking I/O to check for EOF */
	    block = 1;
	    (void)ioctl(fd, FIONBIO, &block);
	    if (read(fd, &c, 1) && errno == EWOULDBLOCK) {
		/* reset to blocking I/O */
		block = 0;
		(void)ioctl(fd, FIONBIO, &block);

		FD_SET(fd, &rfdset);
		update_select_width(fd);
		if (TH == prolog_th) {
		    if (h_deadlock & D_PROLOG) {
			wait_for_user();
			return(REQUEUE);
		    }
		    else h_deadlock |= D_PROLOG;
		}
		return(WAIT);
	    }
	}
	else {
	    while (nbytes-- > 0 && chtype(c=getc(infile)) == sep) {
		if (infile == stdin && c == '\n') { 
		    (void) printf("%s", prompt);
		    (void) fflush(stdout);
		}
	    }
	    if (chtype(c) == sep)
		    return(REQUEUE);
	    else (void) ungetc(c, infile);
	}
    }
#endif GNUDOS
    h_deadlock = 0;

    (void)gc_test(3L, 3);
    oldH = H++;		/* allow dummy cell for garbage collection */
    mknil(oldH);	/* to make sure it is not garbage */
    HMAX -= 2;		/* reserve two cells in high memory */
    next_eqn = eqns;

    delnk(reg1);
    delnk(reg2);
    delnk(reg3);

    init_reader();

    prior = intvl(reg2);

    if (io_type(current_input) == IN_STREAM)
	SP1 = ftell(fdes(current_input));
    else if (io_type(current_input) == IN_RAM)
	SP1 = (long) (ramd(current_input)->current);

    if (errcode = setjmp(termjump)) {
	throw(errcode);
    }
    else {
	if (hedtoken(&tok,TRUE) == (toktype)eof_token) {
	    mksymb(t, eof_sym);
	    (void)nextoken(&tok,TRUE);	/* skip the EOF */
	}
	else {
	    term(t,prior);
	    if (nextoken(&tok,TRUE)!=dot && tok->tt != (toktype)eof_token)
		syntax_error(8, &tok);
	}
    }

    mknil(next_eqn);

    if (GC) {
	recompact_heap(t, eqns);
	reg1 = &A[1];	/* the addresses may have changed during */
	reg3 = &A[3];	/* G/C, so need to delink again */
	delnk(reg1);
	delnk(reg3);
    }

    /* bind the variables */
    mkreset(reg1);
    *reg1 = *t;
    mkreset(reg3);
    *reg3 = *eqns;
    return (SUCCEED);
}
