/* Copyright (C) 1992 Imperial College */

/****************************************************************/
/* 	Parlog Primitives to support TCP comunication		*/
/****************************************************************/

/* To add this TCP primitives to JAM Address, insert the following
 * function call in "initialise()" in the file "initial.c"
 *
 *		tcp_init();
 *
 */

	/************************************************/
	/* 		Implementation 			*/
	/************************************************/
extern char *malloc();

#include "objs.h"
#include "ret.h"
#include "instr.h"
#include "macros.h"

#include "tcp.h"

#define EUNIFY		200
#define EBADSEQ		201
#define ECLOSE		202
#define ETOOBIG		203

#define get_socket(Arg, Socket) {	\
	int Index = ShortVal(Arg);	\
	if (Index >= FDSIZE) {		\
		errno = ENOTSOCK;	\
		return(FAIL);		\
	}				\
	Socket = sockets + Index;}


#define tcp_lock(socket)	{if (socket->lock == 1)			\
					return(REQUEUE);		\
				socket->lock = 1; }
#define tcp_unlock(socket)	{socket->lock = 0; }

/****************************************************************/
/*		Functions Declarations				*/
/****************************************************************/

int tcp_init();
int C_tcp_init0();
int C_tcp_socket();
int C_tcp_setsockopt();
int C_tcp_getsockopt();
int C_tcp_bind1();
int C_tcp_listen1();
int C_tcp_lock_sock();
int C_tcp_accept1();
int C_tcp_connect1();
int C_tcp_connect2();
int C_tcp_send1();
int C_tcp_recv1();
int C_tcp_sendto1();
int C_tcp_recvfrom1();
int C_tcp_sendbr1();
int C_tcp_checkrecv1();
int C_tcp_checkconn1();
int C_tcp_close1();
int C_tcp_getsockaddr1();
int C_tcp_getpeeraddr1();
int C_tcp_error();
int C_tcp_gethost1();
int C_tcp_getport1();
int C_tcp_real_socket();

static void buffered_write();
static int tcp_read();
static int tcp_write();

/****************************************************************/
/*		Functions Definitions		(tcp.c)		*/
/****************************************************************/

/****************************************************************/
/*			Global Variables			*/
/****************************************************************/

/*	Sockets Types	*/
static Word atom_connection;
static Word atom_broadcast;
static Word atom_connectionless;

/*	Flags 		*/
static Word atom_peek_data;
static Word atom_peek_raw_data;
static Word atom_normal_data;
static Word atom_raw_data;

/*	Protocols 		*/
static Word atom_tcp;
static Word atom_udp;
static Word atom_default;

static Word atom_ipproto_ip;
static Word atom_ipproto_udp;
static Word atom_ipproto_tcp;
static Word atom_ipproto_icmp;
static Word atom_ipproto_raw;

/*	Stream Types		*/
static Word atom_sock_stream;
static Word atom_sock_dgram;
static Word atom_sock_raw;

/*	TimeOuts 		*/
static Word atom_block;
static Word atom_poll;

static Word atom_inaddr_any;

extern Word atom_eof;

/****************************************************************/
/*		functions to support tcp primitives		*/
/****************************************************************/

int
tcp_init()
{
	(void) C_tcp_init0((Word *) NULL);

/*	Definitions		*/
	define_c_predicate("tcp_init0", 0, C_tcp_init0, enter_io_c, 1);
	define_c_predicate("tcp_socket", 3, C_tcp_socket, enter_io_c, 1);
	define_c_predicate("tcp_setsockopt", 3, C_tcp_setsockopt, enter_io_c,1);
	define_c_predicate("tcp_getsockopt", 3, C_tcp_getsockopt, enter_io_c,1);
	define_c_predicate("tcp_bind1", 3, C_tcp_bind1, enter_io_c, 1);
	define_c_predicate("tcp_listen1", 1, C_tcp_listen1, enter_io_c, 1);
	define_c_predicate("tcp_lock_sock", 1, C_tcp_lock_sock, enter_io_c, 1);
	define_c_predicate("tcp_accept1", 5, C_tcp_accept1, enter_io_c, 1);
	define_c_predicate("tcp_connect1", 3, C_tcp_connect1, enter_io_c, 1);
	define_c_predicate("tcp_connect2", 2, C_tcp_connect2, enter_io_c, 1);
	define_c_predicate("tcp_send1", 3, C_tcp_send1, enter_io_c, 1);
	define_c_predicate("tcp_recv1", 4, C_tcp_recv1, enter_io_c, 1);
	define_c_predicate("tcp_sendto1", 5, C_tcp_sendto1, enter_io_c, 1);
	define_c_predicate("tcp_recvfrom1", 6, C_tcp_recvfrom1, enter_io_c, 1);
	define_c_predicate("tcp_sendbr1", 4, C_tcp_sendbr1, enter_io_c, 1);
	define_c_predicate("tcp_checkrecv1", 2, C_tcp_checkrecv1, enter_io_c,1);
	define_c_predicate("tcp_checkconn1", 1, C_tcp_checkconn1, enter_io_c,1);
	define_c_predicate("tcp_close1", 1, C_tcp_close1, enter_io_c, 1);
	define_c_predicate("tcp_getsockaddr1",3,C_tcp_getsockaddr1,enter_io_c,1);
	define_c_predicate("tcp_getpeeraddr1",3,C_tcp_getpeeraddr1,enter_io_c,1);
	define_c_predicate("tcp_error", 1, C_tcp_error, enter_io_c, 1);
	define_c_predicate("tcp_real_socket", 2, C_tcp_real_socket, enter_io_c, 1);
	define_c_predicate("tcp_gethost1",3,C_tcp_gethost1,enter_io_c,1);
	define_c_predicate("tcp_getport1", 4, C_tcp_getport1, enter_io_c, 1);
}

C_tcp_init0(Args)
Word	*Args;
{
	static int here;

	if (!here) {
		init_socket_list;

	/*	Sockets Types	*/
		atom_connection	= make_atom("connection");
		atom_broadcast	= make_atom("broadcast");
		atom_connectionless= make_atom("connectionless");

	/*	Flags 		*/
		atom_peek_data	= make_atom("peek");
		atom_peek_raw_data = make_atom("peek_raw");
		atom_normal_data= make_atom("normal");
		atom_raw_data	= make_atom("raw");

	/*	Protocols 		*/
		atom_tcp	= make_atom("tcp");
		atom_udp	= make_atom("udp");
		atom_default	= make_atom("default");

		atom_ipproto_ip	= make_atom("ipproto_ip");
		atom_ipproto_udp = make_atom("ipproto_udp");
		atom_ipproto_tcp = make_atom("ipproto_tcp");
		atom_ipproto_icmp = make_atom("ipproto_icmp");
		atom_ipproto_raw = make_atom("ipproto_raw");

	/*	Stream Types		*/
		atom_sock_stream = make_atom("sock_stream");
		atom_sock_dgram	= make_atom("sock_dgram");
		atom_sock_raw	= make_atom("sock_raw");

	/*	 TimeOuts		*/
		atom_block	= make_atom("block");
		atom_poll	= make_atom("poll");

		atom_inaddr_any	= make_atom("inaddr_any");
		here = 1;
	}
	return(SUCCESS);
}

/****************************************************************/
/* tcp_socket(+Type, +Protocol, -Socket)			*/
/*	Type: sock_stream, sock_dgram, sock_raw			*/
/*	Protocol: ipproto_ip, ipproto_udp, ipproto_tcp		*/
/*		  ipproto_icmp, ipproto_raw			*/
/*	socket(family, type, protocol)				*/
/* We have commited ourselves to AF_INET family which means the	*/
/* type = sock_dgram | sock_stream | sock_raw			*/
/* as far as protocols are concerned the following are allowed	*/
/*	any		-> ipproto_ip				*/
/*	sock_dgram	-> ipproto_udp				*/
/*	sock_stream	-> ipproto_tcp				*/
/*	sock_raw	-> ipproto_icmp | ipproto_raw		*/
/****************************************************************/
C_tcp_socket(Args)
Word	*Args;
{
	int type, protocol;
	socketpo sockid;

	wait_for_argument(A0);
	if (A0 == atom_sock_stream)
		type = SOCK_STREAM;
	else if (A0 == atom_sock_dgram)
		type = SOCK_DGRAM;
	else if (A0 == atom_sock_raw)
		type = SOCK_RAW;
	else {
		bu_error(A0, "tcp_socket/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	wait_for_argument(A1);
	if (A1 == atom_ipproto_ip)
		protocol = IPPROTO_IP;
	else if (A1 == atom_ipproto_udp)
		protocol = IPPROTO_UDP;
	else if (A1 == atom_ipproto_tcp)
		protocol = IPPROTO_TCP;
	else if (A1 == atom_ipproto_icmp)
		protocol = IPPROTO_ICMP;
	else if (A1 == atom_ipproto_raw)
		protocol = IPPROTO_RAW;
	else {
		bu_error(A1, "tcp_socket/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	if (!(sockid = getNextSocket())) {
		errno = EMFILE;
		return(FAIL);
	}

	if ((socket_fp(sockid) = socket(AF_INET, type, protocol)) < 0)
		return(FAIL);
	
	if (type = SOCK_STREAM)
		set_open_c(sockid);
	else
		set_open_cl(sockid);

	errno = EUNIFY;
	return(unify(A2, ToShort(socket_ident(sockid))));
}

/************************************************************************/
/* tcp_setsockopt(+Socket, +Optname, +Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_setsockopt(Socket Opt, Val) maps onto the C calls		*/
/*	int val = Val, opt = Opt, sock = Sock;				*/
/*	setsockopt(sock, SOL_SOCKET, opt, (char *)&val, sizeof(val));	*/
/************************************************************************/
C_tcp_setsockopt(Args)
Word *Args;
{
	int opt, val;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_setsockopt/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);

	wait_for_argument(A1);
	if (!IsConst(A1)) {
		bu_error(A1, "tcp_setsockopt/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	} else if (A1 == make_atom("so_debug"))
		opt = SO_DEBUG;
	else if (A1 == make_atom("so_reuseaddr"))
		opt = SO_REUSEADDR;
	else if (A1 == make_atom("so_keepalive"))
		opt = SO_KEEPALIVE;
	else if (A1 == make_atom("so_dontroute"))
		opt = SO_DONTROUTE;
	else if (A1 == make_atom("so_broadcast"))
		opt = SO_BROADCAST;
	else if (A1 == make_atom("so_oobinline"))
		opt = SO_OOBINLINE;
	else if (A1 == make_atom("so_sndbuf"))
		opt = SO_SNDBUF;
	else if (A1 == make_atom("so_rcvbuf"))
		opt = SO_RCVBUF;
	else if (A1 == make_atom("so_sndtimeo"))
		opt = SO_SNDTIMEO;
	else if (A1 == make_atom("so_rcvtimeo"))
		opt = SO_RCVTIMEO;
	else {
		bu_error(A1, "tcp_setsockopt/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	if (!IsShort(A2)) {
		bu_error(A2, "tcp_setsockopt/3: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	val = ToShort(A2);

	if (setsockopt(socket_fp(sockid), SOL_SOCKET, opt, (char *)&val, sizeof(val)))
		return(FAIL);

	return(SUCCESS);
}

/************************************************************************/
/* tcp_getsockopt(+Socket, +Optname, -Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*		so_error	= get error status and clear		*/
/*		so_type		= get socket type			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_getsockopt(Socket Opt, Val) maps onto the C calls		*/
/*	int val = Val, opt = Opt, sock = Sock;				*/
/*	getsockopt(sock, SOL_SOCKET, opt, (char *)&val, sizeof(val));	*/
/************************************************************************/
C_tcp_getsockopt(Args)
Word *Args;
{
	int opt, val;
	int len = sizeof(val);
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_getsockopt/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);

	wait_for_argument(A1);
	if (!IsConst(A1)) {
		bu_error(A1, "tcp_getsockopt/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	} else if (A1 == make_atom("so_debug"))
		opt = SO_DEBUG;
	else if (A1 == make_atom("so_reuseaddr"))
		opt = SO_REUSEADDR;
	else if (A1 == make_atom("so_keepalive"))
		opt = SO_KEEPALIVE;
	else if (A1 == make_atom("so_dontroute"))
		opt = SO_DONTROUTE;
	else if (A1 == make_atom("so_broadcast"))
		opt = SO_BROADCAST;
	else if (A1 == make_atom("so_oobinline"))
		opt = SO_OOBINLINE;
	else if (A1 == make_atom("so_sndbuf"))
		opt = SO_SNDBUF;
	else if (A1 == make_atom("so_rcvbuf"))
		opt = SO_RCVBUF;
	else if (A1 == make_atom("so_sndtimeo"))
		opt = SO_SNDTIMEO;
	else if (A1 == make_atom("so_rcvtimeo"))
		opt = SO_RCVTIMEO;
	else if (A1 == make_atom("so_error"))
		opt = SO_ERROR;
	else if (A1 == make_atom("so_type"))
		opt = SO_TYPE;
	else {
		bu_error(A1, "tcp_getsockopt/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	if (getsockopt(socket_fp(sockid), SOL_SOCKET, opt, (char *)&val, &len))
		return(FAIL);

	return(unify(A2, make_int(val)));
}

static u_long machine_address(addr)
Word addr;
{
	if (IsLong(addr))
		return(LongVal(addr));
	else if (IsShort(addr))
		return(ShortVal(addr));
	else if (IsAtom(addr)) {
		struct hostent *hp;
		if (!(hp = gethostbyname(string_val(addr)))) {
			struct in_addr in;
			in.s_addr = inet_addr(string_val(addr));
			hp = gethostbyaddr((char *) &in, sizeof(in), AF_INET);
		}
		endhostent();
		if (hp)
			return(ntohl(*(int *)hp->h_addr_list[0]));
	}
	return(0);
}

/****************************************************************/
/*	tcp_bind(Socket?, Port?, Address?)			*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number or Const inaddr_any		*/
/****************************************************************/
/* tcp_bind(Socket, Port, Address) <-				*/
/*	tcp_lock_sock(Socket) &					*/
/*	tcp_bind1(Socket, Port, Address).			*/
/****************************************************************/
/*	tcp_bind1(Socket?, Port?, Address?)			*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number or Const inaddr_any		*/
/****************************************************************/
C_tcp_bind1(Args)
Word	*Args;
{
	struct sockaddr_in server;
	int length = sizeof server;
	socketpo sockid;
	int ip;
	u_long machine;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_bind/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);

	wait_for_argument(A1);
	tcp_unlock(sockid);
	if (IsShort(A1))
		ip = ShortVal(A1);
	else if (IsLong(A1))
		ip = LongVal(A1);
	else {
		bu_error(A1, "tcp_bind/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	wait_for_argument(A2);
	if (!(machine = machine_address(A2))) {
		if (IsConst(A2) && (A2 == atom_inaddr_any))
			machine = INADDR_ANY;
		else {
			bu_error(A2, "tcp_bind/3: 3rd arg incorrect");
			errno = EINVAL;
			return(FAIL);
		}
	}

	if (!bind_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}

	bzero( (char *)&server, length );
        server.sin_family = AF_INET;
	server.sin_addr.s_addr = htonl(machine);
        server.sin_port = htons(ip);
        if (bind(socket_fp(sockid), (struct sockaddr *)&server, length) < 0)
                return(FAIL);
	set_bind(sockid);
	return(SUCCESS);
		
}

/****************************************************************/
/* tcp_listen(Socket) <-					*/
/*	tcp_lock_sock(Socket) &					*/
/*	tcp_listen1(Socket).					*/
/****************************************************************/
/*	tcp_listen1(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
C_tcp_listen1(Args)
Word	*Args;
{
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_listen/1: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	get_socket(A0, sockid);
	tcp_unlock(sockid);
	if (!listen_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}
	if (listen(socket_fp(sockid), 5) < 0)
		return(FAIL);
	set_listen(sockid);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_lock_sock(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
C_tcp_lock_sock(Args)
Word	*Args;
{
	socketpo sockid;
	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_lock_sock/1: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	tcp_lock(sockid);
	return(SUCCESS);
}

/************************************************************************/
/*	tcp_accept(Socket?, TimeOut?, Port^, Address^, NewSocket^)	*/
/*		Socket: Short						*/
/*		TimeOut: block, poll, Number				*/
/*		Port: Number						*/
/*		Address: Number						*/
/*		NewSocket: Short					*/
/************************************************************************/
/* tcp_accept(Socket?, TimeOut?, Port^, Address^, NewSocket^) <-	*/
/*	tcp_time(TimeOut, RealTime) &					*/
/*	tcp_accept1(Socket, RealTime, Port, Address, NewSocket).	*/
/************************************************************************/
/*	tcp_accept1(Socket?, NewSocket^, Port^, Address^, RealTime?)	*/
/*		Socket: Short						*/
/*		RealTime: block, poll, Number				*/
/*		Port: Number						*/
/*		Address: Number						*/
/*		NewSocket: Short					*/
/************************************************************************/
C_tcp_accept1(Args)
Word	*Args;
{
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in add;
	time_t ti;
	socketpo sockid, newsockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_accept/5: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	if (just_closed(sockid)) {
		tcp_unlock(sockid);
		errno = ECLOSE;
		return(FAIL);
	}
	if (!accept_allowed(sockid)) {
		tcp_unlock(sockid);
		errno = EBADSEQ;
		return(FAIL);
	}
	if (reading_allowed(sockid)) {
		tcp_unlock(sockid);
		if (!(newsockid = getNextSocket())) {
			errno = EMFILE;
			return(FAIL);
		}
		if ((socket_fp(newsockid) = accept(socket_fp(sockid), (struct sockaddr *)&add, &length)) == -1)
			return(FAIL);
		set_accept(sockid, newsockid);
		errno = EUNIFY;
		h_deadlock = 0;
		return(unify(A3, make_int((int) ntohl(add.sin_addr.s_addr))) &
			unify(A2, make_int((int) ntohs(add.sin_port))) &
			unify(A1, ToShort(socket_ident(newsockid))));
	}

	wait_for_argument(A4);
	if (A4 == atom_block)
		return(REQUEUE);
	else if (A4 == atom_poll) {
		tcp_unlock(sockid);
		errno = ETIMEDOUT;
		return(FAIL);
	} else if (IsShort(A4))
		ti = ShortVal(A4);
	else if (IsLong(A4))
		ti = LongVal(A4);
	else {
		tcp_unlock(sockid);
		bu_error(A4, "tcp_accept/5: 5th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (time((time_t *) NULL) < ti) {
		set_select_time(ti);
		return(REQUEUE);
	}
	tcp_unlock(sockid);
	errno = ETIMEDOUT;
	return(FAIL);
}

/****************************************************************/
/*	tcp_connect(Socket?, Port?, Address?, TimeOut?)		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number					*/
/*		TimeOut: block, poll, Number			*/
/****************************************************************/
/* tcp_connect(Socket?, Port?, Address?, TimeOut?) <-		*/
/*	tcp_time(TimeOut, RealTime) &				*/
/*	tcp_connect1(Socket, Port, Address) &			*/
/*	tcp_connect2(Socket, RealTime).				*/
/****************************************************************/
/*	tcp_connect1(Socket?, Port?, Address?)	PRIVATE		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: block, poll, Number			*/
/****************************************************************/
C_tcp_connect1(Args)
Word	*Args;
{
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in add;
	int machine, port;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_connect/4: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A1);
	if (IsShort(A1))
		port = ShortVal(A1);
	else if (IsLong(A1))
		port = LongVal(A1);
	else {
		bu_error(A1, "tcp_connect/4: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A2);
	if (!(machine = machine_address(A2))) {
		bu_error(A2, "tcp_connect/4: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}

	if (!connect_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}

	tcp_lock(sockid);

	add.sin_family = AF_INET;
	add.sin_addr.s_addr = htonl(machine);
	add.sin_port = htons(port);
	if (fcntl(socket_fp(sockid), F_SETFL, FASYNC) < 0) {
		tcp_unlock(sockid);
		return(FAIL);
	}
	if ((connect(socket_fp(sockid), (struct sockaddr *)&add, length) < 0)&&
	    (errno != EINPROGRESS)) {
		tcp_unlock(sockid);
		return(FAIL);
	}
	if (fcntl(socket_fp(sockid), F_SETFL, 0) < 0) {
		tcp_unlock(sockid);
		return(FAIL);
	}
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_connect2(Socket?, TimeOut?)		PRIVATE		*/
/*		Socket: Short					*/
/*		TimeOut: block, poll, Number			*/
/****************************************************************/
C_tcp_connect2(Args)
Word	*Args;
{
	time_t ti;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_connect/4: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
 	if (writing_allowed(sockid)) {
		set_connect(sockid);
		tcp_unlock(sockid);
		return(SUCCESS);
	}

	wait_for_argument(A1);
	if (A1 == atom_block)
		return(REQUEUE);
	else if (A1 == atom_poll) {
		tcp_unlock(sockid);
		errno = ETIMEDOUT;
		return(FAIL);
	} else if (IsShort(A1))
		ti = ShortVal(A1);
	else if (IsLong(A1))
		ti = LongVal(A1);
	else {
		tcp_unlock(sockid);
		bu_error(A1, "tcp_connect/4: 4th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (time((time_t *) NULL) < ti) {
		set_select_time(ti);
		return(REQUEUE);
	}
	tcp_unlock(sockid);
	errno = ETIMEDOUT;
	return(FAIL);
}

/****************************************************************/
/*	tcp_send(Socket?, Term?, EFlag?)			*/
/*		Socket: Short					*/
/*		Term: term					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
/* tcp_send(Socket?, Term?, EFlag?) <-				*/
/*	tcp_send1(Socket, Term, EFlag).				*/
/****************************************************************/
/*	tcp_send1(Socket?, Term?, EFlag?)			*/
/*		Socket: Short					*/
/*		Term: term					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
C_tcp_send1(Args)
Word	*Args;
{
	int flag;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_send/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A2);
	if (A2 == atom_normal_data)
		flag = NORMAL_DATA;
	else if (A2 == atom_raw_data)
		flag = RAW_DATA;
	else {
		bu_error(A2, "tcp_send/3: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A1);
	if ((flag != NORMAL_DATA) && !IsConst(A1)) {
		bu_error(A1, "tcp_send/3: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (!send_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}
	return(tcp_write(A1,buffered_send,sockid,flag,0,0,0L));
}

/************************************************************************/
/*	tcp_recv(Socket?, Term^, EFlag?, TimeOut?)			*/
/*		Socket: Short						*/
/*		Term: term						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
/* tcp_recv(Socket?, Term^, EFlag?, TimeOut?) <-			*/
/*	tcp_time(TimeOut, RealTime) &					*/
/*	tcp_lock_sock(Socket) &						*/
/*	tcp_recv1(Socket, Term, EFlag, RealTime).			*/
/************************************************************************/
/*	tcp_recv1(Socket?, Term^, EFlag?, TimeOut?)			*/
/*		Socket: Short						*/
/*		Term: term						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
C_tcp_recv1(Args)
Word	*Args;
{
	int flag;
	time_t ti;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_recv/4: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A2);
	if (A2 == atom_normal_data)
		flag = NORMAL_DATA;
	else if (A2 == atom_raw_data)
		flag = RAW_DATA;
	else if (A2 == atom_peek_data)
		flag = PEEK_DATA;
	else if (A2 == atom_peek_raw_data)
		flag = PEEK_RAW_DATA;
	else {
		tcp_unlock(sockid);
		bu_error(A2, "tcp_recv/4: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (!recv_allowed(sockid)) {
		tcp_unlock(sockid);
		errno = EBADSEQ;
		return(FAIL);
	}
	flag = tcp_read(A1, buffered_recv, sockid, flag);
	if (flag != REQUEUE)
		return(flag);

	wait_for_argument(A3);
	if (A3 == atom_block)
		return(REQUEUE);
	else if (A3 == atom_poll) {
		tcp_unlock(sockid);
		errno = ETIMEDOUT;
		return(FAIL);
	} else if (IsShort(A3))
		ti = ShortVal(A3);
	else if (IsLong(A3))
		ti = LongVal(A3);
	else {
		tcp_unlock(sockid);
		bu_error(A3, "tcp_recv/4: 4th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (time((time_t *) NULL) < ti) {
		set_select_time(ti);
		return(REQUEUE);
	}
	tcp_unlock(sockid);
	errno = ETIMEDOUT;
	return(FAIL);
}

/****************************************************************/
/*	tcp_sendto(Socket?, Term?, Port?, Address?, EFlag?)	*/
/*		Socket: Short					*/
/*		Term: term					*/
/*		Port: Number					*/
/*		Address: Number					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
C_tcp_sendto1(Args)
Word	*Args;
{
	int flag;
	int machine, port;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_sendto/5: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A4);
	if (A4 == atom_normal_data)
		flag = NORMAL_DATA;
	else if (A4 == atom_raw_data)
		flag = RAW_DATA;
	else {
		bu_error(A4, "tcp_sendto/5: 5th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A2);
	if (IsShort(A2))
		port = ShortVal(A2);
	else if (IsLong(A2))
		port = LongVal(A2);
	else {
		bu_error(A2, "tcp_sendto/5: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A3);
	if (!(machine = machine_address(A3))) {
		bu_error(A3, "tcp_sendto/5: 4th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A1);
	if ((flag == RAW_DATA) && !IsConst(A1)) {
		bu_error(A1, "tcp_sendto/5: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (!sendto_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}
	flag = tcp_write(A1,buffered_sendto,sockid,flag, (u_short) port,
			AF_INET, (u_long) machine);
	if ((flag == SUCCESS) && (bind_allowed(sockid)))
		set_bind(sockid);
	return(flag);
}

/************************************************************************/
/*	tcp_recvfrom(Socket?, Term^, Port^, Address^, EFlag?, TimeOut?)	*/
/*		Socket: Short						*/
/*		Term: term						*/
/*		Port: Number						*/
/*		Address: Number						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
/* tcp_recvfrom(Socket?, Term^, Port^, Address^, EFlag?, TimeOut?) <-	*/
/*	tcp_time(TimeOut, RealTime) &					*/
/*	tcp_lock_sock(Socket) &						*/
/*	tcp_recvfrom1(Socket,Term,Address,Port,EFlag,RealTime).		*/
/************************************************************************/
/*	tcp_recvfrom1(Socket?, Term^, Port^, Address^, EFlag?, TimeOut?)*/
/*		Socket: Short						*/
/*		Term: term						*/
/*		Address: Number						*/
/*		Port: Number						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
C_tcp_recvfrom1(Args)
Word	*Args;
{
	int flag;
	time_t ti;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_recvfrom/6: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A4);
	if (A4 == atom_normal_data)
		flag = NORMAL_DATA;
	else if (A4 == atom_raw_data)
		flag = RAW_DATA;
	else if (A4 == atom_peek_data)
		flag = PEEK_DATA;
	else if (A4 == atom_peek_raw_data)
		flag = PEEK_RAW_DATA;
	else {
		tcp_unlock(sockid);
		bu_error(A4, "tcp_recvfrom/6: 5th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (!recvfrom_allowed(sockid)) {
		tcp_unlock(sockid);
		errno = EBADSEQ;
		return(FAIL);
	}
	flag = tcp_read(A1,buffered_recvfrom,sockid,flag);
	if (flag != REQUEUE)
		return(flag &
			unify(A3, make_int((int) ntohl(sockid->add.sin_addr.s_addr))) &
			unify(A2, make_int((int) ntohs(sockid->add.sin_port))));

	wait_for_argument(A5);
	if (A5 == atom_block)
		return(REQUEUE);
	else if (A5 == atom_poll) {
		tcp_unlock(sockid);
		errno = ETIMEDOUT;
		return(FAIL);
	}
	else if (IsShort(A5))
		ti = ShortVal(A5);
	else if (IsLong(A5))
		ti = LongVal(A5);
	else {
		tcp_unlock(sockid);
		bu_error(A5, "tcp_recvfrom/6: 6th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (time((time_t *) NULL) < ti) {
		set_select_time(ti);
		return(REQUEUE);
	}
	tcp_unlock(sockid);
	errno = ETIMEDOUT;
	return(FAIL);
}

/****************************************************************/
/*	tcp_sendbr1(Socket?, Term?, Port?, EFlag?)		*/
/*		Socket: Short					*/
/*		Term: term					*/
/*		Port: Number					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
C_tcp_sendbr1(Args)
Word	*Args;
{
	int flag, port;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_sendbr/4: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A3);
	if (A3 == atom_normal_data)
		flag = NORMAL_DATA;
	else if (A3 == atom_raw_data)
		flag = RAW_DATA;
	else {
		bu_error(A3, "tcp_sendbr/4: 4th arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A2);
	if (IsShort(A2))
		port = ShortVal(A2);
	else if (IsLong(A2))
		port = LongVal(A2);
	else {
		bu_error(A2, "tcp_sendbr/4: 3rd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	wait_for_argument(A1);
	if ((flag == RAW_DATA) && !IsConst(A1)) {
		bu_error(A1, "tcp_sendbr/4: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	if (!sendbr_allowed(sockid)) {
		errno = EBADSEQ;
		return(FAIL);
	}
	return(tcp_write(A1,buffered_sendbr,sockid,flag,(u_short) port,0,0L));
}

/****************************************************************/
/*	tcp_checkrecv(Socket?, EFlag?)				*/
/*		Socket: Short					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
/* tcp_checkrecv(Socket, EFlag) <-				*/
/*	tcp_lock_sock(Socket) &					*/
/*	tcp_checkrecv1(Socket, EFlag).				*/
/****************************************************************/
/*	tcp_checkrecv1(Socket?, EFlag?)				*/
/*		Socket: Short					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
C_tcp_checkrecv1(Args)
Word	*Args;
{
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_checkrecv/2: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	wait_for_argument(A1);
	tcp_unlock(sockid);
	if ((A1 == atom_normal_data) || (A1 == atom_raw_data)) {
		if (reading_allowed(sockid))
			return(SUCCESS);
		errno = ETIMEDOUT;
		return(FAIL);
	} else {
		bu_error(A1, "tcp_checkrecv/2: 2nd arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
}

/****************************************************************/
/*	tcp_checkconn(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
/* tcp_checkconn(Socket) <-					*/
/*	tcp_checkconn1(Socket).					*/
/****************************************************************/
/*	tcp_checkconn1(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
C_tcp_checkconn1(Args)
Word	*Args;
{
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_checkconn/1: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	if (writing_allowed(sockid)) {
		set_connect(sockid);
		return(SUCCESS);
	}
	errno = ETIMEDOUT;
	return(FAIL);
}

/****************************************************************/
/*	tcp_close(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
/* tcp_close(Socket) <-						*/
/*	tcp_close1(Socket).					*/
/****************************************************************/
/*	tcp_close1(Socket?)					*/
/*		Socket: Short					*/
/****************************************************************/
C_tcp_close1(Args)
Word	*Args;
{
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_close/1: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	set_close(sockid);
	if (close(socket_fp(sockid)) < 0)
		return(FAIL);
	return(SUCCESS);
}

/****************************************************************/
/*	tcp_getsockaddr(Socket?, Port^, Address^)		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
/* tcp_getsockaddr(Socket, Port, Address) <-			*/
/*	tcp_getsockaddr1(Socket, Port, Address).		*/
/****************************************************************/
/*	tcp_getsockaddr1(Socket?, Port^, Address^)		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
C_tcp_getsockaddr1(Args)
Word	*Args;
{
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in addr;
	struct hostent *hp;
	socketpo sockid;
	char tcp_buf[MAX_STRING];

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_getsockaddr/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	if (getsockname(socket_fp(sockid),(struct sockaddr *)&addr,&length) < 0)
                return(FAIL);
	if (gethostname(tcp_buf, MAX_STRING) < 0)
		return(FAIL);
	if ((hp = gethostbyname(tcp_buf)) == NULL)
		return(FAIL);
	errno = EUNIFY;
	return(unify(A2, make_int((int) ntohl(*(int *)hp->h_addr))) &
		unify(A1, make_int((int) ntohs(addr.sin_port))));
}

/****************************************************************/
/*	tcp_getpeeraddr(Socket?, Port^, Address^)		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
/* tcp_getpeeraddr(Socket, Port, Address) <-			*/
/*	tcp_getpeeraddr1(Socket, Port, Address).		*/
/****************************************************************/
/*	tcp_getpeeraddr1(Socket?, Port, Address^^)		*/
/*		Socket: Short					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
C_tcp_getpeeraddr1(Args)
Word	*Args;
{
	int length = sizeof(struct sockaddr_in);
	struct sockaddr_in addr;
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_getpeeraddr/3: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	if (getpeername(socket_fp(sockid),(struct sockaddr *)&addr,&length) < 0)
                return(FAIL);
	errno = EUNIFY;
	return(unify(A2, make_int((int) ntohl(addr.sin_addr.s_addr))) &
		unify(A1, make_int((int) ntohs(addr.sin_port))));
}

/****************************************************************/
/*	tcp_error(Err^)						*/
/*		Err: Atom					*/
/****************************************************************/
C_tcp_error(Args)
Word	*Args;
{
	int e = errno;
	char	sbuf[80];

	errno = EUNIFY;
	switch (e) {
		case ETIMEDOUT:		/* Connection timed out */
			return(unify(A0, make_atom("etimedout")));
		case EINVAL:		/* Invalid argument */
			return(unify(A0, make_atom("einval")));
		case EADDRINUSE:	/* Address already in use */
			return(unify(A0, make_atom("eaddrinuse")));
		case EADDRNOTAVAIL:	/* Can't assign requested address */
			return(unify(A0, make_atom("eaddrnotavail")));
		case ENETUNREACH:	/* Network is unreachable */
			return(unify(A0, make_atom("enetunreach")));
		case ECONNREFUSED:	/* Connection refused */
			return(unify(A0, make_atom("econnrefused")));
		case EALREADY:		/* Operation already in progress */
			return(unify(A0, make_atom("ealready")));
		case EINTR:		/* Interrupted system call */
			return(unify(A0, make_atom("eintr")));
		case EBADF:		/* Bad file number */
			return(unify(A0, make_atom("ebadf")));
		case EACCES:		/* Permission denied */
			return(unify(A0, make_atom("eacces")));
		case ENFILE:		/* File table overflow */
			return(unify(A0, make_atom("enfile")));
		case EMFILE:		/* Too many open files */
			return(unify(A0, make_atom("emfile")));
		case ENOTSOCK:		/* Socket operation on non-socket */
			return(unify(A0, make_atom("enotsock")));
		case EOPNOTSUPP:	/* Operation not supported on socket */
			return(unify(A0, make_atom("eopnotsupp")));
		case EMSGSIZE:		/* Message too long */
			return(unify(A0, make_atom("emsgsize")));
		case EISCONN:		/* Socket is already connected */
			return(unify(A0, make_atom("eisconn")));
		case ENOBUFS:		/* No buffer space available */
			return(unify(A0, make_atom("enobufs")));
		case EAFNOSUPPORT:	/* Address family not supported by protocol family */
			return(unify(A0, make_atom("eafnosupport")));
		case EUNIFY:		/* Unification Fail */
			return(unify(A0, make_atom("eunify")));
		case EBADSEQ:		/* The socket status does not support that operation */
			return(unify(A0, make_atom("ebadseq")));
		case ECLOSE:		/* socket closed while waiting for accept */
			return(unify(A0, make_atom("socket closed")));
		case ETOOBIG:		/* data too big for tcp buffer */
			return(unify(A0, make_atom("data too long")));
		default:		/* Error Unknown */
			(void) sprintf(sbuf, "error code %d", e);
			return(unify(A0, make_atom(sbuf)));
		}
}

/****************************************************************/
/*	tcp_real_socket(+SocketIndex, -Socket)			*/
/*		SocketIndex: Integer				*/
/*		Socket: Integer					*/
/****************************************************************/
C_tcp_real_socket(Args)
Word	*Args;
{
	socketpo sockid;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		bu_error(A0, "tcp_real_socket/2: 1st arg incorrect");
		errno = EINVAL;
		return(FAIL);
	}
	get_socket(A0, sockid);
	errno = EUNIFY;
	return(unify(A1, ToShort(socket_fp(sockid))));
}

/****************************************************************/
/* 	Parlog Primitives to support local network		*/
/****************************************************************/

/****************************************************************/
/*	tcp_gethost1(Mode?, Name?, Address^)			*/
/*		Name: Atom					*/
/*		Address: Number					*/
/****************************************************************/
C_tcp_gethost1(Args)
Word	*Args;
{
	struct hostent *hp;
	u_long machine;
	int ret;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		errno = EINVAL;
		return(FAIL);
	}
	if (ShortVal(A0) == 0) {	/* neither given */
		char tcp_buf[MAX_STRING];

		if (gethostname(tcp_buf, MAX_STRING) == -1)
			return(FAIL);
		hp = gethostbyname(tcp_buf);
		if (!hp)
			return(FAIL);
		endhostent();
		machine = ntohl(*(int *)hp->h_addr_list[0]);
		errno = EUNIFY;
		ret = unify(A1, make_atom(tcp_buf)) &
			unify(A2, make_int(machine));
	} else if (ShortVal(A0) == 1) {	/* Name given */
		wait_for_argument(A1);
		if (!IsAtom(A1)) {
			bu_error(A1, "tcp_gethost/2: 1st arg incorrect");
			errno = EINVAL;
			return(FAIL);
		}
		if (!(hp = gethostbyname(string_val(A1)))) {
			struct in_addr in;
			in.s_addr = inet_addr(string_val(A1));
			hp = gethostbyaddr((char *) &in, sizeof(in), AF_INET);
		}
		endhostent();
		if (!hp)
			return(FAIL);
		machine = ntohl(*(int *)hp->h_addr_list[0]);
		errno = EUNIFY;
		ret = unify(A2, make_int(machine));
	} else {
		wait_for_argument(A2);
		if (IsShort(A2))
			machine = htonl(ShortVal(A2));
		else if (IsLong(A2))
			machine = htonl(LongVal(A2));
		else {
			bu_error(A2, "tcp_gethost/2: 2nd arg incorrect");
			errno = EINVAL;
			return(FAIL);
		}
		hp = gethostbyaddr((char *)&machine, sizeof(int), AF_INET);
		endhostent();
		if (!hp)
			return(FAIL);
		errno = EUNIFY;
		ret = unify(A1, make_atom(hp->h_name));
	}
	return(ret);
}

/****************************************************************/
/*	tcp_getport1(?Mode, Name?, Proto?, Port?)		*/
/*		Mode: Number					*/
/*		Name: Atom					*/
/*		Proto: Atom					*/
/*		Port: Number					*/
/****************************************************************/
C_tcp_getport1(Args)
Word	*Args;
{
	struct servent *sp;
	char *proto;
	int port, ret;

	wait_for_argument(A0);
	if (!IsShort(A0)) {
		errno = EINVAL;
		return(FAIL);
	}
	if (IsAtom(A2))
		proto = string_val(A2);
	else
		proto = NULL;
	if (ShortVal(A0) == 1) {	/* Name given */
		wait_for_argument(A1);
		if (!IsAtom(A1)) {
			bu_error(A1, "tcp_getport/3: 1st arg incorrect");
			errno = EINVAL;
			return(FAIL);
		}
		sp = getservbyname(string_val(A1), proto);
		(void) endservent();
		if (!sp)
			return(FAIL);
		errno = EUNIFY;
		ret = unify(A3, make_int(ntohl(sp->s_port)));
	} else {
		wait_for_argument(A3);
		if (IsShort(A3))
			port = htonl(ShortVal(A3));
		else if (IsLong(A3))
			port = htonl(LongVal(A3));
		else {
			bu_error(A3, "tcp_getport/3: 3rd arg incorrect");
			errno = EINVAL;
			return(FAIL);
		}
		sp = getservbyport(port, proto);
		(void) endservent();
		if (!sp)
			return(FAIL);
		errno = EUNIFY;
		ret = unify(A1, make_atom(sp->s_name));
	}
	if (ret)
		ret = (ret & unify(A2, make_atom(sp->s_proto)));
	return(ret);
}

/************************************************************************/
/*			Internal functions				*/
/************************************************************************/
static void buffered_write(c)
char c;
{
	if (write_socket.length >= TCPBUFSIZE) {
		longjmp(write_error, ETOOBIG);
	}
	write_socket.buf[write_socket.length++] = c;
}

static int tcp_read(reg, function, sockid, flag)
Word reg;
int (*function)();
socketpo sockid;
int flag;
{
	int ret;
	static int count = 0;

	if (!reading_allowed(sockid)) {
/*		fprintf(stderr, "TRACE(tcp_read): requeue\n"); */
		return(REQUEUE);
	}
	errno = 0;
	h_deadlock = 0;
	if (bind_allowed(sockid))
		set_bind(sockid);
	tcp_unlock(sockid);

	sockid->funct = function;
	sockid->peek  = 0;
	read_socket = sockid;

	switch(flag) {
		case PEEK_DATA:
			sockid->peek  = 1;
			/* fall through */
		case NORMAL_DATA:
			ret = h_encoded_read(reg, buffered_read);
			if (ret == FAIL) {
				reinit_read_buf(sockid);
				if (errno == 0)
					errno = EUNIFY;
			}
			break;
		case PEEK_RAW_DATA:
			sockid->peek  = 1;
			/* fall through */
		case RAW_DATA:
			if (need_read(sockid)) {
				ret = function();
				if (ret == -1) {
					reinit_read_buf(sockid);
					return(FAIL);
				} else if (ret == 0) {
					errno = EUNIFY;
					reinit_read_buf(sockid);
					return(unify(reg, atom_eof));
				}
			}
			errno = EUNIFY;
			sockid->pos = sockid->end;
			*(sockid->end) = '\0';
			ret = unify(reg, make_atom(sockid->start));
			break;
	}
	fix_read_buf_after(sockid);
	return(ret);
}

static int tcp_write(reg, function, sockid, flag,port,family, machine)
Word reg;
int (*function)();
socketpo sockid;
int flag;
u_short port;
u_long machine;
{
	int ret, err, block;
	static int count = 0;

	if (!writing_allowed(sockid)) {
/*		fprintf(stderr, "TRACE(tcp_write): requeue\n"); */
		return(REQUEUE);
	}
	h_deadlock = 0;
	errno = 0;
	write_socket.fp = socket_fp(sockid);
	write_socket.port = port;
	write_socket.machine = machine;
	write_socket.family = family;

	init_vars();

	switch(flag) {
		case NORMAL_DATA:
			write_socket.length = 0;
			write_socket.buf = write_buffer;
			if (errno = setjmp(write_error)) {
				fprintf(stderr, "ERROR(tcp_write): toobig\n");
				return(FAIL);
			} else
				h_encoded_write(reg, buffered_write);
			break;
		case RAW_DATA:
			write_socket.length = AtomVal(reg)->a_length;
			write_socket.buf = string_val(reg);
			break;
	}
	block = 1;
	(void) ioctl(write_socket.fp, FIONBIO, (char *) &block);
	err = function();
	block = 0;
	(void) ioctl(write_socket.fp, FIONBIO, (char *) &block);
	switch (err) {
		case 0:		/* success */
			ret = SUCCESS;
			break;
		case 717:	/* send() error */
		case 719:	/* sendto() error */
			if (errno == EWOULDBLOCK) {
				fprintf(stderr, " TRACE(tcp_write): would block\n");
				ret = REQUEUE;
			} else {
				fprintf(stderr, " ERROR(tcp_write): other1\n");
				ret = FAIL;
			}
			break;
		default:	/* other error (eg ioctl) */
			fprintf(stderr, " ERROR(tcp_write): other2\n");
			ret = FAIL;
			break;
	}
	return(ret);
}
