/*  $Id: pl-pce.c,v 1.3 92/03/23 11:12:08 jan Exp $

    Part of XPCE
    Designed and implemented by Anjo Anjewierden and Jan Wielemaker
    E-mail: jan@swi.psy.uva.nl

    Copyright (C) 1992 University of Amsterdam. All rights reserved.
*/

extern	void	(*thread_hook)();

#define PCEHOME	"/home/laotzu/system/xpce"

#define NDEBUG				/* delete assert() */
#define MODULE	0			/* Tag selector with module */
#define DLD	0

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
This module is the actual  interface  to  the  PCE  virtual  machine  as
defined  in  the  Esprit  P1098  technical  report C1-UvA-002-TR by Anjo
Anjewierden.  It implements the following Prolog predicates:

  - $pce_new(Reference, Description).
  - $pce_send(PceObject, Selector, arguments(...Arguments...) ).
  - $pce_get(PceObject, Selector, arguments(...Arguments...), Result).
  - $pce_get_object(PceObject, Selector, arguments(...Arguments...), Result).
  - $pce_object(PceObject).
  - $pce_object(PceObject, Description).

It also defines the functions called by PCE: hostSend()  and  hostGet().
Besides  this  it  initilises PCE via the function prolog_pce_init().

This module was originally written using direct communication  with  the
internal  SWI-Prolog  data-structures.   This  version  uses the foreign
language interface functions and data structures to gain portability and
enable us  to  ship  the  interface  in  source  version  together  with
SWI-Prolog in object form.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#include <stdio.h>
#include <assert.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifndef MD				/* can be provided via make */
#define MD "../../src/md.h"
#endif

#include "ICprolog.h"			/* Prolog C interface include file */
#undef arg
#undef arity
extern symbpo define_symbol();
extern cellpo define_functor();

#include "pl-itf.h"
#include "itf-interface.h"		/* PCE interface include file */
#include <string.h>
extern char *index(), *rindex();

#define TEST(g)		{ if ((g) == PCE_FAIL) PL_fail; }
#define PCE_TEST(g)	{ if ((g) == PCE_FAIL) return PCE_FAIL; }
#define PL_TEST(g)	{ if ((g) == FALSE) return PCE_FAIL; }

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#ifndef EOS
#define EOS '\0'
#endif

#define WARN		1
#define NOWARN		0
#define EXISTING	2

typedef void (*VoidFunc)();		/* pointer to void function */

#ifdef __GNUC__
#define ArgVector(name, size)	PceObject name[size]
#else
#define ArgVector(name, size) \
	PceObject *name = (PceObject) alloca(size * sizeof(PceObject))
#endif

static PceObject	pceSelector P((term));
static PceObject	pceObjectName P((term));
static PceObject	termToObject P((term, char *, bool));

		/********************************
		*          DEBUGGING            *
		*********************************/
/*
foreign_t
pl_dbx(cmd)
term cmd;
{ char *sf = (char *) PL_query(PL_QUERY_SYMBOLFILE);
  char command[256];
  char *dbx="dbx";

  if ( PL_type(cmd) == PL_ATOM )
    dbx = PL_atom_value(PL_atomic(cmd));

  sprintf(command, "%s %s %d &\n", dbx, sf, getpid());

  if ( system(command) == 0 )
    PL_succeed;
    
  PL_fail;
}
*/
		/********************************
		*        INITIALISATION         *
		*********************************/
/*
#define PREDICATE(n, a, f)		{ extern foreign_t f(); \
					  PL_register_foreign(n, a, f, \
						PL_FA_TRANSPARENT, 0); \
					}
#define HIDDEN_PREDICATE(n, a, f)	{ extern foreign_t f(); \
					  PL_register_foreign(n, a, f, \
						PL_FA_TRANSPARENT, \
						PL_FA_NOTRACE, 0); \
					}

static
registerPredicates()
{ HIDDEN_PREDICATE("$pce_send",		3, pl_pce_send);
  HIDDEN_PREDICATE("$pce_get",		4, pl_pce_get);
  HIDDEN_PREDICATE("$pce_get_object",	4, pl_pce_get_object);
  PREDICATE("object",			1, pl_object1);
  PREDICATE("object",			2, pl_object2);
  PREDICATE("new",			2, pl_pce_new);

  PL_register_foreign("dbx", 1, pl_dbx, 0);
}
*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Code used constants are fetched from bot Prolog and PCE at boot time  to
simplify code and improve performance.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static PceObject	NAME_functor;		/* "functor" */
static PceObject	NAME_Arity;		/* "Arity" */
static PceObject	NAME_Arg;		/* "Arg" */
static PceObject	NAME_obtain;		/* "?" */
static PceObject	PROLOG;			/* @prolog object */
static PceObject	PCEOBJ;			/* @pce object */
static PceObject	NIL;			/* @nil (special) object */

static
initPceConstants()
{ NAME_functor = cToPceName("functor");
  NAME_Arity   = cToPceName("_arity");
  NAME_Arg     = cToPceName("_arg");
  NAME_obtain  = cToPceName("?");

  PROLOG       = cToPceAssoc("host");
  PCEOBJ       = cToPceAssoc("pce");
  NIL	       = cToPceAssoc("nil");
}

static symbpo ATOM_obtain;			/* "?" */
static symbpo ATOM_call;			/* "call" */
static symbpo ATOM_message;			/* "message" */
#if MODULE
static module MODULE_user;			/* public module "user" */
#endif
static fnctor_t FUNCTOR_ref1;			/* @/1 */
static fnctor_t FUNCTOR_string1;		/* string/1 */
static fnctor_t FUNCTOR_module2;		/* :/2 */
static fnctor_t FUNCTOR_new1;			/* new/1 */
static fnctor_t FUNCTOR_new2;			/* new/2 */

static
initPrologConstants()
{ ATOM_obtain     = define_symbol("?");
  ATOM_call	  = define_symbol("call");
  ATOM_message    = define_symbol("message");
#if MODULE
  MODULE_user     = PL_new_module(PL_new_atom("user"));
#endif
  FUNCTOR_ref1    = define_functor("@", 1);
  FUNCTOR_string1 = define_functor("string", 1);
  FUNCTOR_module2 = define_functor(":", 2);
  FUNCTOR_new1    = define_functor("new", 1);
  FUNCTOR_new2    = define_functor("new", 2);
}

int	(*PL_dispatch_events)() = NULL;
void	(*PL_foreign_reinit_function) Proto((int argc, char **argv));

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Transform a Prolog term into a PCE object reference.  It recognises  the
following constructs:

  <ref>		::= '@'(<atom>)
		  | '@'(<integer>)

If `warn' is WARN it prints an error  message  and  return  PCE_FAIL  on
failure.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static PceObject
pceObjectName(ref)
term ref;
{ PceObject obj;
  term arg;

  delnk(ref);

  assert(PL_type(ref) == PL_TERM && PL_unify_functor(ref, FUNCTOR_ref1));

  arg = PL_arg(ref, 1);
  switch( PL_type(arg) )
  { case PL_INTEGER:
      if ( !(obj = cToPceReference(PL_integer_value(PL_atomic(arg)))) )
      { PL_warning("Bad PCE integer reference: @%d",
		   PL_integer_value(PL_atomic(arg)));
	return PCE_FAIL;
      }

      return obj;
    case PL_ATOM:
      { atomic a = PL_atomic(arg);

	if ( !(obj = cToPceAssoc(PL_atom_value(a))) )
	{ PL_warning("Unknown PCE object: @%s", PL_atom_value(a));
	  return PCE_FAIL;
	}

	return obj;
      }
    default:
      PL_warning("Illegal PCE object reference");
      return PCE_FAIL;
  }
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Transform a Prolog atom into a PCE name object to be used as a selector.
If the Prolog datum is not an atom an error message  is  displayed,  the
tracer is started and pceSelector() returns PCE_FAIL
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static
PceObject
pceSelector(sel)
term sel;
{ PceObject selector;

  delnk(sel);

  if ( PL_type(sel) == PL_ATOM )
  { TEST(selector = cToPceName(PL_atom_value(PL_atomic(sel))) );
    return selector;
  }

  PL_warning("Illegal PCE selector");
  return PCE_FAIL;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Tranform a term  arguments(...Arguments...) to  an  array of  objects.
The caller  should allocate the array.   TRUE is returned  on success,
FALSE on failure, in which case an error message is displayed as well.
We don't check the functor of the term (it should be  `arguments', but
if it isn't, it does not matter).
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static int
pceArgc(args)
term args;
{ delnk(args);

  switch( PL_type(args) )
  { case PL_ATOM:
      return 0;
    case PL_TERM:
      return PL_functor_arity(args);
    default:
      return -1;
  }
}

static bool
pceArguments(args, argc, argv)
term args;
int argc;
PceObject *argv;
{ int n;

  delnk(args);

  for(n = 1; n <= argc; n++)
    TEST(argv[n-1] = termToObject(PL_arg(args, n), NULL, FALSE) );

  PL_succeed;
}
  

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Convert a Prolog term (as the 2nd argument to new/2 or the arguments  to
send/[2-12] or get/[3-13]) into a Prolog object.  The term can be an @/1
reference  or  a term, whose functor denotes the class and arguments are
the initialising arguments.  Note that an atom is ambiguous  as  it  can
both represent a name or an object which initialisation does not require
arguments (e.g. `chain' can represent a the PCE-name "chain" or an empty
chain.   For the principal functor to new/2 we will treat atoms as class
names.  Otherwise they are treated as PCE-names.  This is  indicated  by
the  last  argument.  Thus new(@ch, chain(foo)) will create a chain with
one member: the PCE-name "foo".
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static
PceObject
termToObject(t, assoc, new)
term t;
char *assoc;
bool new;
{ fnctor_t f;
  symbpo fa;

  delnk(t);

  switch( PL_type(t) )
  { case PL_INTEGER:
      return cToPceInteger(PL_integer_value(PL_atomic(t)));
    case PL_FLOAT:
      return cToPceReal(PL_float_value(PL_atomic(t)));
    case PL_ATOM:
      if (new == FALSE)
	return cToPceName(PL_atom_value(PL_atomic(t)));
      else
	return pceNew(assoc, cToPceName(PL_atom_value(PL_atomic(t))), 0, NULL);
    case PL_TERM:
      break;
    default:
      PL_warning("Illegal PCE object description");
      return PCE_FAIL;
  }

  f = PL_functor(t);
  fa = PL_functor_name(f);

  if ( PL_unify_functor(t, FUNCTOR_new1) )
    return termToObject(PL_arg(t, 1), NULL, TRUE);
  if ( PL_unify_functor(t, FUNCTOR_new2) )
  { PceObject rval;
    struct pce_type pt;
    term r = PL_arg(t, 1);

    if ( PL_unify_functor(r, FUNCTOR_ref1) == FALSE )
    { PL_warning("send: new/2: Illegal object reference");
      return PCE_FAIL;
    }
      
    r = PL_arg(r, 1);
    switch(PL_type(r))
    { case PL_ATOM:
	PCE_TEST(rval = termToObject(PL_arg(t, 2),
				     PL_atom_value(PL_atomic(r)),
				     TRUE));
	PCE_TEST(pceToC(rval, &pt));
	switch(pt.type)
	{ case PCE_REFERENCE:
	  case PCE_ASSOC:
	    return rval;
	  default:
	    PL_fatal_error("Unknown PCE return type: %d", pt.type);
	    return PCE_FAIL;
	}
      case PL_VARIABLE:
	PCE_TEST( rval = termToObject(PL_arg(t, 2), NULL, TRUE) );
	PCE_TEST(pceToC(rval, &pt) );
	switch(pt.type)
	{ case PCE_REFERENCE:
	    PL_TEST( PL_unify_atomic(r, PL_new_integer((int)pt.value)) );
	    return rval;
	  case PCE_ASSOC:
	    PL_TEST( PL_unify_atomic(r, PL_new_atom((char *)pt.value)) );
	    return rval;
	  default:
	    PL_fatal_error("Unknown PCE return type: %d", pt.type);
	    return PCE_FAIL;
	}
    }
  }

  if ( PL_unify_functor(t, FUNCTOR_ref1) )
    return pceObjectName(t);

  if ( PL_unify_functor(t, FUNCTOR_string1) )
  { char tmp[25];
    char *s = tmp;
    term a;

    a = PL_arg(t, 1);
    switch( PL_type(a) )
    { case PL_ATOM:	s = PL_atom_value(PL_atomic(a));
			break;
      case PL_INTEGER:	sprintf(tmp, "%d", PL_integer_value(PL_atomic(a)));
			break;
      case PL_FLOAT:	sprintf(tmp, "%f", PL_float_value(PL_atomic(a)));
			break;
      case PL_STRING:	s = PL_string_value(PL_atomic(a));
			break;
/*
      case PL_TERM:	if ( (s = PL_list_string_value(a)) != NULL )
			   break;
*/
      default:		PL_warning("Argument to string/1 should be atomic");
			return PCE_FAIL;
    }
    return cToPceString(assoc, s);
  }

  { int n;
    int arity = PL_functor_arity(t);
    PceObject class = cToPceName(symbname(fa));
    ArgVector(argv, arity);
    int done = 0;

    PCE_TEST( argv[0] = termToObject(PL_arg(t,1), NULL, FALSE) );

#if MODULE
    if ( arity >= 2 )
    { if ( argv[0] == PROLOG && (samesymb(fa, ATOM_message) || samesymb(fa, ATOM_obtain)) )
      { module m = 0;
	term a;

	a = PL_arg(t, 2);
	if ( arity >= 3 && PL_type(a) == PL_ATOM && samesymb(symbvl(a), ATOM_call) )
	{ a = PL_arg(t, 3);
	  done = 3;
	} else
	  done = 2;

	if ((a = PL_strip_module(a, &m)) == NULL)
	  return PCE_FAIL;

	switch( PL_type(a) )
	{ case PL_ATOM:
	      if ( m != MODULE_user &&
		   index(PL_atom_value(PL_atomic(a)), ':') == NULL )
	      { char tmp[1024];

		sprintf(tmp, "%s:%s", PL_atom_value(PL_module_name(m)),
				      PL_atom_value(PL_atomic(a)));
		PCE_TEST(argv[done-1]=cToPceName(tmp));
	      } else
		PCE_TEST(argv[done-1]=cToPceName(PL_atom_value(PL_atomic(a))));
	      break;
	  case PL_TERM:
	      if ( samesymb(PL_functor_name(PL_functor(a)), ATOM_obtain) )
	      { PCE_TEST(argv[done-1] = termToObject(a, NULL, FALSE));
		break;
	      }
	      /*FALLTHROUGH*/
	  default:
	      PL_warning("Illegal selector to message");
	      return PCE_FAIL;
	}
      }
    }
#endif MODULE

    for(n=2; n <= arity; n++)
    { if ( n != done )
	PCE_TEST(argv[n-1] = termToObject(PL_arg(t, n), NULL, FALSE) );
    }

    return pceNew(assoc, class, arity, argv);
  }
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
unifyObject() unifies a Prolog term with an  object  `description'.   An
object  description consists of a term whose functor is the class of the
object and whose arguments are the same argument than those that have to
be presented to `new' for initialisation of the object.

unifyObject() works recursively.  On the toplevel the object  is  always
expanded  to  a  description.   On the lower levels expansion only takes
place if the corresponding part of the Prolog term is instantiated to an
atom (class name) or a term, not equal to @/1. Thus

	?- new(@ch, chain(chain(0), chain(1, 2))), 
	   object(@ch, O), 
	   object(@ch, chain(chain(X), Y)).

	O = chain(@477532, @477600), X = 0, Y = @477600

Note that strings are treated special.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static bool
unifyObject(t, obj, top)
term t;
PceObject obj;
bool top;
{ struct pce_type pt;
  char *s;
  atomic name;
  int type;

  delnk(t);

  TEST(pceToC(obj, &pt) );

  switch(pt.type)
  { case PCE_INTEGER:
	return PL_unify_atomic(t, PL_new_integer((int)pt.value) );
    case PCE_REAL:
      { union { PceValue v; float f; } vf;
	vf.v = pt.value;
	return PL_unify_atomic(t, PL_new_float(vf.f) );
      }
    case PCE_NAME:
    case PCE_TYPE:			/* both return char * */
	return PL_unify_atomic(t, PL_new_atom(pt.value) );
  }

  if ( pceObject(obj) )
  { if ((s = pceStringToCharp(obj)) != PCE_FAIL)
    { switch( PL_type(t) )
      { case PL_TERM:
	  if ( PL_unify_functor(t, FUNCTOR_string1) )
	    return PL_unify_atomic(PL_arg(t, 1), PL_new_atom(s));
	  goto unify_ref;
	case PL_VARIABLE:
	  if ( top )
	  { PL_unify_functor(t, FUNCTOR_string1);
	    return PL_unify_atomic(PL_arg(t, 1), PL_new_atom(s));
	  }
	  goto unify_ref;
	default:
	  return PCE_FAIL;
	}
      }

    if ( top ||
	(type = PL_type(t)) == PL_ATOM ||
	(type == PL_TERM && !PL_unify_functor(t, FUNCTOR_ref1)))
    { PceObject got, arg;
      int arity, n;
      atomic f;

      TEST(got = pceGet(obj, NAME_functor, 0, NULL) );
      TEST(pceToC(got, &pt) );
      if (pt.type != PCE_NAME)
	PL_fail;
      f = PL_new_atom((char *)pt.value);

      TEST(got = pceGet(obj, NAME_Arity, 0, NULL) );
      TEST(pceToC(got, &pt) );
      if (pt.type != PCE_INTEGER)
	PL_fail;
      if ((arity = (int)pt.value) == 0)
	return PL_unify_atomic(t, f);

      TRY(PL_unify_functor(t, PL_new_functor(f, arity) ));

      for(n=1; n <= arity; n++)
      { ArgVector(argv, 1);
	argv[0] = cToPceInteger(n);
	TEST(arg = pceGet(obj, NAME_Arg, 1, argv) );
	TRY(unifyObject(PL_arg(t, n), arg, FALSE) );
      }
      PL_succeed;
    }
  }

unify_ref:
  switch(pt.type)
  { case PCE_REFERENCE:	name = PL_new_integer((int)pt.value);	break;
    case PCE_ASSOC:	name = PL_new_atom((char *)pt.value);	break;
  }

  TRY(PL_unify_functor(t, FUNCTOR_ref1) );
  t = PL_arg(t, 1);

  return PL_unify_atomic(t, name);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
pl_pce_get() and pl_pce_get_object() are called  directly  from  Prologs
virtual   machine   interpreter.   See  the  called  functions  and  the
description of Prolog calling foreign predicates above for details.   In
Prolog these predicates are named $pce_get/4 and $pce_get_object/4.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static bool
pce_get(rec, sel, args, val, ref)
term rec, sel, args, val;
bool ref;
{ PceObject receiver, selector, value;
  int argc = pceArgc(args);
  AnswerMark mark;

  markAnswerStack(mark);

  TEST(receiver = termToObject(rec, NULL, FALSE) );
  TEST(selector = pceSelector(sel) );

  switch(argc)
  { case -1:
      return PL_warning("Bad argument term");
    case 0:
      value = pceGet(receiver, selector, 0, NULL);
      break;
    default:
    { ArgVector(argv, argc);

      TRY( pceArguments(args, argc, argv) );
      value = pceGet(receiver, selector, argc, argv);
    }
  }

  if ( value != PCE_FAIL &&
       unifyObject(val, value, ref ? FALSE : TRUE) == TRUE )
  { rewindAnswerStack(mark, ref ? value : NIL);

    PL_succeed;
  } else
  { rewindAnswerStack(mark, NIL);
    PL_fail;
  }
}


bool
pl_pce_get()
{ term	rec  = &A[1],
	sel  = &A[2],
	args = &A[3],
	val  = &A[4];

  delnk(rec);
  delnk(sel);
  delnk(args);
  delnk(val);

  return pce_get(rec, sel, args, val, TRUE);
}


bool
pl_pce_get_object()
{ term	rec  = &A[1],
	sel  = &A[2],
	args = &A[3],
	val  = &A[4];

  delnk(rec);
  delnk(sel);
  delnk(args);
  delnk(val);

  return pce_get(rec, sel, args, val, FALSE);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implementation of $pce_send/3.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
pl_pce_send()
{ term	rec  = &A[1],
	sel  = &A[2],
	args = &A[3];

  PceObject receiver;
  PceObject selector;
  int argc;
  bool rval;
  AnswerMark mark;
  
  delnk(rec);
  delnk(sel);
  delnk(args);

  argc = pceArgc(args);
  markAnswerStack(mark);

  TEST(receiver = termToObject(rec, NULL, FALSE));
  TEST(selector = pceSelector(sel));

  switch(argc)
  { case -1:
      return PL_warning("Bad argument term");
    case 0:
      rval = pceSend(receiver, selector, 0, NULL);
      break;
    default:
    { ArgVector(argv, argc);

      TRY( pceArguments(args, argc, argv) );
      rval = pceSend(receiver, selector, argc, argv);
    }
  }
  
  rewindAnswerStack(mark, NIL);

  if ( rval == PCE_FAIL )
    PL_fail;
  PL_succeed;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implementation  of new/2  and new_global/2.  First  the PCE  object is
created  from the description.  Then the   first arguments is  matched
against  or instantiated with the @/1  operator.   If  the argument to
this operator turns out  to be a variable it  is instantiated with the
the object reference.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static bool
pce_new(ref, descr)
term ref, descr;
{ PceObject obj;
  term r;
  struct pce_type pt;
  AnswerMark mark;
  bool rval;

  delnk(ref);

  if (PL_unify_functor(ref, FUNCTOR_ref1) == FALSE)
    return PL_warning("new/2: Illegal object reference");

  markAnswerStack(mark);

  r = PL_arg(ref, 1);
  switch( PL_type(r) )
  { case PL_ATOM:
      obj = termToObject(descr,
			 PL_atom_value(PL_atomic(r)),
			 TRUE);
      rval = obj ? TRUE : FALSE;
      break;
    case PL_VARIABLE:
      { if ( (obj = termToObject(descr, NULL, TRUE)) &&
	     pceToC(obj, &pt) )
	{ switch(pt.type)
	  { case PCE_REFERENCE:
	      rval = PL_unify_atomic(r, PL_new_integer((int)pt.value));
	      break;
	    case PCE_ASSOC:
	      rval = PL_unify_atomic(r, PL_new_atom((char *)pt.value));
	      break;
	    default:
	      PL_fatal_error("Unknown PCE return type: %d", pt.type);
	      rval = FALSE;
	  }
	} else
	  rval = FALSE;
	break;
      }
    default:
      PL_warning("new/2: Illegal object reference");
      rval = FALSE;
  }

  rewindAnswerStack(mark, obj);
  return rval;
}


bool
pl_pce_new()
{ term	ref   = &A[1],
	descr = &A[2];

  delnk(ref);
  delnk(descr);

  return pce_new(ref, descr);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Implement  object/[1-2]: checking for  the existence of an  object and
transforming object references into object descriptions.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
pl_object1()
{ term	ref = &A[1];
  delnk(ref);

  if ( PL_type(ref) == PL_TERM && PL_unify_functor(ref, FUNCTOR_ref1) )
  { term arg = PL_arg(ref, 1);
    
    switch( PL_type(arg) )
    { case PL_INTEGER:
	if ( pceExistsReference(PL_integer_value(PL_atomic(arg))) )
	  PL_succeed;
	PL_fail;
      case PL_ATOM:
	if ( pceExistsAssoc(PL_atom_value(PL_atomic(arg))) )
	  PL_succeed;
	PL_fail;
    }
  }

  PL_fail;
}


bool
pl_object2()
{ term	ref   = &A[1],
	descr = &A[2];
  PceObject obj;

  delnk(ref);
  delnk(descr);


  if ((obj = termToObject(ref, NULL, FALSE)) == PCE_FAIL)
    PL_fail;

  return unifyObject(descr, obj, TRUE);
}


		/********************************
		*         CALLING PROLOG        *
		*********************************/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
Transform a vector of arguments and a selector into a Prolog term  which
we  can  give to PL_call(). The second argument indicates whether we want
to create a goal for hostSend() or hostGet(). For the latter we  add  an
extra variable for the result. NULL is returned on failure.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static term
vectorToGoal(sel, argc, argv, send, m)
PceObject sel;
int argc;
PceObject *argv;
bool send;
module *m;
{ term goal = PL_new_term();
  term st = goal;
  struct pce_type pt;
  atomic f;
  char *s;
  int n;

  if (pceToC(sel, &pt) == PCE_FAIL)
    return (term) NULL;
  if (pt.type != PCE_NAME)
  { PL_warning("Illegal selector in host%s()", send ? "Send" : "Get");
    return (term) NULL;
  }

#if MODULE
  if ((s = index(pt.value, ':')) != NULL && s == rindex(pt.value, ':'))
  { *s = EOS;
    *m = PL_new_module(PL_new_atom((char *)pt.value));
    f = PL_new_atom(&s[1]);
    *s = ':';
  } else
  { *m = MODULE_user;
    f = PL_new_atom((char *)pt.value);
  }
#else
  f = PL_new_atom((char *)pt.value);
#endif MODULE

  if (argc == 0 && send == TRUE)
  { PL_unify_atomic(st, f);
    return goal;
  }

  PL_unify_functor(st, PL_new_functor(f, send ? argc : argc+1));

  for(n = 1; n <= argc; n++, argv++)
    if ( unifyObject(PL_arg(st, n), *argv, FALSE) == FALSE )
      PL_fatal_error("Internal error in PCE object conversion");

  return goal;
}  

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
hostSend() is called by PCE to invoke behaviour in Prolog.  As the terms
built by vectorToGoal() and the actual calling are not related to  other
material  we  can savely reset the global stack's top pointer to discard
these terms.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int
hostSend(prolog, sel, argc, argv)
PceObject prolog;
PceObject sel;
int argc;
PceObject *argv;
{ term goal;
  module m;
  bktrk_buf buf;
  bool rval;

  PL_mark(&buf);

  if ((goal = vectorToGoal(sel, argc, argv, TRUE, &m)) == (term) NULL)
    return PCE_FAIL;

  rval = PL_call(goal, m);
  PL_bktrk(&buf);

  return (rval == FALSE) ? PCE_FAIL : PCE_SUCCEED;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
hostGet()  calls  prolog  and  transforms  the  last  argument  of   the
(succeeded) goal into an object.  It then returns this object to PCE.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

PceObject
hostGet(prolog, sel, argc, argv)
PceObject prolog;
PceObject sel;
int argc;
PceObject argv[];
{ term goal;
  module m;
  bktrk_buf buf;
  PceObject answer;

  PL_mark(&buf);

  if ((goal = vectorToGoal(sel, argc, argv, FALSE, &m)) == (term) NULL)
    return PCE_FAIL;

  if (PL_call(goal, m) == FALSE)
  { PL_bktrk(&buf);
    return PCE_FAIL;
  }

  answer = termToObject(PL_arg(goal, argc+1), NULL, FALSE);
  PL_bktrk(&buf);

  return answer;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
hostAction() is called by PCE to manipulate or query the  hosts'  status
or call goals from the PCE tracer.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

static int
_hostAction(action, args)
int action;
va_list args;
{ switch(action)
  { case HOST_QUERY:
      { term goal = PL_new_term();

	PL_unify_functor(goal, PL_new_functor(PL_new_atom("$call_atom"), 1));
	PL_unify_atomic(PL_arg(goal, 1), PL_new_atom(va_arg(args, char *)));
	if (PL_call(goal, NULL) == TRUE)
	  PL_action(PL_ACTION_WRITE, "Yes\n");
	else
	  PL_action(PL_ACTION_WRITE, "No\n");

	return PCE_SUCCEED;
      }
    case HOST_TRACE:
      PL_action(PL_ACTION_TRACE, NULL);
      return PCE_SUCCEED;
    case HOST_BACKTRACE:
      PL_action(PL_ACTION_BACKTRACE, NULL);
      return PCE_SUCCEED;
    case HOST_HALT:
      PL_action(PL_ACTION_HALT, NULL);		/* should not return */
      return PCE_FAIL;
    case HOST_BREAK:
      PL_action(PL_ACTION_BREAK, NULL);
      return PCE_SUCCEED;
    case HOST_ABORT:
      PL_action(PL_ACTION_ABORT, NULL);
      return PCE_SUCCEED;
    case HOST_SIGNAL:
    { int sig = va_arg(args, int);
      VoidFunc func = va_arg(args, VoidFunc);
      PL_signal(sig, func);
      return PCE_SUCCEED;
    }
    case HOST_RECOVER_FROM_FATAL_ERROR:
      PL_action(PL_ACTION_ABORT, NULL);
      return PCE_FAIL;			/* could not abort: failure */
    case HOST_WRITE:
      PL_action(PL_ACTION_WRITE, va_arg(args, char *));
      return PCE_SUCCEED;
    case HOST_FLUSH:
      PL_action(PL_ACTION_FLUSH, NULL);
      return PCE_SUCCEED;
    default:
      PL_fatal_error("Unknown action request from PCE: %d", action);
      return PCE_FAIL;
  }
}


#ifdef __STDC__
int
hostAction(int action, ...)
{ va_list args;
  int rval;

  va_start(args, action);
  rval = _hostAction(action, args);
  va_end(args);
  return rval;
}

#else

int
hostAction(va_alist)
va_dcl
{ va_list args;
  int action, rval;

  va_start(args);
  action = va_arg(args, int);
  rval = _hostAction(action, args);
  va_end(args);
  return rval;
}

#endif

static int
pce_dispatch()
{ if ( pceDispatch() == PCE_DISPATCH_INPUT )
    return PL_DISPATCH_INPUT;

  return PL_DISPATCH_TIMEOUT;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
prolog_pce_init() is called by  Prolog  before  it  performs  any  other
actions.   It  passes the program's Argc and Argv.  After initialisation
pceDoDispatch() is called to allow event  dispatching  while  Prolog  is
reading from the users' input channal.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

bool
prolog_pce_init()
{ ArgVector(av, 1);
  int argc;
  char **argv;
  static bool initialised = FALSE;
  char name_buf[100];
  
#if DLD
{ extern char *(*getFunctionNameFromAddress)();

  getFunctionNameFromAddress = dld_find_function;
}
#endif

  PL_foreign_reinit_function = (VoidFunc) prolog_pce_init;
/*
  notify_status.abort_is_save = TRUE;
*/

/*
  argc = (int)     PL_query(PL_QUERY_ARGC);
  argv = (char **) PL_query(PL_QUERY_ARGV);
*/
  argc = 0;
  argv = NULL;

  if ( pceInitialise(argc, argv) != PCE_SUCCEED )
    PL_fatal_error("Could not initialise PCE");

  if ( initialised == TRUE )
    PL_succeed;
  initialised = TRUE;

  initPceConstants();			/* get code used PCE constants */  
  initPrologConstants();		/* get code used Prolog constants */
/*  registerPredicates();			/* make the interface known */

  av[0] = cToPceAssoc("on");
  pceSend(PCEOBJ, cToPceName("catch_error_signals"), 1, av);
  av[0] = cToPceName(PCEHOME);
  pceSend(PCEOBJ, cToPceName("home"), 1, av);

  sprintf(name_buf, "SWI-Prolog version %s", PLVERSION);
  av[0] = cToPceName(name_buf);
  pceSend(PROLOG, cToPceName("system"), 1, av);
  av[0] = cToPceName("prolog");		/* @host --> @prolog */
  pceSend(PROLOG, cToPceName("name_reference"), 1, av);

  PL_dispatch_events = pce_dispatch;
/*
  PL_abort_handle(pceReset);
*/
  thread_hook = (void (*))pce_dispatch;
  PL_succeed;
}


		/********************************
		*       QUERY THE HOST          *
		*********************************/

typedef int	Char;		/* char that can pass EOF */

static Char
do_get_char()
{ Char c;

  if ( PL_dispatch_events != NULL )
  { /* DEBUG(3, printf("do_get_char() --> ")); */
    for(;;)
    { /* mayNotify(); */
      if ( (*PL_dispatch_events)() == PL_DISPATCH_INPUT )
      { char chr;

        if (read(0, &chr, 1) == 0)
          c = EOF;
        else
          c = (Char) chr;
        /* hasNotified(); */
        break;
      }
      /* hasNotified(); */
    }

    /* DEBUG(3, printf("%d (%c) --> ", c, c)); */
  } else
    c = (Char) getchar();

  return c;
}

int
hostQuery(what, value)
int what;
PceValue *value;
{
  switch(what)
  {
/*
    case HOST_SYMBOLFILE:
	if ((*value = (PceValue)PL_query(PL_QUERY_SYMBOLFILE)) != (char *)NULL)
	  return PCE_SUCCEED;
	return PCE_FAIL;
*/
    case HOST_GETC:
/*
	*value = (PceValue) PL_query(PL_QUERY_GETC);
*/
	*value = (PceValue) do_get_char();
	return PCE_SUCCEED;
    default:
	PL_warning("Unknown query from PCE: %d", what);
	return PCE_FAIL;
  }
}
