%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%            T E M P O R A L    P R O P E R T Y    C H E C K E R
%
%                 Author: Mantis H.M. Cheng (May/30/1994)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% check( +Property, +Agent, -Trace )
%	holds if "Agent" satisfies the "Property" and "Trace" is a path
%	that demonstrates it
%
check( op('not',Prop), Agent, [] ) :-
	\+ check( Prop, Agent, _ ).
check( op('or',Prop1,_), Agent, T ) :-
	check( Prop1, Agent, T ).
check( op('or',_,Prop2), Agent, T ) :- !,
	check( Prop2, Agent, T ).
check( op('and',Prop1,Prop2), Agent, T3 ) :- !,
	check( Prop1, Agent, T1 ),
	check( Prop2, Agent, T2 ),
	common( T1, T2, T3 ).
  %
  % <Agent> |- <Action1> < <Action2>
  % Whenever <Action2> occurs, it must be preceded by <Action1>.
  %
check( op('<',Act1,Act2), Agent, Trace1 ) :- !,
        precedes( Agent, Act1, Act2, Trace ),
	reverse( [Act2|Trace], Trace1 ).
  %
  % <Agent> |- <Action1> ==> <Action2>
  % Whenever <Action1> occurs, then <Action2> will eventually occur.
  % But, <Action2> may occur before or without <Action1>.
  %
check( op('==>',Act1,Act2), Agent, Trace1 ) :- !,
        leads_to( Agent, Act1, Act2, Trace ),
	reverse( [Act2|Trace], Trace1 ).
  %
  % <Agent> |- enabled "{" <Action> ... <Action> "}"
  % There exists a trace such that <Action>...<Action> are the possible
  % next actions.
check( op('enabled',set([A|As])), Agent1, Trace1 ) :-
	!,
	find_first_subset( Agent1, [A|As], [], [], [], Trace ),
	reverse( Trace, Trace1 ).
check( op('enabled',Act), Agent1, Trace ) :-
	label( Act ), !,
	check( op('enabled',set([Act])), Agent1, Trace ).
  %
  % <Agent> |- <Action>
  % Eventually <Action> occurs.
  %
check( lid(deadlock), Agent1, [] ) :-
	\+ trans( Agent1, _, _ ), !.
check( lid(deadlock), Agent1, Trace1 ) :- !,
	trans( Agent1, Act2, Agent2 ),
        find_first( Act2, Agent2, lid(deadlock), _, [Agent1], _, [], Trace ),
	reverse( Trace, Trace1 ).
check( Act, Agent1, Trace1 ) :-
	label( Act ),
	trans( Agent1, Act2, Agent2 ),
        find_first( Act2, Agent2, Act, _, [Agent1], _, [], Trace ),
	reverse( [Act|Trace], Trace1 ).


%% common( T1, T2, T3 )
%	holds if T1 or T2 is a common prefix of the other.
%
common( [], [], [] ) :- !.
common( [], L,  L ) :- !.
common( L,  [], L ) :- !.
common( [U|Xs], [U|Ys], [U|L] ) :-
	common( Xs, Ys, L ).


echo_trace( Trace ) :- 
	print_expression( Trace ), nl,
	ask_input( '(n)ext ? ', L ),
	tokenise( L, Cmds ),
	(Cmds = [lid(n)|_] -> fail ; true).


%% leads_to( +Agent, +Action1, +Action2, +Actions )
%	holds if an occurrence of "Action1" in "Agent" always leads to an
%	eventual occurrence of "Action2", where "Actions" are those
%	that have occurred but not including an "Action2"
%	That is, whenever "Action1" occurs, "Action2" will eventually
%	occurs.
%
%% precedes( +Agent, +Action1, +Action2, +Actions )
%	same as "leads_to" except "Action2" is not allowed to occur
%	before "Action1".
%
leads_to( P, A, An, T ) :-
	trans( P, A1, P1 ),
	find_first( A1, P1, A, Pn, [P], Ps, [], T1 ),
	find_first( A, Pn, An, _,  Ps,  _,  T1, T  ).

precedes( P, A, An, T ) :-
	trans( P, A1, P1 ),
	find_first( A1, P1, A, Pn, [P], Ps, [], T1 ),
	non_member( An, T1 ),
	find_first( A, Pn, An, _,  Ps,  _,  T1, T  ).

%% find_first( A, P, An, Pn, Ps, Ps1, T, T1 )
%	holds if there exists a (reversed) trace T1 given T is the
%       trace so far such that "An" is the last action equals to "A" and
%       "Pn" is the agent after performing the action "An", where
%	"Ps" is the set of agents seen so far and "Ps1" is the set of
%	agents seen up to when "An" is found not including "Pn".
%
find_first( A, P, A,  P,  Ps, Ps,  T, T  ) :- !.
find_first( A, P, lid(deadlock), Pn, Ps, Ps1, T, T1 ) :-
	non_member( P, Ps ),
	trans( P, A1, P1 ),
	find_first( A1, P1, lid(deadlock), Pn, [P|Ps], Ps1, [A|T], T1 ).
find_first( A, P, lid(deadlock), P, Ps, Ps, T, [A|T] ) :-
	!,
	non_member( P, Ps ),
	\+ trans( P, _, _ ).
find_first( A, P, An, Pn, Ps, Ps1, T, T1 ) :-
	A \== An, 
	non_member( P, Ps ),
	trans( P, A1, P1 ),
	find_first( A1, P1, An, Pn, [P|Ps], Ps1, [A|T], T1 ).


%% find_first_subset( +P, +As, +Ps, +Succs, +T, -T1 )
%	holds if there exists a (reversed) trace T1 given T is the
%       trace so far such that "As" is the set of actions that "P" allows,
%	where "Ps" is the set of agents seen so far. "Succs" is the set
%	descendants to be explored.
%
find_first_subset( P, As, Ps, Succs, T, T1 ) :-
	non_member( P, Ps ),
	successors( P, Succ ),
	(enabled( As, Succ ) ->
	  (T=T1)
	; (union( Succs, Succ, Succs1 ),
	   delete_one( Succs1, pr(A,P1), Succs2),
	   find_first_subset( P1, As, [P|Ps], Succs2, [A|T], T1 )) ).


delete_one( [X|Xs], X, Xs ).
delete_one( [_|Xs], X, Ys ) :-
	delete_one( Xs, X, Ys ).


%% enables( +Actions, +Successors )
%	holds if the set of "Actions" are possible in the set of
%	"Successors"
%
enabled( [], _ ) :- !.
enabled( [A|As], Succ ) :-
	member( pr(A,_), Succ ),
	enabled( As, Succ ).

