%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%            M O D A L    M U - C A L C U L U S    C H E C K E R
%
%                 Author: Mantis H.M. Cheng (Jun/13/1994)
%
%              (Tableau Property Checker for Finite-State Systems)
%                ( with a fixed-point rule but no thinning rule )
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% satisfy( +Form, +Agent, +Def, +Sat, +Count, -Count1 )
%
%	Form = a modal (mu-calculus) formula
%	Agent = an agent to be shown to satisfy "Form"
%	Def = a list of definitions for fixed-point unfolding
%	Sat = a list of pairs (P,E) each denoting P |= E has been proven
%	Count,Count1 = counter for introducing new definitional constants
%
  %
  % P |= tt   for all P
  %
satisfy( form(tt), _, _, _, N, N ) :- !.
  %
  % P |=/= ff   for all P
  %
satisfy( form(ff), _, _, _, _, _ ) :-
	!, 
	fail.
  %
  % P |= not F  iff  P |=/= F
  %
satisfy( op('not',Prop), P, Def, Sat, N, N ) :- !,
	\+ satisfy( Prop, P, Def, Sat, N, _ ).
  %
  % P |= F1 and F2   iff  P |= F1 and P |= F2
  %
satisfy( op('and',Prop1,Prop2), P, Def, Sat, N, N2 ) :- !,
	satisfy( Prop1, P, Def, Sat, N,  N1 ),
	satisfy( Prop2, P, Def, Sat, N1, N2 ).
  %
  % P |= F1 or F2   iff  P |= F1 or P |= F2
  %
satisfy( op('or',Prop1,_), P, Def, Sat, N, N1 ) :-
	satisfy( Prop1, P, Def, Sat, N, N1 ).
satisfy( op('or',_,Prop2), P, Def, Sat, N, N1 ) :- !,
	satisfy( Prop2, P, Def, Sat, N, N1 ).
  %
  % P |= U  iff U ::= max Z.E(Z) or min Z.E(Z), U has not been satisfied
  %             before, if so, then it is true for 'max' and false for
  %             'min'; otherwise, P |= E(Z) { U/Z } and remember that
  %             P |= U.
  %
satisfy( num(C), P, Def, Sat, N, N1 ) :- 
	member( num(C)=op('.',op(O,Z),E), Def ), !,
	(member( sat(P,num(C)), Sat )->
		(O = 'max' -> true ; fail)
	; ( replace( Z, E, num(C), E1 ),
	    satisfy( E1, P, Def, [sat(P,num(C))|Sat], N, N1 ) )).
  %
  % P |= D  iff D ::= F and P |= F
  %
satisfy( uid(C), P, Def, Sat, N, N1 ) :- !,                % definition
	defn( uid(C), Prop, _ ),
	satisfy( Prop, P, Def, Sat, N, N1 ).
satisfy( func(uid(C),M,T), P, Def, Sat, N, N1 ) :- !,      % definition
	eval( func(uid(C),M,T), Form ),
	defn( Form, Prop, _ ),
	satisfy( Prop, P, Def, Sat, N, N1 ).
  %
  % P |= O Z . E(Z)  iff  P |= U where U = O Z . E(Z), and O is 'max' or 'min'
  %
satisfy( op('.',op(O,Z),E), P, Def, Sat, N, N2 ) :- 
	member( O, ['min','max'] ), !,
	N1 is N+1, 
	satisfy(num(N),P,[num(N)=op('.',op(O,Z),E)|Def], Sat, N1, N2 ).
  %
  % P |= [ - ].E     iff R |= E for all R in { Q : P ==a==> Q, any a }
  %
satisfy( op('.',box('-'),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	all_satisfy( Succs, E, Def, Sat, N, N1 ).
  %
  % P |= [ K ].E     iff R |= E for all R in { Q : P ==a==> Q, a in K }
  %
satisfy( op('.',box(set(K)),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	partition_successors( Succs, K, SuccsK, _ ),
	all_satisfy( SuccsK, E, Def, Sat, N, N1 ).
  %
  % P |= [-K ].E     iff R |= E for all R in { Q : P ==a==> Q, a not in K }
  %
satisfy( op('.',box(op('-',set(K))),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	partition_successors( Succs, K, _, SuccsNotK ),
	all_satisfy( SuccsNotK, E, Def, Sat, N, N1 ).
  %
  % P |= [ a ].E     iff P |= [ {a} ].E
  %
satisfy( op('.',box(Act),E), P, Def, Sat, N, N1 ) :-
	label( Act ), !,
	findall( pr(Act,Q), trans(P,Act,Q), Succs ),
	all_satisfy( Succs, E, Def, Sat, N, N1 ).
  %
  % P |= [ -a ].E     iff P |= [ -{a} ].E
  %
satisfy( op('.',box(op('-',Act)),E), P, Def, Sat, N, N1 ) :-
	label( Act ), !,
	successors( P, Succs ),
	partition_successors( Succs, [Act], _, SuccsNotAct ),
	all_satisfy( SuccsNotAct, E, Def, Sat, N, N1 ).
  %
  % P |= [ a.b.c ].E  = P |= [a].[b].[c].E
  %
satisfy( op('.',box(op('.',A1,As)),E), P, Def, Sat, N, N1 ) :-
	label( A1 ), !,
	satisfy( op('.',box(A1),op('.',box(As),E)), P, Def, Sat, N, N1 ).
  %
  % P |= < - >.E     iff R |= E for some R in { Q : P ==a==> Q }
  %
satisfy( op('.',diamond('-'),E), P, Def, Sat, N, N1 ) :- !,
	trans( P, _, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= < K >.E     iff R |= E for some R in { Q : P ==a==> Q, a in K }
  %
satisfy( op('.',diamond(set(K)),E), P, Def, Sat, N, N1 ) :- !,
	member( Act, K ),
	trans( P, Act, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= < -K >.E     iff R |= E for some R in { Q : P ==a==> Q, a not in K }
  %
satisfy( op('.',diamond(op('-',set(K))),E), P, Def, Sat, N, N1 ) :- !,
	trans( P, Act, Q ),
	non_member( Act, K ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= < a >.E   iff P |= < {a} >.E
  %
satisfy( op('.',diamond(Act),E), P, Def, Sat, N, N1 ) :-
	label( Act ), !,
	trans( P, Act, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= < -a >.E  iff  P  |= < -{a} >.E
  %
satisfy( op('.',diamond(op('-',Act)),E), P, Def, Sat, N, N1 ) :-
	label( Act ), !,
	trans( P, Act1, Q ),
	Act1 \== Act,
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= < a.b.c >.E  = P |= <a>.<b>.<c>.E
  %
satisfy( op('.',diamond(op('.',A1,As)),E), P, Def, Sat, N, N1 ) :-
	label( A1 ), !,
	satisfy( op('.',diamond(A1),op('.',diamond(As),E)), P, 
                 Def, Sat, N, N1).


%% partition_successors( +Succs, +K, -SuccsK, -SuccsNotK )
%	holds if "SuccsK" is the set of successors pr(A,Q) in "Succs", 
%	where A is in "K", and "SuccsNotK" is the rest.
%
partition_successors( [],              _, [],     []        ) :- !.
partition_successors( [pr(A,Q)|Succs], K, SuccsK, SuccsNotK ) :- !,
	(member( A, K )->
		(SuccsK = [pr(A,Q)|SuccsK1],
		 partition_successors( Succs, K, SuccsK1, SuccsNotK ))
	; 	(SuccsNotK = [pr(A,Q)|SuccsNotK1],
		 partition_successors( Succs, K, SuccsK, SuccsNotK1 )) ).


%% all_satisfy( +Succs, +E, +Def, +Sat, +Count, -Count1 )
%	holds just in case when all successors in "Succs" satisfy "E".
%	Note: "ff" is only satisfiable by an empty set of successors.
%
all_satisfy( [],                   form(ff), _,  _  , N, N ) :- !.
all_satisfy( [pr(_,Q)],            E,       Def, Sat, N, N1 ) :- !,
	satisfy( E, Q, Def, Sat, N, N1 ).
all_satisfy( [pr(_,Q),Succ|Succs], E,       Def, Sat, N, N2 ) :-
	satisfy( E, Q, Def, Sat, N, N1 ),
	all_satisfy( [Succ|Succs], E, Def, Sat, N1, N2 ).



%% replace( +Z, +E, +R, -E1 ).
%	"E1" is the formula after replacing every free occurrance of "Z"
%	in "E" by "R", i.e., E1 = E { R/Z }.
%
replace( Z,  Z,  E,  E ) :- !.
replace( Z, op(O,L,R), E, op(O,L1,R1) ) :-
        member( O, ['and','or'] ), !,
	replace(Z,L,E,L1),
	replace(Z,R,E,R1).
replace( Z, op('.',op(C,X),R), E, op('.',op(C,X),R1) ) :-
	member( C, ['min', 'max'] ), 
	Z \== X, !,
	replace(Z,R,E,R1).
replace( Z, op('.',box(M),R), E, op('.',box(M),R1) ) :- !,
	replace(Z,R,E,R1).
replace( Z, op('.',diamond(M),R), E, op('.',diamond(M),R1) ) :- !,
	replace(Z,R,E,R1).
replace( _, E, _, E ).
