%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%              O P E R A T I O N A L     S E M A N T I C S
%
%          (Communicating Sequential Processes by C.A.R.Hoare)
%
%                 Author: Mantis H.M. Cheng (May/30/1994)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
%% trans( +Agent, -Action, -Agent1)
%	The "Agent" can make a transition "Action" and becomes "Agent1"
%

%
% SKIP
%
%  ----------------------
%   SKIP == done ==> STOP
%
trans( 'SKIP', done, 'STOP' ) :- !.

%
% Agent Constant (C)
%
%  A ::= P, P == a ==> P' 
% ------------------------ provided the evaluation of C unifies with A.
%     C == a ==> P'
%
trans(uid(C), A, P1) :- !,             % simple constant
        defn( uid(C), R, _), 
        trans( R, A, P1).
trans(func(uid(C),N,T), A, P1) :- !,   % compound constant
        eval( func(uid(C),N,T), P2), 
        defn( P2, R, _), 
        trans( R, A, P1).

%
% Fixed point
%
%   P{(fix X.P)/X} == a ==> P' 
% --------------------------
%        X.P == a ==> P'
%
trans( op('.',op('fix',X),P), A, P2 ) :- !,
	substitute( P, P1, X, op('.',op('fix',X),P) ),
	trans( P1, A, P2 ).


%
% Prefix 
%
%  ----------------- a may a simple action, an input/output action
%   a->P == a ==> P
%
%  ------------------------------ each ai is an action
%   _x:{a1,...an}->P == ai ==> P
%
%         ---------------------------
%	  (a & b)->P == a ==> b->P
%
%         ---------------------------
%	  (a & b)->P == b ==> a->P
%
trans( op('->',op('&',A,A2),P), A1, op('->',A2,P1) ) :-
	trans( op('->',A,P), A1, P1 ).
trans( op('->',op('&',A1,A),P), A2, op('->',A1,P1) ) :-
	!,
	trans( op('->',A,P), A2, P1 ).
trans( op('->',op('&',_,A2),P), A2, P ) :- !.
trans( op('->',op(':',X,set(As)),P), X, P ) :- 
	!,
	member( X, As ).
trans( op('->',op('?',C,T),P), op('?',C,T1), P ) :- !,
	simplify( T, T1 ).
trans( op('->',op('!',C,T),P), op('!',C,T1), P ) :- !,
	% eval( T, T1 ).
	simplify( T, T1 ).
trans( op('->',A,P), A1, P ) :- 
	!,
	ground_label( A ),
	eval( A, A1 ).

%
% Loop  @P = P;(@P)
%
trans( op('@',P), A, Q ) :-
	!,
	trans( op(';',P,op('@',P)), A, Q ).

%
% Sequential Composition
%
%       P == a ==> P1
% ------------------------- provided a =/= done
%    P ; Q == a ==> P' ; Q
%
%  P == done ==> P1, Q == b ==> Q'
% -------------------------------
%       P ; Q == b ==> Q'
%
trans( op(';',P,Q), A, R ) :-
	!,
	trans(P, A1, P1),
	(A1 \== done -> (A=A1, R=op(';',P1,Q)) ;
	                     (trans( Q, A, R ))).
%
% Choices (general and deterministic)
%
%  In the official CSP definition of "|", "|" is not a binary operator.
%  But, we shall treat it as one.
%
%    P == a ==> P'
% -------------------
%  P + Q == a ==> P' 
%
%    Q == a ==> Q'
% -------------------
%  P + Q == a ==> Q' 
%
trans( op('|',P,Q), A, R)  :- trans( op('+',P,Q), A, R ).
trans( op('+',P,_), A, P1) :- trans(P, A, P1).
trans( op('+',_,Q), A, Q1) :- !, trans(Q, A, Q1).


%
% Parallel Composition 
%
%
%    P == a ==> P', Q == a ==> Q'
% ---------------------------------- provided a in L and L =/= {}
%  (P || Q)^L == a ==> (P' || Q')^L
%
%            P == a ==> P'
% ---------------------------------- provided a not-in L
%  (P || Q)^L == a ==> (P' || Q)^L
%
%            Q == a ==> Q'
% ---------------------------------- provided a not-in L
%  (P || Q)^L == a ==> (P || Q')^L
%
% (Note: When L={}, then (P || Q)^L = P ||| Q.)
%
trans( op('^',op('||',P,Q),set([X|Xs])), A, 
       op('^',op('||',P1,Q1),set([X|Xs])) ) :-
        trans(P, A1, P1),
	is_in( A1, [X|Xs] ),
	matching_action( A1, A2, A ),
	trans(Q, A2, Q1).
trans( op('^',op('||',P,Q),set(L)), A, op('^',op('||',P1,Q),set(L)) ):-
        trans(P, A, P1),
	\+ is_in( A, L ).
trans( op('^',op('||',P,Q),set(L)), A, op('^',op('||',P,Q1),set(L)) ):-
	!,
        trans(Q, A, Q1),
	\+ is_in( A, L ).

%
% Interleaving
%
%
%        P == a ==> P'
% -----------------------------
%  P ||| Q == a ==> P' ||| Q
%
%        Q == a ==> Q'
% -----------------------------
%  P ||| Q == a ==> P ||| Q'
%
%
trans( op('|||',P,Q), A, op('|||',P1,Q) ) :-
        trans(P, A, P1).
trans( op('|||',P,Q), A, op('|||',P,Q1) ) :-
	!,
        trans(Q, A, Q1).

%
% Concealment
%
%      P == a ==> P'
%  -------------------- provided a not-in L
%   P\L  == a ==> P'\L
%
%  P == a ==> P', P'\L == b ==> P''\L
%  ----------------------------------- provided a in L
%         P\L  == b ==> P''\L
%
trans( op('\',P,set(L)), A, P2 ) :- !,
        trans(P, A1, P1), 
        (is_in(A1, L) -> 
               trans(op('\',P1,set(L)),A,P2) ;
               (A=A1, P2=op('\',P1,set(L))) ).

%
% After
%
%      P == s ==> P', P' == a ==> P''
%  -----------------------------------
%            P/s  == a ==> P''
%
%
trans( op('/',P,S), A, P2 ) :-
	!,
	after( P, S, P1 ),
	trans( P1, A, P2 ).

%
% Renaming
%
%      P == a ==> P'
%  ----------------------
%   P#f  == f(a) ==> P'#f
%
% Note: f(a)=a if a is not in the domain of f.
%
trans( op('#',P,set(F)), A1, op('#',P1,set(F)) ) :-
	!,
	trans( P, A, P1 ),
	renames( A, A1, F ).

%
% Pipes
%
% P |> Q = (((P#{right=comm}) || (Q#{left=comm}))^{comm})\{comm}
%
trans( op('|>',P,Q), A, R ) :- !,
	trans( op('\',op('^',
                      op('||',
                      op('#',P,set([op('=',lid(right),lid(comm))])),
                      op('#',Q,set([op('=',lid(left),lid(comm)) ]))),
                             set([lid(comm)]) ), 
                      set([lid(comm)]) ), A, R ).


%
% Conditional agent
%
%      P == a ==> P'
%  -------------------------- provided C evaluates to true
%   if C then P  == a ==> P'
%
trans( ifthen(C,P), A, P1 ) :- 
	eval( C, bool(true) ),
	trans( P, A, P1 ).


%
% Auxilary predicates
%

matching_action( op('?',C,T), op('!',C,T), op('$',C,T) ) :- !.
matching_action( op('!',C,T), op('?',C,T), op('$',C,T) ) :- !.
matching_action( A,           A          , A           ).


%
% substitute( +Exp, -Exp1, +Var, +Exp2 )
%	replaces every occurrence of Var in Exp by Exp2 and returns Exp1
%       i.e., Exp1 = Exp{Exp2/Var}
%
substitute( X, X, _, _ ) :-
	var( X ), !.
substitute( X, T, X, T ) :- !.
substitute( [P|Ps], [Q|Qs], X, T ) :- !,
	substitute( P,  Q,  X, T ),
	substitute( Ps, Qs, X, T ).
substitute( func(F,N,As), func(F,N,As1), X, T ) :- !,
	substitute( As, As1, X, T ).
substitute( ifthen(B,E), ifthen(B,E1), X, T ) :- !,
	substitute( E, E1, X, T ).
substitute( op(O,P), op(O,P1), X, T ) :- !,
	substitute( P, P1, X, T ).
substitute( op(O,P,Q), op(O,P1,Q1), X, T ) :- !,
	substitute( P, P1, X, T ),
	substitute( Q, Q1, X, T ).
substitute( X, X, _, _ ).  % everything else remains the same


%
%%renames( +Action, +Action1, +Renames )
%    Action is renamed to Action1 according to the function Renames.
%   
renames( A, A, [] ) :- !.
	% This is the case for a channel communication.
renames( op('?',C,T), op('?',B,T), [op('=',C,B)|_] ) :- !.
renames( op('!',C,T), op('!',B,T), [op('=',C,B)|_] ) :- !.
renames( op('$',C,T), op('$',B,T), [op('=',C,B)|_] ) :- !.
renames( A, B, [op('=',A,B)|_] ) :- !.
renames( A, B, [_|L] ) :-
	% A is not in "_"
	renames( A, B, L ).


%% is_in(+X, +L)
%       succeeds if X is a member of L
% 
is_in(op('?',C,_),[C|_] ) :- !.  % for a channel communication
is_in(op('!',C,_),[C|_] ) :- !.  % for a channel communication
is_in(op('$',C,_),[C|_] ) :- !.  % for a channel communication
is_in(A,          [B|_] ) :- ground_label(A), A=B, !.
is_in(A,          [_|L] ) :- 
	% A \== _
	is_in(A,L).


%
%% after( +Agent, +Trace, -Agent1 )
%     Agent1 is the agent after Agent performs the sequence of actions in
%     Trace
%
after( P, [], P ) :- !.
after( P, [A|S], P2 ) :-
	trans( P, A1, P1 ),
	eval( A1, A ),
	after( P1, S, P2 ).


%% successors(+Agent, -L)
%	"L" is a collection of immediate successors of "Agent"; each
%	is of the form "pr(A,P)" where "A" is an action and "P" is the
%	agent after performing "A".
%
successors(N, L) :- 
        findall( pr(X,Y), trans(N, X, Y), L).

